/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.TreeItem;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.ProjectOperation;
import ch.kuramo.javie.app.project.RemoveKeyframesOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.InPlaceEditorShell;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.ArithmeticalAnimatableValue;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Util;

public abstract class AnimatableValueElementDelegate<V> {

	private static final int[] UNDERLINE_DASH = new int[] { 2, 2 };

	protected static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	protected static final boolean WIN32 = SWT.getPlatform().equals("win32");


	protected final AnimatableValueElement element;

	protected final String name;

	protected final AnimatableValue<V> avalue;

	protected V valueWithExpr;

	protected V valueWithoutExpr;

	protected final Rectangle stopwatchArea = new Rectangle(Integer.MIN_VALUE, Integer.MIN_VALUE, 13, 14);

	protected Rectangle valueColBounds;

	protected final List<Rectangle> valueArea = Util.newList();

	protected boolean drawValueWithExpr;

	protected V valueToDraw;

	protected final SwitchGroup keyframeNavSwitch = new SwitchGroup();

	protected int yKeyframe = Integer.MIN_VALUE;

	protected V originalValue;

	protected int valueIndex;

	protected DragGestureEditor dragGestureEditor;

	protected boolean canInPlaceEdit;

	protected Element[] children;


	public AnimatableValueElementDelegate(
			AnimatableValueElement element, String name, AnimatableValue<V> avalue) {

		this.element = element;
		this.name = name;
		this.avalue = avalue;

		keyframeNavSwitch.add(new PrevKeyframeSwitch());
		keyframeNavSwitch.add(new ToggleKeyframeSwitch());
		keyframeNavSwitch.add(new NextKeyframeSwitch());
	}

	public Image getColumnImage(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return ImageUtil.getStopwatchIcon(avalue.hasKeyframe());

			default:
				return null;
		}
	}

	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return name;

			case LayerCompositionView.VALUE_COL:
				// ここでnullや空文字を返すと、なぜかCellEditorがすぐにフォーカスアウトしてしまう場合がある。
				return " ";

			default:
				return null;
		}
	}

	public boolean hasChildren() {
		return hasExpression();
	}

	public Element[] getChildren() {
		if (hasExpression()) {
			if (children == null) {
				children = new Element[] { new ExpressionElement() };
			}
			return children;
		} else {
			return Element.EMPTY_CHILDREN;
		}
	}

	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.NAME_COL:
				stopwatchArea.x = event.x + 5;
				stopwatchArea.y = event.y + (WIN32 ? 1 : 4);
				break;

			case LayerCompositionView.VALUE_COL:
				updateValue();

				valueColBounds = ((TreeItem) event.item).getBounds(event.index);
				valueArea.clear();

				drawValueWithExpr = (hasExpression()
									&& !(dragGestureEditor != null && dragGestureEditor.dragDetected)
									&& !isInPlaceEditorActive());
				valueToDraw = drawValueWithExpr ? valueWithExpr : valueWithoutExpr;

				if (valueToDraw != null) {
					GC gc = event.gc;
					int x = event.x + 5;
					int y = event.y;
					int height = valueColBounds.height;
					boolean focused = ((event.detail & SWT.SELECTED) != 0 && ((Control) event.widget).isFocusControl());
	
					drawValue(gc, x, y, height, focused);
				}
				break;

			case LayerCompositionView.SHOWHIDE_COL:
				keyframeNavSwitch.paint(event);
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				yKeyframe = tm.drawKeyframes(event, avalue);
				break;
		}
	}

	public void updateCursor(MouseEvent event, int columnIndex) {
		Cursor cursor = null;
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				for (Rectangle r : valueArea) {
					if (r.contains(event.x, event.y)) {
						cursor = event.display.getSystemCursor(SWT.CURSOR_HAND);
						break;
					}
				}
				break;
		}
		element.viewer.getTree().setCursor(cursor);
	}

	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (stopwatchArea.contains(event.x, event.y)) {
					toggleStopwatchButon();
				}
				break;

			case LayerCompositionView.VALUE_COL:
				if (event.button == 1) {
					for (int i = 0, n = valueArea.size(); i < n; ++i) {
						Rectangle r = valueArea.get(i);
						if (r.contains(event.x, event.y)) {
							if (!canDragGestureEdit()) {
								valueIndex = i;
								openInPlaceEditor();
							} else if (dragGestureEditor == null) {
								originalValue = valueWithoutExpr;
								valueIndex = i;
								dragGestureEditor = new DragGestureEditor(event);
							}
							break;
						}
					}
				}
				break;

			case LayerCompositionView.SHOWHIDE_COL:
				keyframeNavSwitch.mouseDown(event);
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.mouseDown(event, yKeyframe, element, avalue);
				break;
		}
	}

	public boolean canEdit(int columnIndex) {
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
	}

	public boolean updateValue() {
		boolean changed = false;

		@SuppressWarnings("unchecked")
		Map<AnimatableValue<?>, ?> animatableValues = (Map<AnimatableValue<?>, ?>)
				element.viewer.getData(LayerCompositionView.ANIMATABLE_VALUES);

		@SuppressWarnings("unchecked")
		V newValue = (V) animatableValues.get(avalue);

		if ((valueWithExpr != null && !valueWithExpr.equals(newValue))
				|| (valueWithExpr == null && newValue != null)) {

			valueWithExpr = newValue;
			changed = true;
		}

		TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
		newValue = avalue.valueWithoutExpressionAtTime(tm.getCurrentTime());

		if ((valueWithoutExpr != null && !valueWithoutExpr.equals(newValue))
				|| (valueWithoutExpr == null && newValue != null)) {

			valueWithoutExpr = newValue;
			changed = true;
		}

		return changed;
	}

	public boolean hasExpression() {
		return (avalue.getExpression() != null);
	}

	protected void openInPlaceEditor() {
		element.viewer.getTree().setCursor(null);

		canInPlaceEdit = true;
		element.viewer.editElement(element, LayerCompositionView.VALUE_COL);
		canInPlaceEdit = false;
	}

	protected void modifyValue(V newValue) {
		modifyValue(newValue, null);
	}

	protected void modifyValue(V newValue, String relation) {
		if (newValue != null && avalue instanceof ArithmeticalAnimatableValue<?>) {
			newValue = ((ArithmeticalAnimatableValue<V>) avalue).clamp(newValue);
		}

		if ((valueWithoutExpr != null && !valueWithoutExpr.equals(newValue))
				|| (valueWithoutExpr == null && newValue != null)) {

			ProjectManager pm = (ProjectManager) element.viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);

			pm.postOperation(avalue.hasKeyframe()
					? element.createModifyKeyframeOperation(pm, tm.getCurrentTime(), newValue, relation)
					: element.createModifyValueOperation(pm, newValue, relation));
		}
	}

	protected void modifyExpression(String newExpr) {
		if (newExpr != null) {
			newExpr = newExpr.trim();
			if (newExpr.length() == 0) {
				newExpr = null;
			}
		}

		String oldExpr = avalue.getExpression();
		if ((oldExpr != null && !oldExpr.equals(newExpr)) || (oldExpr == null && newExpr != null)) {
			ProjectManager pm = (ProjectManager) element.viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			ProjectOperation operation = element.createModifyExpressionOperation(pm, newExpr);
			pm.postOperation(operation);
		}
	}

	protected void toggleStopwatchButon() {
		ProjectManager pm = (ProjectManager) element.viewer.getData(LayerCompositionView.PROJECT_MANAGER);
		TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);

		pm.postOperation(avalue.hasKeyframe()
				? element.createClearKeyframesOperation(pm)
				: element.createModifyKeyframeOperation(pm, tm.getCurrentTime(), valueWithoutExpr, null));
	}

	protected abstract void drawValue(GC gc, int x, int y, int height, boolean focused);

	protected abstract boolean canDragGestureEdit();

	protected abstract V dragGesture(double dx, double dy);

	protected abstract boolean isInPlaceEditorActive();


	protected void setForeground(GC gc, boolean focused) {
		gc.setForeground(gc.getDevice().getSystemColor(
				focused ? SWT.COLOR_WHITE : SWT.COLOR_BLACK));
	}

	protected void setValueForeground(GC gc, boolean focused) {
		gc.setForeground(gc.getDevice().getSystemColor(
				drawValueWithExpr ? SWT.COLOR_DARK_RED :
				focused ? SWT.COLOR_WHITE : SWT.COLOR_DARK_BLUE));
	}

	protected int drawValue(
			GC gc, int x, int y, int height, boolean focused, String value) {

		setValueForeground(gc, focused);

		Point extent = gc.textExtent(value, SWT.DRAW_TRANSPARENT);

		y += (height - extent.y) / 2;
		gc.drawString(value, x, y, true);

		valueArea.add(new Rectangle(x, y, extent.x, extent.y));

		y += extent.y;
		gc.setLineDash(UNDERLINE_DASH);
		if (COCOA) {
			gc.drawLine(x, y - 1, x + extent.x, y - 1);
		} else if (WIN32) {
			gc.drawLine(x, y, x + extent.x - 2, y);
		} else {
			gc.drawLine(x, y, x + extent.x, y);
		}

		return x + extent.x;
	}

	protected int drawValue(
			GC gc, int x, int y, int height, boolean focused,
			Double value, int precision, String unit) {

		return drawValue(gc, x, y, height, focused, value, false, precision, unit);
	}

	protected int drawValue(
			GC gc, int x, int y, int height, boolean focused,
			Double value, boolean alwaysSign, int precision, String unit) {

		x = drawValue(gc, x, y, height, focused, String.format(
				(alwaysSign ? "%+." : "%.") + precision + "f", value));

		if (unit != null) {
			x = drawString(gc, x + 1, y, height, unit);
		}

		return x;
	}

	protected int drawValue(
			GC gc, int x, int y, int height, boolean focused, Integer value, String unit) {

		x = drawValue(gc, x, y, height, focused, value.toString());

		if (unit != null) {
			x = drawString(gc, x + 1, y, height, unit);
		}

		return x;
	}

	protected int drawString(GC gc, int x, int y, int height, String str) {
		Point extent = gc.textExtent(str, SWT.DRAW_TRANSPARENT);

		y += (height - extent.y) / 2;
		gc.drawString(str, x, y, true);

		return x + extent.x;
	}

	protected String formatValue(Double value, int precision) {
		if (precision < 0) throw new IllegalArgumentException();
		String s = String.format("%." + precision + "f", value);
		int dotIndex = s.indexOf('.');
		if (dotIndex != -1) {
			for (int i = s.length()-1; i >= dotIndex; --i) {
				char c = s.charAt(i);
				if (c == '0') continue;
				return s.substring(0, (c == '.') ? i : i+1);
			}
		}
		return s;
	}


	private static Method cocoaGetCurrentButtonStateMethod;

	private static int getCocoaCurrentButtonState() {
		try {
			if (cocoaGetCurrentButtonStateMethod == null) {
				Class<?> clazz = Class.forName("org.eclipse.swt.internal.cocoa.OS");
				cocoaGetCurrentButtonStateMethod = clazz.getMethod("GetCurrentButtonState");
			}
			return (Integer) cocoaGetCurrentButtonStateMethod.invoke(null);
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new JavieRuntimeException(e);
		}
	}


	protected class DragGestureEditor {

		private final String relation = Util.randomId();

		private final long downTime;

		private final Point downPoint;

		private boolean dragDetected;

		private Point detectPoint;

		private Point prevPoint;

		private double[] currentPoint;


		protected DragGestureEditor(MouseEvent event) {
			Control control = (Control) event.widget;
			downTime = System.currentTimeMillis();
			downPoint = control.toDisplay(event.x, event.y);

			init(control);
		}

		private void init(final Control control) {
			final Display display = control.getDisplay();

			Listener listener = new Listener() {
				public void handleEvent(Event e) {
					switch (e.type) {
						case SWT.MouseMove:
							if (!control.isDisposed()) {
								// MacOSXの場合、マウスボタンを離してからも少しの間 MouseMove イベントが発生することがあり、
								// ユーザーの意図しないところまでドラッグ状態が続いてしまうことがある。
								// マウスボタンの状態を直接取得するとすでにマウスボタンが離されていることが検出できるので、それで対処している。
								if (COCOA && (getCocoaCurrentButtonState() & 1) == 0) {
									break;
								}

								Point pt = display.getCursorLocation();

								if (!dragDetected) {
									dragDetected = (System.currentTimeMillis() - downTime > 100)
											&& (Math.abs(pt.x - downPoint.x) > 3 || Math.abs(pt.y - downPoint.y) > 3);
									if (dragDetected) {
										currentPoint = new double[] { pt.x, pt.y };
										prevPoint = detectPoint = pt;
									}
								}

								if (dragDetected) {
									double deltaScale = (e.stateMask & SWT.MOD2) != 0 ? 10 :
														(e.stateMask & SWT.MOD1) != 0 ? 0.1 : 1;
									currentPoint[0] += (pt.x - prevPoint.x) * deltaScale;
									currentPoint[1] += (pt.y - prevPoint.y) * deltaScale;

									prevPoint = pt;

									V newValue = dragGesture(currentPoint[0] - detectPoint.x, currentPoint[1] - detectPoint.y);
									modifyValue(newValue, relation);
								}

								break;
							}
							// fall through

						case SWT.MouseUp:
						case SWT.Deactivate:
							dragGestureEditor = null;

							display.removeFilter(SWT.MouseMove, this);
							display.removeFilter(SWT.MouseUp, this);
							display.removeFilter(SWT.Deactivate, this);

							if (!control.isDisposed()) {
								if (e.type == SWT.MouseUp && !dragDetected) {
									openInPlaceEditor();

								} else if (hasExpression() && valueColBounds != null) {
									element.viewer.getTree().redraw(
											valueColBounds.x, valueColBounds.y,
											valueColBounds.width, valueColBounds.height, false);
								}
							}
							break;
					}
				}
			};

			display.addFilter(SWT.MouseMove, listener);
			display.addFilter(SWT.MouseUp, listener);
			display.addFilter(SWT.Deactivate, listener);
		}

	}


	public class ExpressionElement extends Element {

		private Rectangle exprArea;

		private Point editorSize;


		public ExpressionElement() {
			super(element);
		}

		public String getColumnText(int columnIndex) {
			switch (columnIndex) {
				case LayerCompositionView.NAME_COL:
					return "エクスプレッション：" + name;

				case LayerCompositionView.TIMELINE_COL:
					return avalue.getExpression().replaceAll("\\r?\\n|\\r", " ");

				default:
					return super.getColumnText(columnIndex);
			}
		}

		public void paintColumn(Event event) {
			if (event.index == LayerCompositionView.TIMELINE_COL) {
				exprArea = ((TreeItem) event.item).getBounds(LayerCompositionView.TIMELINE_COL);
			}
			super.paintColumn(event);
		}

		public void mouseDown(MouseEvent event, int columnIndex) {
			super.mouseDown(event, columnIndex);

			if (columnIndex == LayerCompositionView.TIMELINE_COL && event.button == 1) {
				openInPlaceEditor();
			}
		}

		public void openInPlaceEditor() {
			if (exprArea == null) {
				viewer.getTree().getDisplay().asyncExec(new Runnable() {
					public void run() {
						if (exprArea != null) {
							openInPlaceEditor();
						}
					}
				});
				return;
			}

			if (editorSize == null) {
				editorSize = new Point(exprArea.width, 200);
			}

			final Shell shell = InPlaceEditorShell.create(
					viewer.getTree(), exprArea,
					editorSize.x, editorSize.y, true);

			shell.setMinimumSize(100, 100);

			FillLayout fillLayout = new FillLayout();
			fillLayout.marginWidth = 5;
			fillLayout.marginHeight = 5;
			shell.setLayout(fillLayout);

			final Text text = new Text(shell, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
			text.setTabs(4);

			final Font font;
			if (WIN32) {
				font = new Font(text.getDisplay(), "MS Gothic", 10, SWT.NORMAL);
				text.setFont(font);
			} else if (COCOA) {
				font = new Font(text.getDisplay(), "Monaco", 11, SWT.NORMAL);
				text.setFont(font);
			} else {
				font = null;
			}

			text.setText(avalue.getExpression());
			text.selectAll();

			shell.addDisposeListener(new DisposeListener() {
				public void widgetDisposed(DisposeEvent e) {
					modifyExpression(text.getText());
					editorSize = shell.getSize();
					if (font != null) {
						text.setFont(null);
						font.dispose();
					}
				}
			});

			shell.open();
		}

	}


	private static final Image prevKeyframeImage			= Activator.getDefault().getImageRegistry().get(Activator.IMG_KFNAV_PREV);
	private static final Image prevKeyframeDisabledImage	= Activator.getDefault().getImageRegistry().get(Activator.IMG_KFNAV_PREV_DISABLED);
	private static final Image nextKeyframeImage			= Activator.getDefault().getImageRegistry().get(Activator.IMG_KFNAV_NEXT);
	private static final Image nextKeyframeDisabledImage	= Activator.getDefault().getImageRegistry().get(Activator.IMG_KFNAV_NEXT_DISABLED);

	private class PrevKeyframeSwitch implements Switch {

		private final TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
		private final LayerComposition comp = (LayerComposition) element.viewer.getData(LayerCompositionView.LAYER_COMPOSITION);

		private Time getPrevTime() {
			Time frameDuration = comp.getFrameDuration();
			Time halfOfFrameDuration = new Time(frameDuration.timeValue/2, frameDuration.timeScale);

			SortedMap<Time, Keyframe<V>> headMap = avalue.getKeyframeMap().headMap(
					tm.getCurrentTime().subtract(halfOfFrameDuration));
			if (headMap.isEmpty()) {
				return null;
			}

			Time time = headMap.lastKey().add(halfOfFrameDuration);
			return Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);
		}

		public boolean hasBorder() {
			return false;
		}

		public void mouseDown(MouseEvent event) {
			if (avalue.hasKeyframe()) {
				Time prevTime = getPrevTime();
				if (prevTime != null) {
					tm.setTime(prevTime);
				}
			}
		}

		public Image getImage() {
			return avalue.hasKeyframe() ? (getPrevTime() != null) ? prevKeyframeImage : prevKeyframeDisabledImage : null;
		}
	}

	private class NextKeyframeSwitch implements Switch {

		private final TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
		private final LayerComposition comp = (LayerComposition) element.viewer.getData(LayerCompositionView.LAYER_COMPOSITION);

		private Time getNextTime() {
			Time frameDuration = comp.getFrameDuration();
			Time halfOfFrameDuration = new Time(frameDuration.timeValue/2, frameDuration.timeScale);

			SortedMap<Time, Keyframe<V>> tailMap = avalue.getKeyframeMap().tailMap(
					tm.getCurrentTime().add(frameDuration).subtract(halfOfFrameDuration));
			if (tailMap.isEmpty()) {
				return null;
			}

			Time time = tailMap.firstKey().add(halfOfFrameDuration);
			return Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);
		}

		public boolean hasBorder() {
			return false;
		}

		public void mouseDown(MouseEvent event) {
			if (avalue.hasKeyframe()) {
				Time nextTime = getNextTime();
				if (nextTime != null) {
					tm.setTime(nextTime);
				}
			}
		}

		public Image getImage() {
			return avalue.hasKeyframe() ? (getNextTime() != null) ? nextKeyframeImage : nextKeyframeDisabledImage : null;
		}
	}

	private class ToggleKeyframeSwitch implements Switch {

		private final TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
		private final ProjectManager pm = (ProjectManager) element.viewer.getData(LayerCompositionView.PROJECT_MANAGER);
		private final LayerComposition comp = (LayerComposition) element.viewer.getData(LayerCompositionView.LAYER_COMPOSITION);

		private SortedMap<Time, Keyframe<V>> getKeyframeMap() {
			Time frameDuration = comp.getFrameDuration();
			Time halfOfFrameDuration = new Time(frameDuration.timeValue/2, frameDuration.timeScale);
			Time time = tm.getCurrentTime();
			Time t1 = time.subtract(halfOfFrameDuration);
			Time t2 = time.add(frameDuration).subtract(halfOfFrameDuration);
			return avalue.getKeyframeMap().subMap(t1, t2);
		}

		public boolean hasBorder() {
			return avalue.hasKeyframe();
		}

		public void mouseDown(MouseEvent event) {
			if (!avalue.hasKeyframe()) {
				return;
			}

			Collection<Keyframe<V>> keyframes = getKeyframeMap().values();

			if (keyframes.isEmpty()) {
				pm.postOperation(element.createModifyKeyframeOperation(pm, tm.getCurrentTime(), valueWithoutExpr, null));

			} else {
				List<Object[]> data = Util.newList();

				if (element instanceof LayerAnimatableValueElement) {
					LayerAnimatableValueElement avalueElem = (LayerAnimatableValueElement) element;
					String id = avalueElem.layer.getId();
					String prop = avalueElem.getProperty();

					for (Keyframe<V> kf : keyframes) {
						data.add(new Object[] { id, -1, prop, kf });
					}

				} else {
					EffectAnimatableValueElement avalueElem = (EffectAnimatableValueElement) element;
					String id = avalueElem.layer.getId();
					int effectIndex = avalueElem.layer.getEffects().indexOf(avalueElem.effect);
					String prop = avalueElem.descriptor.getName();

					for (Keyframe<V> kf : keyframes) {
						data.add(new Object[] { id, effectIndex, prop, kf });
					}
				}

				pm.postOperation(new RemoveKeyframesOperation(pm, comp, data.toArray(new Object[data.size()][])));
			}
		}

		public Image getImage() {
			SortedMap<Time, Keyframe<V>> keyframeMap = getKeyframeMap();
			if (keyframeMap.isEmpty()) {
				return null;
			}

			Time time = tm.getCurrentTime();
			Keyframe<V> kf = keyframeMap.get(time);
			if (kf == null) {
				SortedMap<Time, Keyframe<V>> head = keyframeMap.headMap(time);
				SortedMap<Time, Keyframe<V>> tail = keyframeMap.tailMap(time);
				Keyframe<V> kf1 = head.isEmpty() ? null : head.get(head.lastKey());
				Keyframe<V> kf2 = tail.isEmpty() ? null : tail.get(tail.firstKey());
				if (kf1 != null && kf2 != null) {
					kf = kf2.time.subtract(time).before(time.subtract(kf1.time)) ? kf2 : kf1;
				} else if (kf1 != null) {
					kf = kf1;
				} else if (kf2 != null) {
					kf = kf2;
				}
			}
			return ImageUtil.getKeyframeIcon(kf.interpolation, false);
		}
	}

}
