/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.mmd2javie;

import java.awt.geom.CubicCurve2D;
import java.io.BufferedInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.NullLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectElementFactory;

class MMD2Javie2 {

	private static final Logger logger = LoggerFactory.getLogger(MMD2Javie2.class);

	private List<VmdCamera> data;

	boolean readFile(final File file, IRunnableContext rc) {
		IRunnableWithProgress runnable = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
				try {
					readFile(file, monitor);
				} catch (IOException e) {
					throw new InvocationTargetException(e);
				}
			}
		};

		try {
			logger.info("ファイルの読み込みを開始しました...");
			rc.run(true, true, runnable);
			if (data != null) {
				logger.info("ファイルの読み込みを完了しました");
				return true;
			} else {
				logger.info("データがありません");
				return false;
			}
		} catch (InterruptedException ex) {
			logger.info("ファイルの読み込みを中断しました");
		} catch (InvocationTargetException ex) {
			Throwable t = ex.getTargetException();
			logger.error("エラーが発生しました", (t != null) ? t : ex);
		}
		return false;
	}

	private void readFile(File file, IProgressMonitor monitor) throws IOException {
		List<VmdCamera> data = Util.newList();

		monitor.beginTask("ファイルを読み込み中...", IProgressMonitor.UNKNOWN);
		InputStream in = null;
		try {
			in = new BufferedInputStream(new FileInputStream(file));

			do {
				byte[] bytes = new byte[26];
				int len = in.read(bytes);
				if (bytes[len-1] != 0) break;

				String str = new String(bytes, 0, len-1, "US-ASCII");
				if (!str.equals("Vocaloid Motion Data 0002")) break;

				// 4バイトのパティング + モデル名20バイトを捨てる。
				if (in.skip(24) != 24) break;

				// モーションデータがある場合はスキップする
				int count;
				if ((count = readInt(in)) != 0) {
					count *= 111;
					if (in.skip(count) != count) break;
				}

				// スキンデータがある場合はスキップする
				if ((count = readInt(in)) != 0) {
					count *= 23;
					if (in.skip(count) != count) break;
				}

				// カメラデータが無い場合は終了
				if ((count = readInt(in)) == 0) break;

				monitor.done();
				monitor.beginTask("ファイルを読み込み中...", count+1);

				for (int i = 0; i < count; ++i) {
					data.add(new VmdCamera(in));
					monitor.worked(1);
				}

				Collections.sort(data, new Comparator<VmdCamera>() {
					public int compare(VmdCamera o1, VmdCamera o2) {
						return o1.frameNumber - o2.frameNumber;
					}
				});
				monitor.worked(1);

			} while (false);

			if (data.isEmpty()) {
				this.data = null;
			} else {
				this.data = data;
			}

		} finally {
			monitor.done();
			if (in != null) in.close();
		}
	}

	void createCamera(Time frameDuration, double scale, Vec3d offset, Size2i compSize,
			Project project, ProjectElementFactory elementFactory,
			IProgressMonitor monitor) throws InterruptedException {

		if (data == null) {
			throw new IllegalStateException();
		}

		int frameCount = getFrameCount();
		monitor.beginTask("カメラデータを処理中...", 1 + frameCount);
		try {
			// コンポジションの作成
			Time duration = Time.fromFrameNumber(frameCount, frameDuration);
			LayerComposition layerComp = elementFactory.newLayerComposition(
					ColorMode.RGBA8, compSize, frameDuration, duration);

			CompositionItem compItem = elementFactory.newCompositionItem(layerComp);
			compItem.setName("MMD Camera");

			project.getCompositions().add(layerComp);
			project.getItems().add(compItem);

			// Y回転ヌルレイヤーの作成
			NullLayer yNullLayer = elementFactory.newNullLayer();
			initLayerTimes(yNullLayer, frameDuration, duration);
			yNullLayer.setThreeD(true);

			// XZ回転ヌルレイヤーの作成
			NullLayer xzNullLayer = elementFactory.newNullLayer();
			initLayerTimes(xzNullLayer, frameDuration, duration);
			xzNullLayer.setThreeD(true);
			xzNullLayer.setParentId(yNullLayer.getId());

			// カメラレイヤーの作成
			CameraLayer cameraLayer = elementFactory.newCameraLayer();
			initLayerTimes(cameraLayer, frameDuration, duration);
			cameraLayer.setParentId(xzNullLayer.getId());

			// レイヤーをコンポジションに追加
			yNullLayer.setName("MMD Camera [position, rotationY]");
			xzNullLayer.setName("MMD Camera [rotationXZ]");
			cameraLayer.setName("MMD Camera");
			layerComp.getLayers().add(cameraLayer);
			layerComp.getLayers().add(xzNullLayer);
			layerComp.getLayers().add(yNullLayer);

			// 初期パラメータの設定（キーフレームを設定するので、しなくてもよいが）
			VmdCamera vmdCamera = data.get(0);
			double x = vmdCamera.x*scale + offset.x;
			double y = vmdCamera.y*scale + offset.y;
			double z = vmdCamera.z*scale + offset.z;
			double rx = vmdCamera.rx;
			double ry = vmdCamera.ry;
			double rz = vmdCamera.rz;
			double len = vmdCamera.length*scale;
			double zoom = calcZoom(compSize.height, vmdCamera.viewingAngle);

			yNullLayer.getPosition().reset(new Vec3d(x, y, z));
			yNullLayer.getRotationY().reset(Math.toDegrees(ry));
			xzNullLayer.getRotationX().reset(Math.toDegrees(rx));
			xzNullLayer.getRotationZ().reset(Math.toDegrees(rz));
			cameraLayer.getPosition().reset(new Vec3d(0, 0, len));
			cameraLayer.getZoom().reset(zoom);

			monitor.worked(1);

			// 最初のキーフレーム
			Time time = Time.fromFrameNumber(vmdCamera.frameNumber, frameDuration);
			yNullLayer.getPosition().putKeyframe(time, new Vec3d(x, y, z), Interpolation.LINEAR);
			yNullLayer.getRotationY().putKeyframe(time, Math.toDegrees(ry), Interpolation.LINEAR);
			xzNullLayer.getRotationX().putKeyframe(time, Math.toDegrees(rx), Interpolation.LINEAR);
			xzNullLayer.getRotationZ().putKeyframe(time, Math.toDegrees(rz), Interpolation.LINEAR);
			cameraLayer.getPosition().putKeyframe(time, new Vec3d(0, 0, len), Interpolation.LINEAR);
			cameraLayer.getZoom().putKeyframe(time, zoom, Interpolation.LINEAR);

			monitor.worked(vmdCamera.frameNumber+1);

			VmdCamera prevCamera = null;
			for (int k = 0, i = vmdCamera.frameNumber+1; i < frameCount; ++i) {
				if (monitor.isCanceled()) {
					throw new InterruptedException("canceled");
				}

				if (vmdCamera.frameNumber < i) {
					prevCamera = vmdCamera;
					vmdCamera = data.get(++k);
				}

				time = Time.fromFrameNumber(i, frameDuration);

				if (vmdCamera.frameNumber == i) {
					x = vmdCamera.x*scale + offset.x;
					y = vmdCamera.y*scale + offset.y;
					z = vmdCamera.z*scale + offset.z;
					rx = vmdCamera.rx;
					ry = vmdCamera.ry;
					rz = vmdCamera.rz;
					len = vmdCamera.length*scale;
					zoom = calcZoom(compSize.height, vmdCamera.viewingAngle);

					yNullLayer.getPosition().putKeyframe(time, new Vec3d(x, y, z), Interpolation.LINEAR);
					yNullLayer.getRotationY().putKeyframe(time, Math.toDegrees(ry), Interpolation.LINEAR);
					xzNullLayer.getRotationX().putKeyframe(time, Math.toDegrees(rx), Interpolation.LINEAR);
					xzNullLayer.getRotationZ().putKeyframe(time, Math.toDegrees(rz), Interpolation.LINEAR);
					cameraLayer.getPosition().putKeyframe(time, new Vec3d(0, 0, len), Interpolation.LINEAR);
					cameraLayer.getZoom().putKeyframe(time, zoom, Interpolation.LINEAR);

				} else {
					double[][] ip = vmdCamera.interpolations;
					double t = (double)(i - prevCamera.frameNumber) / (vmdCamera.frameNumber - prevCamera.frameNumber);

					// 位置x,y,zのどれか１つでも曲線補間している場合。
					if (isCurve(ip[0]) || isCurve(ip[1]) || isCurve(ip[2])) {
						x = curve(prevCamera.x, vmdCamera.x, ip[0], t) * scale + offset.x;
						y = curve(prevCamera.y, vmdCamera.y, ip[1], t) * scale + offset.y;
						z = curve(prevCamera.z, vmdCamera.z, ip[2], t) * scale + offset.z;
						yNullLayer.getPosition().putKeyframe(time, new Vec3d(x, y, z), Interpolation.LINEAR);
					}

					// 回転を曲線補間している場合。
					if (isCurve(ip[3])) {
						rx = curve(prevCamera.rx, vmdCamera.rx, ip[3], t);
						ry = curve(prevCamera.ry, vmdCamera.ry, ip[3], t);
						rz = curve(prevCamera.rz, vmdCamera.rz, ip[3], t);
						yNullLayer.getRotationY().putKeyframe(time, Math.toDegrees(ry), Interpolation.LINEAR);
						xzNullLayer.getRotationX().putKeyframe(time, Math.toDegrees(rx), Interpolation.LINEAR);
						xzNullLayer.getRotationZ().putKeyframe(time, Math.toDegrees(rz), Interpolation.LINEAR);
					}

					// 距離を曲線補間している場合。
					if (isCurve(ip[4])) {
						len = curve(prevCamera.length, vmdCamera.length, ip[4], t) * scale;
						cameraLayer.getPosition().putKeyframe(time, new Vec3d(0, 0, len), Interpolation.LINEAR);
					}

					// 視野角を曲線補間している場合。
					if (isCurve(ip[5])) {
						double viewingAngle = curve(prevCamera.viewingAngle, vmdCamera.viewingAngle, ip[5], t);
						zoom = calcZoom(compSize.height, viewingAngle);
						cameraLayer.getZoom().putKeyframe(time, zoom, Interpolation.LINEAR);
					}
				}

				monitor.worked(1);
			}

		} finally {
			monitor.done();
		}
	}

	private void initLayerTimes(Layer layer, Time frameDuration, Time duration) {
		Time time0 = Time.fromFrameNumber(0, frameDuration);
		layer.setStartTime(time0);
		layer.setInPoint(time0);
		layer.setOutPoint(duration);
	}

	private double calcZoom(int compHeight, double viewingAngle) {
		return compHeight / (2*Math.tan(Math.toRadians(viewingAngle)/2));
	}

	private boolean isCurve(double[] interpolation) {
		return interpolation[0] != interpolation[2] || interpolation[1] != interpolation[3];
	}

	private double curve(double v1, double v2, double[] ip, double t) {
		double[] eqn = { -t, 3*ip[0], 3*ip[1]-6*ip[0], 1+3*ip[0]-3*ip[1] };

		// ip[0],ip[1]の値は、0から127の範囲の整数を127で割ったものなので、誤差が無くてもeqn[3]がゼロになることはない。
		// (eqn[3]がゼロになるにはip[0],ip[1]の元の値の差が-42.333…である必要があるが、ip[0],ip[1]の元の値は整数である)
		//
		//if (Math.abs(eqn[3]) < 1e-10) { 
		//	eqn[3] = 0;
		//}
		int n = CubicCurve2D.solveCubic(eqn);

		// FIXME [0,1]の範囲の解が複数ある場合を考慮する必要は？
		double p = -1;
		for (int i = 0; i < n; ++i) {
			if (eqn[i] >= 0 && eqn[i] <= 1) {
				p = eqn[i];
				break;
			}
		}
		// FIXME [0,1]の範囲の解が見つからない場合（MMDの補間の仕様上、そのようなことは無いはずだが）
		if (p == -1) {
			return v1 + (v2-v1) * t;
		}

		double pp = p * p;
		double ppp = pp * p;
		double q = 1 - p;
		double qq = q * q;
		//double qqq = qq * q;

		return v1 + (v2-v1) * (3*qq*p*ip[2] + 3*q*pp*ip[3] + ppp);
	}

	public int getFrameCount() {
		return (data != null) ? data.get(data.size()-1).frameNumber+1 : 0;
	}

	private static byte readByte(InputStream in) throws IOException {
		int b = in.read();
		if (b == -1) throw new EOFException();
		return (byte) b;
	}

	private static int readInt(InputStream in) throws IOException {
		int value = 0;
		for (int i = 0; i < 4; ++i) {
			int b = in.read();
			if (b == -1) throw new EOFException();
			value += (b << (i*8));
		}
		return value;
	}

	private static float readFloat(InputStream in) throws IOException {
		return Float.intBitsToFloat(readInt(in));
	}

	private static boolean readBoolean(InputStream in) throws IOException {
		int b = in.read();
		if (b == -1) throw new EOFException();
		return (b != 0);
	}

	private static class VmdCamera {
		final int frameNumber;
		final double length;
		final double x, y, z;
		final double rx, ry, rz;
		final double[][] interpolations;
		final int viewingAngle;

		// FIXME orthographicの場合はどうする？
		@SuppressWarnings("unused")
		final boolean orthographic;

		VmdCamera(InputStream in) throws IOException {
			frameNumber = readInt(in);
			length = readFloat(in);
			x = readFloat(in);
			y = -readFloat(in);
			z = readFloat(in);
			rx = readFloat(in);
			ry = -readFloat(in);
			rz = readFloat(in);

			interpolations = new double[6][4];
			for (int i = 0; i < 6; ++i) {
				interpolations[i][0] = readByte(in) / 127.0;
				interpolations[i][1] = readByte(in) / 127.0;
				interpolations[i][2] = readByte(in) / 127.0;
				interpolations[i][3] = readByte(in) / 127.0;
			}

			viewingAngle = readInt(in);
			orthographic = readBoolean(in);
		}
	}

}
