/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.Map.Entry;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.MediaOptions;
import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.MediaOptions.Option;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class WindowsMediaFoundationSource implements MediaSource {

	private static final Logger logger = LoggerFactory.getLogger(WindowsMediaFoundationSource.class);


	private static final boolean disabled = Boolean.getBoolean("javie.mfsrc.disabled");
	
	private static final boolean sysMem = !Boolean.getBoolean("javie.mfsrc.noSysMem");

	private static final int bufferFrames = Math.max(1, Integer.getInteger("javie.mfsrc.bufferFrames", 2));

	private static final int cacheFrames = Math.max(1, Integer.getInteger("javie.mfsrc.cacheFrames", 3)) + bufferFrames;


	private VideoThread videoThread;

	private VideoBounds bounds;

	private boolean topDown;

	private Time duration;

	private Time videoFrameDuration;

	private AudioThread audioThread;

	private AudioFormat audioFormat;

	private boolean audioFloat;

	private AudioMode audioMode;

	private AudioInputStream audioStream;

	private long audioStreamPosition;


	private final RenderContext context;

	private final IVideoRenderSupport vrSupport;

	private final GLGlobal glGlobal;

	private final AudioRenderSupport arSupport;

	private final IArrayPools arrayPools;

	@Inject
	public WindowsMediaFoundationSource(RenderContext context,
			IVideoRenderSupport vrSupport, GLGlobal glGlobal,
			AudioRenderSupport arSupport, IArrayPools arrayPools) {

		super();
		this.context = context;
		this.vrSupport = vrSupport;
		this.glGlobal = glGlobal;
		this.arSupport = arSupport;
		this.arrayPools = arrayPools;
	}

	public boolean initialize(File file) {
		if (disabled) {
			return false;
		}

		VideoThread vt = new VideoThread();
		if (!vt.initialize(file)) {
			vt = null;

			String name = file.getName().toLowerCase();
			if (name.endsWith(".avi")) {
				return false;
			}
		}

		AudioThread at = new AudioThread();
		if (!at.initialize(file)) {
			at = null;
		}

		if (vt == null && at == null) {
			return false;
		}

		videoThread = vt;
		audioThread = at;
		return true;
	}

	public void dispose() {
		if (videoThread != null) {
			videoThread.shutdown();
			videoThread = null;
		}
		if (audioThread != null) {
			closeAudioStream();
			audioThread.shutdown();
			audioThread = null;
		}
	}

	private void closeAudioStream() {
		if (audioStream != null) {
			try {
				audioStream.close();
			} catch (IOException e) {
				logger.warn("failed to close AudioInputStream", e);
			}
			audioStream = null;
		}
		audioStreamPosition = 0;
	}

	public boolean isVideoAvailable() {
		return (videoThread != null);
	}

	public boolean isAudioAvailable() {
		return (audioThread != null);
	}

	public Time getDuration(MediaOptions options) {
		return duration;
	}

	public Time getVideoFrameDuration(MediaOptions options) {
		return videoFrameDuration;
	}

	public VideoBounds getVideoFrameBounds(MediaOptions options) {
		return bounds;
	}

	public IVideoBuffer getVideoFrame(Time mediaTime, MediaOptions options) {
		if (videoThread == null) {
			return null;
		}

		if (mediaTime.timeValue < 0 || !mediaTime.before(duration)) {
			IVideoBuffer vb = vrSupport.createVideoBuffer(bounds, ColorMode.RGBA8);
			vb.clear();
			return vb;
		}

		FrameSurface fs = videoThread.lockFrame(
				mediaTime.toFrameNumber(videoFrameDuration));
		if (fs == null) {
			IVideoBuffer vb = vrSupport.createVideoBuffer(bounds, ColorMode.RGBA8);
			vb.clear();
			return vb;
		}

		int pitch = fs.buffer.capacity() / 4 / bounds.height;
		IVideoBuffer vb = vrSupport.createVideoBuffer(new VideoBounds(pitch, bounds.height), ColorMode.RGBA8);
		try {
			int dstTexture = vb.getTexture();

			// 以下、VideoBufferImpl.copyArrayToTexture とほぼ同じ。
			GL2 gl = context.getGL().getGL2();

			if (glGlobal.isIntel()) {

				int[] current = new int[1];
				gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 0);
				try {
					gl.glBindTexture(GL2.GL_TEXTURE_2D, dstTexture);
					gl.glTexSubImage2D(GL2.GL_TEXTURE_2D, 0, 0, 0, pitch, bounds.height,
							GL2.GL_BGRA, GL2.GL_UNSIGNED_BYTE, fs.buffer);
				} finally {
					gl.glBindTexture(GL2.GL_TEXTURE_2D, current[0]);
				}

			} else {
				// FIXME 少し試した限りでは正常に動作するようなので、とりあえずコメントアウトしておく。
				//       (以前は、ロックしないで複数同時再生するとクラッシュすることがあった。ドライバが改善された？)
				//ReentrantLock lock = glGlobal.getGlobalLock();
				//lock.lock();
				//try {

					int[] pbo = new int[1];
					int[] current = new int[2];
					gl.glGetIntegerv(GL2.GL_PIXEL_UNPACK_BUFFER_BINDING, current, 0);
					gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_2D, current, 1);
					try {
						gl.glGenBuffers(1, pbo, 0);
						gl.glBindBuffer(GL2.GL_PIXEL_UNPACK_BUFFER, pbo[0]);
						gl.glBindTexture(GL2.GL_TEXTURE_2D, dstTexture);

						// D3D Surface -> PBO
						gl.glBufferData(GL2.GL_PIXEL_UNPACK_BUFFER, fs.buffer.capacity(), null,
										sysMem ? GL2.GL_STREAM_DRAW : GL2.GL_STREAM_COPY);
										// FIXME openVideo の２番目の引数(sysMem)が false の場合、
										//       fs.buffer はシステムメモリ上のものでは無いので
										//       GL_STREAM_COPY としているが、それが正しいのかどうか不明。
						ByteBuffer mappedPBO = gl.glMapBuffer(GL2.GL_PIXEL_UNPACK_BUFFER, GL2.GL_WRITE_ONLY);
						mappedPBO.put(fs.buffer);
						gl.glUnmapBuffer(GL2.GL_PIXEL_UNPACK_BUFFER);

						// PBO -> テクスチャ
						gl.glTexImage2D(GL2.GL_TEXTURE_2D, 0, GL2.GL_RGBA8,
								pitch, bounds.height, 0, GL2.GL_BGRA, GL2.GL_UNSIGNED_BYTE, 0);

						videoThread.unlockFrame(fs);
						fs = null;

					} finally {
						gl.glBindTexture(GL2.GL_TEXTURE_2D, current[1]);
						gl.glBindBuffer(GL2.GL_PIXEL_UNPACK_BUFFER, current[0]);
						if (pbo[0] != 0) gl.glDeleteBuffers(1, pbo, 0);
					}

				//} finally {
				//	lock.unlock();
				//}
			}
			// ここまで、VideoBufferImpl.copyArrayToTexture とほぼ同じ。

			Resolution resolution = context.getVideoResolution();
			boolean flipVert = options.isFlipVertical() ^ !topDown;
			if (pitch != bounds.width || resolution.scale != 1 || flipVert) {
				return convertByResolutionAndOptions(vb, pitch, resolution, flipVert);
			} else {
				IVideoBuffer result = vb;
				vb = null;
				return result;
			}

		} finally {
			if (fs != null) videoThread.unlockFrame(fs);
			if (vb != null) vb.dispose();
		}
	}

	private IVideoBuffer convertByResolutionAndOptions(
			IVideoBuffer srcBuffer, final int srcPitch, Resolution resolution, final boolean flipVert) {

		final VideoBounds srcBounds = srcBuffer.getBounds();
		final VideoBounds dstBounds = resolution.scale(bounds);
		final float scale = (float)resolution.scale;
		final float texCood = (float)srcBounds.width / srcPitch;

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				GLU glu = context.getGLU();

				gl.glViewport(0, 0, dstBounds.width, dstBounds.height);

				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadIdentity();
				if (flipVert) {
					glu.gluOrtho2D(0, dstBounds.width, dstBounds.height, 0);
				} else {
					glu.gluOrtho2D(0, dstBounds.width, 0, dstBounds.height);
				}

				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();
				gl.glScalef(scale, scale, 1);

				// テクスチャは useFramebuffer によってすでにバインドされている。
				//gl.glBindTexture(GL2.GL_TEXTURE_2D, srcTexture);
				//gl.glEnable(GL2.GL_TEXTURE_2D);

				gl.glColor4f(1, 1, 1, 1);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0); gl.glVertex2f(0, 0);
				gl.glTexCoord2f(texCood, 0); gl.glVertex2f(srcBounds.width, 0);
				gl.glTexCoord2f(texCood, 1); gl.glVertex2f(srcBounds.width, srcBounds.height);
				gl.glTexCoord2f(0, 1); gl.glVertex2f(0, srcBounds.height);
				gl.glEnd();
			}
		};

		IVideoBuffer buffer = null;
		try {
			buffer = vrSupport.createVideoBuffer(dstBounds);

			int pushAttribs = GL2.GL_CURRENT_BIT;
			vrSupport.useFramebuffer(operation, pushAttribs, buffer, srcBuffer);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	public IAudioBuffer getAudioChunk(Time mediaTime, MediaOptions options) {
		if (audioThread == null) {
			return null;
		}

		try {
			AudioMode am = context.getAudioMode();
			long frameNumber = mediaTime.toFrameNumber(am.sampleDuration);

			if (-frameNumber >= context.getAudioFrameCount() || !mediaTime.before(duration)) {
				IAudioBuffer ab = arSupport.createAudioBuffer();
				arSupport.clear(ab);
				return ab;
			}

			if (am != audioMode || am.frameSize*frameNumber != audioStreamPosition) {
				closeAudioStream();
			}

			if (audioStream == null) {
				audioStream = createAudioInputStream();
			}

			return readAudioData(frameNumber);

		} catch (IOException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		} catch (UnsupportedAudioFileException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		}
	}

	private AudioInputStream createAudioInputStream() throws IOException, UnsupportedAudioFileException {
		// サンプルレートと他のパラメータを同時に変換しようとすると失敗するようなので、2段階に分けて変換する。
		//
		// 変換の必要が無い場合、AudioSystem.getAudioInputStream は元のストリームをそのまま返すようなので
		// 以下のように無条件に変換を試みても無駄な変換は発生しないようである。

		audioMode = context.getAudioMode();
		int bits = audioMode.sampleSize * 8;
		int ch = audioMode.channels;

		long length = duration.toFrameNumber(new Time(1, (int)audioFormat.getFrameRate()));
		AudioInputStream s1 = AudioSystem.getAudioInputStream(audioFormat, new AudioInputStream(
				new MFAudioInputStream(audioFormat), audioFormat, length));

		// サンプルレート以外を変換
		AudioFormat f2 = new AudioFormat(s1.getFormat().getSampleRate(), bits, ch, true, false);
		AudioInputStream s2 = AudioSystem.getAudioInputStream(f2, s1);

		// サンプルレートを変換
		AudioFormat f3 = new AudioFormat(audioMode.sampleRate, bits, ch, true, false);
		return AudioSystem.getAudioInputStream(f3, s2);
	}

	private IAudioBuffer readAudioData(long frameNumber) throws IOException {
		IAudioBuffer ab = arSupport.createAudioBuffer();
		AudioMode audioMode = ab.getAudioMode();
		Object data = ab.getData();
		int dataOffset = 0;
		int dataLength = ab.getDataLength();
		int lengthInBytes = dataLength * audioMode.sampleSize;

		if (audioStreamPosition == 0) {
			if (frameNumber > 0) {
				long skip = audioMode.frameSize * frameNumber;
				long skipped = audioStream.skip((skip / audioMode.sampleRate) * audioMode.sampleRate);
				audioStreamPosition = skipped;
				skip -= skipped;
				if (skip > 0) {
					IArray<byte[]> array = arrayPools.getByteArray((int)skip);
					try {
						do {
							skipped = audioStream.read(array.getArray(), 0, (int)skip);
							if (skipped != -1) {
								audioStreamPosition += skipped;
								skip -= skipped;
							}
						} while (skipped != -1 && skip > 0);
					} finally {
						array.release();
					}
				}
				if (skip != 0) {
					logger.warn(String.format("skip failed: frameNumber=%d, skip=%d", frameNumber, skip));
					closeAudioStream();
					arSupport.clear(ab);
					return ab;
				}
			} else if (frameNumber < 0) {
				arSupport.clear(ab, 0, (int)-frameNumber);
				dataOffset = (int)-frameNumber * audioMode.channels;
				dataLength -= dataOffset;
				lengthInBytes -= dataOffset * audioMode.sampleSize;
			}
		}

		IArray<byte[]> array = arrayPools.getByteArray(lengthInBytes);
		try {
			byte[] buffer = array.getArray();

			int readBytes = 0;
			do {
				int n = audioStream.read(buffer, readBytes, lengthInBytes - readBytes);
				if (n == -1) {
					Arrays.fill(buffer, readBytes, lengthInBytes, (byte)0);
					break;
				}
				readBytes += n;
				audioStreamPosition += n;
			} while (readBytes < lengthInBytes);

			switch (audioMode.dataType) {
				case SHORT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asShortBuffer().get((short[]) data, dataOffset, dataLength);
					break;
				case INT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asIntBuffer().get((int[]) data, dataOffset, dataLength);
					break;
				case FLOAT:
					for (int i = dataOffset; i < dataLength; ++i) {
						int value =  (buffer[i*4  ] & 0xff)
								  | ((buffer[i*4+1] & 0xff) <<  8)
								  | ((buffer[i*4+2] & 0xff) << 16)
								  | ((buffer[i*4+3] & 0xff) << 24);
						((float[]) data)[i] = (float) value / Integer.MAX_VALUE;
					}
					break;
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}
		} finally {
			array.release();
		}

		return ab;
	}

	public MediaOptions validateOptions(MediaOptions options) {
		Option[] availableOptions;

		if (isVideoAvailable()) {
			availableOptions = new Option[] {
					/* Option.IGNORE_ALPHA, Option.STRAIGHT_ALPHA, Option.PREMULTIPLIED_ALPHA, */
					/* TODO Option.VIDEO_FRAME_DURATION, */ Option.FLIP_VERTICAL
			};
		} else {
			availableOptions = new Option[0];
		}

		if (options == null) {
			options = new MediaOptions(availableOptions);

			//if (isVideoAvailable()) {
			//	options.setAlphaType(AlphaType.IGNORE);
			//}

		} else {
			options = options.clone();
			options.setAvailableOptions(availableOptions);
		}

		return options;
	}


	private static class FrameSurface {
		private final long frameNumber;
		private long surface;
		private ByteBuffer buffer;

		FrameSurface(long frameNumber) {
			this.frameNumber = frameNumber;
		}
	}

	private class VideoThread extends Thread {

		private File file;

		private final BlockingQueue<FrameSurface> queue = new LinkedBlockingQueue<FrameSurface>();

		private long videoSource;

		@SuppressWarnings("serial")
		private final Map<Long, Long> buffer = new LinkedHashMap<Long, Long>(16, 0.75f, true) {
			protected boolean removeEldestEntry(Entry<Long, Long> eldest) {
				if (size() > cacheFrames) {
					remove(eldest.getKey());
					releaseSurface(videoSource, eldest.getValue());
				}
				return false;
			}
		};

		private long position = Long.MIN_VALUE;

		boolean initialize(File file) {
			this.file = file;
			start();
			
			FrameSurface fs = lockFrame(0);
			if (fs != null) {
				unlockFrame(fs);
				return true;
			} else {
				return false;
			}
		}

		void shutdown() {
			try {
				queue.put(new FrameSurface(Long.MIN_VALUE));
			} catch (InterruptedException e) {
			}
		}

		FrameSurface lockFrame(long frameNumber) {
			FrameSurface fs = new FrameSurface(frameNumber);
			synchronized (fs) {
				try {
					queue.put(fs);
					fs.wait();
				} catch (InterruptedException e) {
				}
			}
			return (fs.surface != 0) ? fs : null;
		}

		void unlockFrame(FrameSurface fs) {
			if (fs.surface == 0) throw new IllegalArgumentException();
			try {
				queue.put(fs);
			} catch (InterruptedException e) {
			}
		}

		public void run() {
			long[] result = new long[7];
			int hr = openVideo(file.getAbsolutePath(), sysMem, result);
			if (hr == 0) {
				videoSource = result[0];
				bounds = new VideoBounds((int) result[1], (int) result[2]);
				topDown = (result[3] != 0);
				duration = new Time(result[4], 10000000);
				videoFrameDuration = new Time(result[6], (int)result[5]);
				setName(WindowsMediaFoundationSource.this.getClass().getSimpleName()
						+ " (Video): " + file.getName());
			} else {
				for (FrameSurface fs; (fs = queue.poll()) != null; ) {
					synchronized (fs) { fs.notify(); }
				}
				return;
			}

			double estFrameDuration = Double.NaN;
			if (duration.toFrameNumber(videoFrameDuration) > 20) {
				long prev = 0, sum = 0;
				long[] ioTime = new long[1];
				for (int i = 0; i <= 10; ++i) {
					ioTime[0] = Long.MIN_VALUE;
					hr = frameSurfaceAtTime(videoSource, ioTime, 0, null);
					if(hr != 1) {
						logger.error("frameSurfaceAtTime: error " + hr);
						break;
					}
					if (i > 0) {
						sum += (ioTime[0] - prev);
					}
					prev = ioTime[0];
				}
				if (hr == 1) {
					estFrameDuration = sum / (10.0 * 10000000);
				}
			}
			if (Double.isNaN(estFrameDuration)) {
				estFrameDuration = videoFrameDuration.toSecond();
			}

			boolean shutdown = false;
			boolean forward = false;
			long prevNumber = 0;
			while (queue.peek() != null || !shutdown) {
				FrameSurface fs = queue.poll();
				if (fs == null) {
					if (forward) {
						long[] ioTime = new long[1];
						long[] newSurface = new long[1];
						while (queue.peek() == null
								&& !buffer.containsKey(position)
								&& position - prevNumber <= bufferFrames) {
							ioTime[0] = Long.MIN_VALUE;
							hr = frameSurfaceAtTime(videoSource, ioTime, Long.MIN_VALUE, newSurface);
							if(hr != 0) {
								logger.error("frameSurfaceAtTime: error " + hr);
								forward = false;
								break;
							}
							long frameNumber = Math.round(ioTime[0] / (10000000 * estFrameDuration));
							position = frameNumber + 1;
							Long old = buffer.put(frameNumber, newSurface[0]);
							if (old != null) releaseSurface(videoSource, old);
						}
					}
					try {
						fs = queue.take();
					} catch (InterruptedException e) {
						continue;
					}
				}

				if (fs.surface != 0) {
					unlockSurface(videoSource, fs.surface);
					continue;
				}

				if (fs.frameNumber == Long.MIN_VALUE) {
					shutdown = true;
					forward = false;
					continue;
				}

				forward = (fs.frameNumber > prevNumber);
				prevNumber = fs.frameNumber;

				long surface;
				Long bufferedSurface = buffer.get(fs.frameNumber);
				if (bufferedSurface != null) {
					surface = bufferedSurface;
				} else {
					long[] ioTime;
					if (fs.frameNumber < position || fs.frameNumber >= position + 10) {	// FIXME 10フレームが妥当かどうか
						ioTime = new long[] { (long)((fs.frameNumber+0.5)*estFrameDuration*10000000) };
					} else {
						ioTime = new long[] { Long.MIN_VALUE };
					}

					long limitTime = (long)((fs.frameNumber-(cacheFrames-bufferFrames)+0.5)*estFrameDuration*10000000);

					long[] newSurface = new long[1];
					for (;;) {
						hr = frameSurfaceAtTime(videoSource, ioTime, limitTime, newSurface);
						if (hr == 1) {
							ioTime[0] = Long.MIN_VALUE;
							continue;
						} else if(hr != 0) {
							logger.error("frameSurfaceAtTime: error " + hr);
							surface = 0;
							break;
						}

						long frameNumber = Math.round(ioTime[0] / (10000000 * estFrameDuration));
						position = frameNumber + 1;
						Long old = buffer.put(frameNumber, newSurface[0]);
						if (old != null) releaseSurface(videoSource, old);

						if (frameNumber >= fs.frameNumber) {
							surface = newSurface[0];
							break;
						} else {
							ioTime[0] = Long.MIN_VALUE;
						}
					}
				}
				if (surface != 0) {
					ByteBuffer[] buffer = new ByteBuffer[1];
					hr = lockSurface(videoSource, surface, buffer);
					if(hr != 0) {
						logger.error("lockSurface: error " + hr);
					} else {
						fs.surface = surface;
						fs.buffer = buffer[0];
					}
				}
				synchronized (fs) {
					fs.notify();
				}
			}

			for (long surface : buffer.values()) {
				releaseSurface(videoSource, surface);
			}
			buffer.clear();

			closeVideo(videoSource);
			videoSource = 0;
		}
	}


	private static class AudioSample {
		private final long position;
		private final byte[] buffer;
		private final int offset;
		private final int length;
		private boolean result;

		AudioSample(long p, byte[] b, int off, int len) {
			position = p;
			buffer = b;
			offset = off;
			length = len;
		}
	}

	private class AudioThread extends Thread {

		private File file;

		private final BlockingQueue<AudioSample> queue = new LinkedBlockingQueue<AudioSample>();

		private final SortedMap<Long, Long> buffer = new TreeMap<Long, Long>();

		private long position = Long.MIN_VALUE;

		boolean initialize(File file) {
			this.file = file;
			start();
			return read(0, new byte[1], 0, 1);
		}

		void shutdown() {
			try {
				queue.put(new AudioSample(Long.MIN_VALUE, null, 0, 0));
			} catch (InterruptedException e) {
			}
		}

		boolean read(long p, byte[] b, int off, int len) {
			AudioSample as = new AudioSample(p, b, off, len);
			synchronized (as) {
				try {
					queue.put(as);
					as.wait();
				} catch (InterruptedException e) {
				}
			}
			return as.result;
		}

		private void flush() {
			for (Long medbuf : buffer.values()) {
				releaseAudioBuffer(medbuf);
			}
			buffer.clear();
		}

		private void flushHead() {
			SortedMap<Long, Long> head = buffer.headMap((long)(position-3*audioFormat.getFrameRate()));
			for (Long medbuf : head.values()) {
				releaseAudioBuffer(medbuf);
			}
			head.clear();
		}

		public void run() {
			long audioSource;
			long[] result = new long[6];
			int hr = openAudio(file.getAbsolutePath(), result);
			if (hr == 0) {
				audioSource = result[0];
				if (duration == null) {
					duration = new Time(result[1], 10000000);
				}
				audioFormat = new AudioFormat(result[2], (int)result[3], (int)result[4], result[3]>=16, false);
				audioFloat = (result[5] != 0);
				setName(WindowsMediaFoundationSource.this.getClass().getSimpleName()
						+ " (Audio): " + file.getName());
			} else {
				for (AudioSample as; (as = queue.poll()) != null; ) {
					synchronized (as) { as.notify(); }
				}
				return;
			}

			float frameRate = audioFormat.getFrameRate();
			int frameSize = audioFormat.getFrameSize();

			boolean shutdown = false;
			boolean forward = false;
			long prevPosition = 0;
			while (queue.peek() != null || !shutdown) {
				AudioSample as = queue.poll();
				if (as == null) {
					if (forward) {
						long[] ioTime = new long[1];
						long[] outMedbuf = new long[1];
						int[] outLen = new int[1];
						while (queue.peek() == null && position - prevPosition < frameRate) {
							ioTime[0] = Long.MIN_VALUE;
							hr = readAudioBufferAtTime(audioSource, ioTime, outMedbuf, outLen);
							if (hr != 0) {
								logger.error("readAudioBufferAtTime: error " + hr);
								forward = false;
								break;
							}
							if (outLen[0] % frameSize != 0) {
								releaseAudioBuffer(outMedbuf[0]);
								forward = false;
								hr = -1;
								logger.error("readAudioBufferAtTime: invalid sample length: " + outLen[0]);
								break;
							}
							Long old = buffer.put(position, outMedbuf[0]); // ここでoldが返ってくることはないはずだが...
							position += outLen[0] / frameSize;
							if (old != null) releaseAudioBuffer(old);
							flushHead();
						}
					}
					try {
						as = queue.take();
					} catch (InterruptedException e) {
						continue;
					}
				}

				if (as.position == Long.MIN_VALUE) {
					shutdown = true;
					forward = false;
					continue;
				}

				forward = (as.position > prevPosition);
				prevPosition = as.position;

				int off = as.offset;
				int len = as.length;

				long[] ioTime = null;
				if (!buffer.isEmpty() && as.position >= buffer.firstKey() && as.position < position) {
					int[] ioLen = new int[1];
					Long fromKey = buffer.headMap(as.position+1).lastKey();
					for (Long key : buffer.tailMap(fromKey).keySet()) {
						int srcOff = (as.position>key) ? (int)((as.position-key)*frameSize) : 0;
						ioLen[0] = len;
						hr = copyAudioBuffer(buffer.get(key), srcOff, as.buffer, off, ioLen);
						if (hr != 0) {
							logger.error("copyAudioBuffer: error " + hr);
							break;
						}
						off += ioLen[0];
						len -= ioLen[0];
						if (len == 0) {
							break;
						}
					}
					if (hr == 0 && len > 0) {
						ioTime = new long[] { Long.MIN_VALUE };
					}
				} else if (as.position >= position && as.position < position + frameRate) {
					ioTime = new long[] { Long.MIN_VALUE };
				} else {
					flush();
					ioTime = new long[] { (long)(as.position / frameRate * 10000000) };
				}
				if (ioTime != null) {
					long[] outMedbuf = new long[1];
					int[] outLen = new int[1];
					int[] ioLen = new int[1];
					while (len > 0) {
						boolean seek = (ioTime[0] != Long.MIN_VALUE);

						hr = readAudioBufferAtTime(audioSource, ioTime, outMedbuf, outLen);
						if (hr != 0) {
							logger.error("readAudioBufferAtTime: error " + hr);
							break;
						}

						if (outLen[0] % frameSize != 0) {
							releaseAudioBuffer(outMedbuf[0]);
							hr = -1;
							logger.error("readAudioBufferAtTime: invalid sample length: " + outLen[0]);
							break;
						}

						long pos = seek ? Math.round(ioTime[0] * frameRate / 10000000) : position;
						Long old = buffer.put(pos, outMedbuf[0]); // ここでoldが返ってくることはないはずだが...
						position = pos + outLen[0] / frameSize;

						if (old != null) releaseAudioBuffer(old);
						flushHead();

						int srcOff = (as.position>pos) ? (int)((as.position-pos)*frameSize) : 0;
						if (srcOff < outLen[0]) {
							ioLen[0] = len;
							hr = copyAudioBuffer(outMedbuf[0], srcOff, as.buffer, off, ioLen);
							if (hr != 0) {
								logger.error("copyAudioBuffer: error " + hr);
								break;
							}
							off += ioLen[0];
							len -= ioLen[0];
						}

						ioTime[0] = Long.MIN_VALUE;
					}
				}
				synchronized (as) {
					as.result = (hr == 0);
					as.notify();
				}
			}

			flush();
			closeAudio(audioSource);
		}
	}

	private class MFAudioInputStream extends InputStream {

		private final AudioFormat format;

		private long positionInBytes;


		private MFAudioInputStream(AudioFormat format) {
			super();
			this.format = format;
		}

		@Override
		public int read() throws IOException {
			byte[] b = new byte[1];
			return (read(b, 0, 0) != -1) ? b[0] : -1;
		}

		@Override
		public int read(byte[] b, int off, int len) throws IOException {
			int frameSize = format.getFrameSize();

			if (len % frameSize != 0) {
				throw new IOException("can't read the length bytes that is not multiples of frameSize");
			}

			if (!audioThread.read(positionInBytes/frameSize, b, off, len)) {
				// End Of Stream の場合でもここに来るので、例外を投げるのはマズイ。
				//throw new IOException();
				Arrays.fill(b, off, off+len, (byte)0);
			}

			if (audioFloat) {
				for (int i = off; i < len; i += 4) {
					int bits =  (b[i  ] & 0xff)
							 | ((b[i+1] & 0xff) <<  8)
							 | ((b[i+2] & 0xff) << 16)
							 | ((b[i+3] & 0xff) << 24);
					int intValue = (int)(Float.intBitsToFloat(bits) * Integer.MAX_VALUE);

					b[i  ] = (byte)( intValue         & 0xff);
					b[i+1] = (byte)((intValue >>>  8) & 0xff);
					b[i+2] = (byte)((intValue >>> 16) & 0xff);
					b[i+3] = (byte)((intValue >>> 24) & 0xff);
				}
			}

			positionInBytes += len;
			return len;
		}

		@Override
		public long skip(long n) throws IOException {
			long actualSkip = 0;
			if (n > 0) {
				int frameSize = format.getFrameSize();
				actualSkip = (n / frameSize) * frameSize; 
				positionInBytes += actualSkip;
			}
			return actualSkip;
		}
	}


	static { System.loadLibrary("MediaFoundationSource"); }


	public static native int initDXVA2(long hwnd);

	private static native int openVideo(String filename, boolean sysMem, long[] outResult);

	private static native void closeVideo(long videoSource);

	private static native int frameSurfaceAtTime(long videoSource, long[] ioTime, long limitTime, long[] outSurface);

	private static native int lockSurface(long videoSource, long surface, ByteBuffer[] outBuffer);

	private static native int unlockSurface(long videoSource, long surface);

	private static native int releaseSurface(long videoSource, long surface);


	private static native int openAudio(String filename, long[] outResult);

	private static native void closeAudio(long audioSource);

	private static native int readAudioBufferAtTime(long audioSource, long[] ioTime, long[] outMedbuf, int[] outLen);

	private static native int copyAudioBuffer(long medbuf, int srcOff, byte[] dst, int dstOff, int[] ioLen);

	private static native int releaseAudioBuffer(long medbuf);

}
