/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import ch.kuramo.javie.api.services.IVBOCache;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class VBOCacheProxy implements IVBOCache {

	private static final ThreadLocal<VBOCacheImpl>
			threadLocal = new ThreadLocal<VBOCacheImpl>();


	static class Lifecycle {

		private final Injector injector;

		@Inject
		Lifecycle(Injector injector) {
			this.injector = injector;
		}

		void createImpl() {
			if (threadLocal.get() != null) {
				throw new IllegalStateException("VBOCacheImpl already exists on this thread.");
			}
			VBOCacheImpl impl = injector.getInstance(VBOCacheImpl.class);
			threadLocal.set(impl);
		}

		void disposeImpl() {
			VBOCacheImpl impl = threadLocal.get();
			if (impl != null) {
				impl.dispose();
				threadLocal.set(null);
			}
		}

		void pollAndDelete() {
			VBOCacheImpl impl = threadLocal.get();
			if (impl != null) {
				impl.pollAndDelete();
			}
		}
	}


	private VBOCacheImpl getImpl() {
		VBOCacheImpl impl = threadLocal.get();
		if (impl == null) {
			throw new IllegalStateException("No VBOCacheImpl exist on this thread.");
		}
		return impl;
	}

	public Object put(Object referent, VBOCacheRecord record) {
		return getImpl().put(referent, record);
	}

	public VBOCacheRecord get(Object referent) {
		return getImpl().get(referent);
	}

	public VBOCacheRecord remove(Object referent) {
		return getImpl().remove(referent);
	}

	public Object delete(Object referent) {
		return getImpl().delete(referent);
	}

}
