/*
 * Copyright (c) 2014 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#import <sys/socket.h>
#import <sys/un.h>
#import <pthread.h>
#import "QTCoreVideoInput.h"

static void* thread_func(void* arg) {
    @autoreleasepool {
        void** thread_args = arg;
        int fd = (int)thread_args[0];
        AudioInput* audioInput = thread_args[1];
        
        EnterMoviesOnThread(0);
        CSSetComponentsThreadMode(kCSAcceptAllComponentsMode);
        
        OSStatus error = AttachMovieToCurrentThread([audioInput movie]);
        if (error == noErr) {
            int desc[4] = {0};
            
            void** buffer = NULL;
            int bufferSize = 0;
            
            for (;;) @autoreleasepool {
                int buf[8];
                if (recv(fd, buf, sizeof(buf), MSG_WAITALL) == -1) {
                    perror("recv");
                    break;
                }
                
                if (memcmp(desc, buf, sizeof(desc)) != 0) {
                    memcpy(desc, buf, sizeof(desc));
                    NSLog(@"setAudioDescription: %d, %d, %d, %d", buf[0], buf[1], buf[2], buf[3]);
                    
                    NSDictionary* description = [NSDictionary dictionaryWithObjectsAndKeys:
                                                 [NSNumber numberWithInt:buf[0]], @"channels",
                                                 [NSNumber numberWithInt:buf[1]], @"sampleRate",
                                                 [NSNumber numberWithInt:buf[2]], @"sampleSize",
                                                 [NSNumber numberWithBool:buf[3]], @"floatingPoint",
                                                 NULL];
                    error = [audioInput setAudioDescription:description];
                    if (error != noErr) {
                        break;
                    }
                }
                
                long long timeValue = *((long long*)(buf+4));
                int timeScale = buf[6];
                int frameCount = buf[7];
                
                int size = [audioInput bytesPerFrame] * frameCount;
                if (size > bufferSize) {
                    buffer = reallocf(buffer, size);
                    if (buffer == NULL) {
                        perror("reallocf");
                        break;
                    }
                    bufferSize = size;
                }
                
                error = [audioInput audioChunkFromTime:QTMakeTime(timeValue, timeScale) :buffer :0 :frameCount];
                if (error != noErr) {
                    break;
                }
                
                if (send(fd, buffer, size, 0) == -1) {
                    perror("send");
                    break;
                }
            }
            
            free(buffer);
        }
        
        [audioInput release];
        ExitMoviesOnThread();
        
        shutdown(fd, SHUT_RDWR);
        close(fd);
        
        NSLog(@"closed");
    }
    
    alarm(3);
    return NULL;
}

static void alarm_handler(int sig)
{
    if (getppid() == 1) {
        exit(0);
    }
    alarm(60);
}

int main (int argc, const char * argv[])
{
    if (argc != 2) {
        fprintf(stderr, "usage: %s socket_path\n", argv[0]);
        exit(1);
    }
    
    struct sockaddr_un server = {0};
    
    if (strlen(argv[1]) >= sizeof(server.sun_path)) {
        fprintf(stderr, "socket path is too long: %s\n", argv[1]);
        exit(1);
    }
    
    server.sun_family = AF_UNIX;
    strcpy(server.sun_path, argv[1]);
    
    //if (remove(server.sun_path) == -1 && errno != ENOENT) {
    //    perror("remove");
    //    exit(1);
    //}
    
    int sock = socket(AF_UNIX, SOCK_STREAM, 0);
    if (sock == -1) {
        perror("socket");
        exit(1);
    }
    
    if (bind(sock, (struct sockaddr *)&server, sizeof(server)) == -1) {
        perror("bind");
        exit(1);
    }
    
    if (listen(sock, 100) == -1) {
        perror("listen");
        exit(1);
    }
    
    if (signal(SIGALRM, alarm_handler) == SIG_ERR) {
        perror("signal");
        exit(1);
    }
    alarm(60);
    
    EnterMovies();
    
    for (;;) {
        int fd = accept(sock, NULL, NULL);
        if (fd == -1) {
            perror("accept");
            exit(1);
        }
        
        int fnlen;
        if (recv(fd, &fnlen, sizeof(fnlen), MSG_WAITALL) == -1) {
            perror("recv");
            exit(1);
        }
        
        char fn[fnlen + 1];
        if (recv(fd, fn, fnlen, MSG_WAITALL) == -1) {
            perror("recv");
            exit(1);
        }
        fn[fnlen] = 0;
        
        @autoreleasepool {
            NSString* filename = [NSString stringWithCString:fn encoding:NSUTF8StringEncoding];
            NSLog(@"%@", filename);
            
            AudioInput* audioInput = [[AudioInput alloc] initWithFile:filename];
            
            long long buf[4] = { (audioInput != NULL), [audioInput sampleCount],
                                 [audioInput duration], [audioInput timeScale] };
            
            if (send(fd, buf, sizeof(buf), 0) == -1) {
                perror("send");
                exit(1);
            }
            
            if (audioInput) {
                DetachMovieFromCurrentThread([audioInput movie]);
                
                void* thread_args[] = { (void*)fd, audioInput };
                pthread_t thread;
                int error = pthread_create(&thread, NULL, thread_func, thread_args);
                if (error != 0) {
                    fprintf(stderr, "pthread_create failed: error=%d\n", error);
                    exit(1);
                }
            }
        }
    }
    
    return 0;
}
