/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#import "QTCoreVideoInput.h"
#import <OpenGL/glu.h>
#import <AudioToolbox/AudioFormat.h>


@interface InputBase (private)

-(void) openMovie:(CFStringRef)filename;
-(BOOL) getMediaInfo:(Track)track;

@end

@implementation InputBase

-(void) openMovie:(CFStringRef)filename
{
	Boolean trueValue = true;
	QTVisualContextRef visualContext = NULL;
	
	QTNewMoviePropertyElement newMovieProperties[] = {
		{ kQTPropertyClass_DataLocation,     kQTDataLocationPropertyID_CFStringPosixPath, sizeof(CFStringRef),   &filename,      0 },
		{ kQTPropertyClass_Context,          kQTContextPropertyID_VisualContext,          sizeof(visualContext), &visualContext, 0 },
		{ kQTPropertyClass_NewMovieProperty, kQTNewMoviePropertyID_Active,                sizeof(trueValue),     &trueValue,     0 },
		{ kQTPropertyClass_NewMovieProperty, kQTNewMoviePropertyID_DontInteractWithUser,  sizeof(trueValue),     &trueValue,     0 }
	};
	
	OSStatus error = NewMovieFromProperties(sizeof(newMovieProperties) / sizeof(newMovieProperties[0]),
											newMovieProperties, 0, NULL, &mMovie);
	if (error != noErr) {
		NSLog(@"NewMovieFromProperties: error %d", error);
		mMovie = NULL;
	}
}

-(BOOL) getMediaInfo:(Track)track
{
	Media media = GetTrackMedia(track);
	if (media == NULL) {
		NSLog(@"can't get media");
		return FALSE;
	}
	
	GetMediaHandlerDescription(media, &mMediaType, NULL, NULL);
	OSErr error = GetMoviesError();
	if (error != noErr) {
		NSLog(@"GetMediaHandlerDescription: error %d", error);
		return FALSE;
	}
	
	mSampleCount = GetMediaSampleCount(media);
	error = GetMoviesError();
	if (error != noErr) {
		NSLog(@"GetMediaSampleCount: error %d", error);
		return FALSE;
	}
	
	mDuration = GetMediaDisplayDuration(media);
	error = GetMoviesError();
	if (error != noErr) {
		NSLog(@"GetMediaDisplayDuration: error %d", error);
		return FALSE;
	}
	
	mTimeScale = GetMediaTimeScale(media);
	error = GetMoviesError();
	if (error != noErr) {
		NSLog(@"GetMediaTimeScale: error %d", error);
		return FALSE;
	}
	
	if (mMediaType == MPEGMediaType) {
		MediaHandler mpegMediaHandler = GetMediaHandler(media);
		error = GetMoviesError();
		if (error != noErr) {
			NSLog(@"GetMediaHandler: error %d", error);
			return FALSE;
		}
		
		MHInfoEncodedFrameRateRecord encodedFrameRate;
		Size encodedFrameRateSize = sizeof(encodedFrameRate);
		
		MoviesTask(mMovie, 0);
		
		error = MediaGetPublicInfo(mpegMediaHandler, kMHInfoEncodedFrameRate,
								   &encodedFrameRate, &encodedFrameRateSize);
		if (error != noErr) {
			NSLog(@"MediaGetPublicInfo: error %d", error);
			return FALSE;
		}
		
		mSampleCount = FixedToFloat(encodedFrameRate.encodedFrameRate) * mDuration / mTimeScale;
		
	} else if (mMediaType == kQTQuartzComposerMediaType) {
		// デュレーションは、JNI_OnLoad内でQuartzComposerDefaultMovieDurationを300*60*60秒に設定してある
		mDuration = mTimeScale * 300*60*60;
		mSampleCount = 0;
	}
	
	return TRUE;
}

-(void) dealloc
{
	if (mMovie != NULL) {
		DisposeMovie(mMovie);
		mMovie = NULL;
	}
	
	[super dealloc];
}

-(long) sampleCount
{
	return mSampleCount;
}

-(TimeValue64) duration
{
	return mDuration;
}

-(TimeScale) timeScale
{
	return mTimeScale;
}

-(Movie) movie
{
    return mMovie;
}

@end


#ifndef JAVIE_AUDIO_SERVER

@interface VideoInput (private)

-(void) setGraphicsMode;

@end

@implementation VideoInput

-(id) init
{
	NSException *exception = [NSException exceptionWithName:@"UnsupportedOperationException"
													 reason:@"init method is not availavle, use initWith*** method instead."
												   userInfo:NULL];
	@throw exception;
}

-(id) initWithFile:(NSString*)filename
				  :(CGLContextObj)cglContext
				  :(CGLPixelFormatObj)cglPixelFormat
{
	[super init];
	
	[self openMovie:(CFStringRef)filename];
	if (mMovie == NULL) {
		[self dealloc];
		return NULL;
	}
	
	
	Rect movieBox;
	GetMovieBox(mMovie, &movieBox);
	OSStatus error = GetMoviesError();
	if (error != noErr) {
		NSLog(@"GetMovieBox: error %d", error);
		[self dealloc];
		return NULL;
	}
	
	mWidth = mTargetWidth = movieBox.right - movieBox.left;
	mHeight = mTargetHeight = movieBox.bottom - movieBox.top;
	
	
	Track visualTrack = GetMovieIndTrackType(mMovie, 1, VisualMediaCharacteristic,
											 movieTrackCharacteristic | movieTrackEnabledOnly);
	if (visualTrack == NULL) {
		NSLog(@"can't get visual track");
		[self dealloc];
		return NULL;
	}
	if (![self getMediaInfo:visualTrack]) {
		NSLog(@"can't get visual media info");
		[self dealloc];
		return NULL;
	}
	
//	if (mMediaType != kQTQuartzComposerMediaType) {
//		[self setGraphicsMode];
//	}
	
	
	NSDictionary* dimensions = [NSDictionary dictionaryWithObjectsAndKeys:
								[NSNumber numberWithInt:mTargetWidth], kQTVisualContextTargetDimensions_WidthKey, 
								[NSNumber numberWithInt:mTargetHeight], kQTVisualContextTargetDimensions_HeightKey,
								NULL];
	
	NSDictionary* attrs = [NSDictionary dictionaryWithObjectsAndKeys:
						   dimensions, kQTVisualContextTargetDimensionsKey,
						   [[NSColorSpace genericRGBColorSpace] CGColorSpace], kQTVisualContextOutputColorSpaceKey,
						   NULL];
	
	error = QTOpenGLTextureContextCreate(
							kCFAllocatorDefault, cglContext, cglPixelFormat,
							(CFDictionaryRef)attrs, &mVisualContext);
	
	if (error != noErr) {
		NSLog(@"QTOpenGLTextureContextCreate: error %d", error);
		[self dealloc];
		return NULL;
	}
	
	error = SetMovieVisualContext(mMovie, mVisualContext);
	if (error != noErr) {
		NSLog(@"SetMovieVisualContext: error %d", error);
		[self dealloc];
		return NULL;
	}
	
	return self;
}

-(void) setGraphicsMode
{
//	int bottomLayer = 0x80000000;
//	Track bottomTrack = NULL;
//	
//	for (int i = 1; ; ++i) {
//		Track track = GetMovieIndTrackType(mMovie, i, VisualMediaCharacteristic,
//										   movieTrackCharacteristic | movieTrackEnabledOnly);
//		if (track == NULL) {
//			break;
//		}
//		int layer = GetTrackLayer(track);
//		if (layer > bottomLayer) {
//			bottomLayer = layer;
//			bottomTrack = track;
//		}
//	}
//	
//	if (bottomTrack != NULL) {
//		MediaHandler mh = GetMediaHandler(GetTrackMedia(bottomTrack));
//		ComponentResult result = MediaSetGraphicsMode(mh, graphicsModeComposition, NULL);
//		if (result != noErr) {
//			NSLog(@"MediaSetGraphicsMode: error %d", result);
//		}
//	}
	
	for (int i = 1; ; ++i) {
		Track track = GetMovieIndTrackType(mMovie, i, VisualMediaCharacteristic,
										   movieTrackCharacteristic | movieTrackEnabledOnly);
		if (track == NULL) {
			break;
		}
		MediaHandler mh = GetMediaHandler(GetTrackMedia(track));
		ComponentResult result = MediaSetGraphicsMode(mh, graphicsModeComposition, NULL);
		if (result != noErr) {
			NSLog(@"MediaSetGraphicsMode: error %d", result);
		}
	}
}

-(void) dealloc
{
	SetMovieVisualContext(mMovie, NULL);
	
	CVOpenGLTextureRelease(mCurrentFrame);
	mCurrentFrame = NULL;
	
	QTVisualContextRelease(mVisualContext);
	mVisualContext = NULL;
	
	[super dealloc];
}

-(long) width
{
	return mWidth;
}

-(long) height
{
	return mHeight;
}

-(OSStatus) frameImageAtTime:(QTTime)time
							:(GLint)texture
							:(GLint)width
							:(GLint)height
							:(int)alphaType
							:(float)colorMatteR
							:(float)colorMatteG
							:(float)colorMatteB
							:(BOOL)flipVertical
{
	if (width != mTargetWidth || height != mTargetHeight) {
		mTargetWidth = width;
		mTargetHeight = height;
		
		NSDictionary* dimensions = [NSDictionary dictionaryWithObjectsAndKeys:
									[NSNumber numberWithInt:width], kQTVisualContextTargetDimensions_WidthKey, 
									[NSNumber numberWithInt:height], kQTVisualContextTargetDimensions_HeightKey,
									NULL];
		
		QTVisualContextSetAttribute(mVisualContext, kQTVisualContextTargetDimensionsKey, dimensions);
	}
	
	TimeRecord timeRec;
	if (!QTGetTimeRecord(time, &timeRec)) {
		NSLog(@"QTGetTimeRecord: failed");
		return -1;
	}
	
	SetMovieTime(mMovie, &timeRec);
	MoviesTask(mMovie, 0);
	
	CVOpenGLTextureRelease(mCurrentFrame);
	mCurrentFrame = NULL;
	
	OSStatus error = QTVisualContextCopyImageForTime(mVisualContext, NULL, NULL, &mCurrentFrame);
	if (error != noErr) {
		NSLog(@"QTVisualContextCopyImageForTime: error %d", error);
		return error;
	}
	
	GLenum target = CVOpenGLTextureGetTarget(mCurrentFrame);
	GLint name = CVOpenGLTextureGetName(mCurrentFrame);
	
	GLfloat topLeft[2], topRight[2], bottomRight[2], bottomLeft[2];
	CVOpenGLTextureGetCleanTexCoords(mCurrentFrame, bottomLeft, bottomRight, topRight, topLeft);
	
	
	glPushAttrib(GL_COLOR_BUFFER_BIT | GL_ENABLE_BIT | GL_TEXTURE_BIT | GL_CURRENT_BIT);
	
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, texture, 0);
	glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);
	
	switch (alphaType) {
		// AlphaType.IGNORE
		case 0:
			glClearColor(0, 0, 0, 1);
			glClear(GL_COLOR_BUFFER_BIT);
			
			glEnable(GL_BLEND);
			glBlendFuncSeparate(GL_ONE, GL_ZERO, GL_ZERO, GL_ONE);
			glBlendEquation(GL_FUNC_ADD);
			break;
			
		// AlphaType.STRAIGHT
		case 1:
			glClearColor(0, 0, 0, 0);
			glClear(GL_COLOR_BUFFER_BIT);

			glEnable(GL_BLEND);
			glBlendFuncSeparate(GL_SRC_ALPHA, GL_ZERO, GL_ONE, GL_ZERO);
			glBlendEquation(GL_FUNC_ADD);
			break;
			
		// AlphaType.PREMULTIPLIED
		case 2:
			if (colorMatteR == 0 && colorMatteG == 0 && colorMatteB == 0) {
				glDisable(GL_BLEND);

			} else {
				glClearColor(colorMatteR, colorMatteG, colorMatteB, 1);
				glClear(GL_COLOR_BUFFER_BIT);
				
				glEnable(GL_BLEND);
				glBlendFuncSeparate(GL_ONE, GL_ONE_MINUS_SRC_ALPHA, GL_ONE, GL_ZERO);
				glBlendEquation(GL_FUNC_SUBTRACT);
			}
			break;
	}
	
	glViewport(0, 0, width, height);
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if (flipVertical) {
		gluOrtho2D(0, width, height, 0);
	} else {
		gluOrtho2D(0, width, 0, height);
	}
	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	glBindTexture(target, name);
	glEnable(target);
	
	glColor4d(1, 1, 1, 1);
	
	glBegin(GL_QUADS);
	glTexCoord2fv(topLeft);     glVertex2i(0, 0);
	glTexCoord2fv(topRight);    glVertex2i(width, 0);
	glTexCoord2fv(bottomRight); glVertex2i(width, height);
	glTexCoord2fv(bottomLeft);  glVertex2i(0, height);
	glEnd();
	
	glFinish();
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, 0, 0);
	
	glPopAttrib();
	
	QTVisualContextTask(mVisualContext);
	
	return noErr;
}

@end

#endif


@implementation AudioInput

-(id) init
{
	NSException *exception = [NSException exceptionWithName:@"UnsupportedOperationException"
													 reason:@"init method is not availavle, use initWith*** method instead."
												   userInfo:NULL];
	@throw exception;
}

-(id) initWithFile:(NSString*)filename
{
	[super init];
	
	[self openMovie:(CFStringRef)filename];
	if (mMovie == NULL) {
		[self dealloc];
		return NULL;
	}
	
	Track audioTrack = GetMovieIndTrackType(mMovie, 1, AudioMediaCharacteristic,
											movieTrackCharacteristic | movieTrackEnabledOnly);
	if (audioTrack == NULL) {
		NSLog(@"can't get audio track");
		[self dealloc];
		return NULL;
	}
	if (![self getMediaInfo:audioTrack]) {
		NSLog(@"can't get audio media info");
		[self dealloc];
		return NULL;
	}
	
	return self;
}

-(void) dealloc
{
	MovieAudioExtractionEnd(mAudioExtraction);
	mAudioExtraction = NULL;
	
	[super dealloc];
}

-(UInt32) bytesPerFrame
{
    return mASBD.mBytesPerFrame;
}

-(OSStatus) setAudioDescription:(NSDictionary*)description
{
	int channels = [(NSNumber*)[description objectForKey:@"channels"] intValue];
	int sampleRate = [(NSNumber*)[description objectForKey:@"sampleRate"] intValue];
	int sampleSize = [(NSNumber*)[description objectForKey:@"sampleSize"] intValue];
	BOOL floatingPoint = [(NSNumber*)[description objectForKey:@"floatingPoint"] boolValue];
	
	mASBD.mFormatID = kAudioFormatLinearPCM;
	mASBD.mFormatFlags = kAudioFormatFlagIsPacked | kAudioFormatFlagsNativeEndian
			| (floatingPoint ? kAudioFormatFlagIsFloat : kAudioFormatFlagIsSignedInteger);
	mASBD.mSampleRate = sampleRate;
	mASBD.mBitsPerChannel = sampleSize * 8;
	mASBD.mChannelsPerFrame = channels;
	mASBD.mBytesPerFrame = sampleSize * mASBD.mChannelsPerFrame;
	mASBD.mFramesPerPacket = 1;
	mASBD.mBytesPerPacket = mASBD.mBytesPerFrame * mASBD.mFramesPerPacket;
	
	
	MovieAudioExtractionEnd(mAudioExtraction);
	mAudioExtraction = NULL;
	
	OSStatus error = MovieAudioExtractionBegin(mMovie, 0, &mAudioExtraction);
	if (error != noErr) {
		NSLog(@"MovieAudioExtractionBegin: error %d", error);
		return error;
	}
	
	error = MovieAudioExtractionSetProperty(mAudioExtraction,
				kQTPropertyClass_MovieAudioExtraction_Audio,
				kQTMovieAudioExtractionAudioPropertyID_AudioStreamBasicDescription,
				sizeof(AudioStreamBasicDescription), &mASBD);
	if (error != noErr) {
		NSLog(@"MovieAudioExtractionSetProperty: error %d", error);
		return error;
	}
	
	AudioChannelLayoutTag layoutTag;
	switch (channels) {
		case 2:
			layoutTag = kAudioChannelLayoutTag_Stereo;
			break;
		default:
			NSLog(@"unsupported number of channels: channels=%d", channels);
			return paramErr;
	}
	
	UInt32 layoutSize;
	error = AudioFormatGetPropertyInfo(
				kAudioFormatProperty_ChannelLayoutForTag,
				sizeof(AudioChannelLayoutTag), &layoutTag, &layoutSize);
	if (error != noErr) {
		NSLog(@"AudioFormatGetPropertyInfo: error %d", error);
		return error;
	}
	
	NSMutableData* layoutData = [NSMutableData dataWithLength:layoutSize];
	error = AudioFormatGetProperty(
				kAudioFormatProperty_ChannelLayoutForTag,
				sizeof(AudioChannelLayoutTag), &layoutTag, &layoutSize, [layoutData mutableBytes]);
	if (error != noErr) {
		NSLog(@"AudioFormatGetProperty: error %d", error);
		return error;
	}
	
	error = MovieAudioExtractionSetProperty(mAudioExtraction,
				kQTPropertyClass_MovieAudioExtraction_Audio,
				kQTMovieAudioExtractionAudioPropertyID_AudioChannelLayout,
				layoutSize, [layoutData bytes]);
	if (error != noErr) {
		NSLog(@"MovieAudioExtractionSetProperty: error %d", error);
		return error;
	}
	
	return noErr;
}

-(OSStatus) audioChunkFromTime:(QTTime)time
							  :(void*)buffer
							  :(long)offset
							  :(long)frameCount
{
	if (mAudioExtraction == NULL) {
		NSLog(@"AudioExtraction is not initialized");
		return -1;
	}
	
	TimeRecord timeRec;
	if (!QTGetTimeRecord(time, &timeRec)) {
		NSLog(@"QTGetTimeRecord: failed");
		return -1;
	}
	
	OSStatus error = MovieAudioExtractionSetProperty(mAudioExtraction,
							kQTPropertyClass_MovieAudioExtraction_Movie,
							kQTMovieAudioExtractionMoviePropertyID_CurrentTime,
							sizeof(TimeRecord), &timeRec);
	if (error != noErr) {
		NSLog(@"MovieAudioExtractionSetProperty: error %d", error);
		return error;
	}
	
	char* p = (char*)buffer + mASBD.mBytesPerFrame * offset;
	
	AudioBufferList bufList;
	bufList.mNumberBuffers = 1;
	bufList.mBuffers[0].mNumberChannels = mASBD.mChannelsPerFrame;
	bufList.mBuffers[0].mDataByteSize = mASBD.mBytesPerFrame * frameCount;
	bufList.mBuffers[0].mData = p;
	
	UInt32 flags;
	UInt32 filledFrames = frameCount;
	
	error = MovieAudioExtractionFillBuffer(mAudioExtraction, &filledFrames, &bufList, &flags);
	if (error != noErr) {
		NSLog(@"MovieAudioExtractionFillBuffer: error %d", error);
		return error;
	}
	
	if (filledFrames < frameCount) {
		UInt32 filledBytes = filledFrames * mASBD.mBytesPerFrame;
		UInt32 restBytes = (frameCount - filledFrames) * mASBD.mBytesPerFrame;
		memset(p + filledBytes, 0, restBytes);
	}
	
	return noErr;
}

@end
