/*
 * Copyright (c) 2014 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#import "QTKitSourceJNI.h"
#import <QTKit/QTKit.h>
#import <OpenGL/glu.h>
#import <objc/runtime.h>
#import <sys/socket.h>
#import <sys/un.h>
#import <sys/stat.h>
#import <dirent.h>

#define JLONG(p) ((jlong) (pointer_t) p)
#define POINTER(type, p) ((type) (pointer_t) p)

// ポインタ値をキーとして使う。
static int textureCacheKey;

// ファイル名のアドレスをキーにして
// objc_setAssociatedObjectでソケットのファイルディスクリプタを保持する。
static NSObject* audioSocketHolder;


static NSString* jstringToNSString(JNIEnv* env, jstring jstr)
{
	const jchar* chars = (*env)->GetStringChars(env, jstr, NULL);
	NSString* nsstr = [NSString stringWithCharacters:(UniChar*)chars
											  length:(*env)->GetStringLength(env, jstr)];
	(*env)->ReleaseStringChars(env, jstr, chars);
	return nsstr;
}

JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
    audioSocketHolder = [[NSObject alloc] init];
    return JNI_VERSION_1_4;
}

JNIEXPORT jlongArray JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTKitSource_openVideo
    (JNIEnv *env, jobject jthis, jstring jfilename, jlong pixelFormat)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
    NSString* filename = jstringToNSString(env, jfilename);
    __block QTMovie* movie = NULL;
    dispatch_sync(dispatch_get_main_queue(), ^{
        NSError* error;
        movie = [[QTMovie alloc] initWithFile:filename error:&error];
        if (movie) {
            [movie detachFromCurrentThread];
        } else {
            NSLog(@"QTMovie initWithFile: error %ld, %@", error.code, error.description);
        }
    });
	
	jlongArray result = NULL;
	
    if (movie) {
        [QTMovie enterQTKitOnThread];
        [movie attachToCurrentThread];
        
        NSArray* tracks = [movie tracksOfMediaType:QTMediaTypeVideo];
        if ([tracks count] > 0) {
            CVOpenGLTextureCacheRef textureCache;
            CVReturn error = CVOpenGLTextureCacheCreate(kCFAllocatorDefault, NULL, CGLGetCurrentContext(),
                                                        [POINTER(NSOpenGLPixelFormat*, pixelFormat) CGLPixelFormatObj],
                                                        NULL, &textureCache);
            if (error == kCVReturnSuccess) {
                objc_setAssociatedObject(movie, &textureCacheKey,
                                         [NSValue valueWithPointer:textureCache],
                                         OBJC_ASSOCIATION_RETAIN);
                
                NSDictionary* movieAttrs = [movie movieAttributes];
                NSSize naturalSize = [[movieAttrs valueForKey:QTMovieNaturalSizeAttribute] sizeValue];
                //QTTime duration = [[movieAttrs valueForKey:QTMovieDurationAttribute] QTTimeValue];
                
                QTMedia* media = [[tracks objectAtIndex:0] media];
                NSDictionary* mediaAttrs = [media mediaAttributes];
                long sampleCount = [[mediaAttrs valueForKey:QTMediaSampleCountAttribute] longValue];
                QTTime duration = [[mediaAttrs valueForKey:QTMediaDurationAttribute] QTTimeValue];
                
                jlong buf[6] = { JLONG(movie), naturalSize.width, naturalSize.height,
                    sampleCount, duration.timeValue, duration.timeScale };
                result = (*env)->NewLongArray(env, 6);
                (*env)->SetLongArrayRegion(env, result, 0, 6, buf);
            }
        }
    }
    
    if (result == NULL) {
        [movie release];
        [QTMovie exitQTKitOnThread];
    }
    
	[pool release];
	return result;
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTKitSource_closeVideo
    (JNIEnv *env, jobject jthis, jlong movieAddress)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    
    QTMovie* movie = POINTER(QTMovie*, movieAddress);

    CVOpenGLTextureCacheRef textureCache = [objc_getAssociatedObject(movie, &textureCacheKey) pointerValue];
    CVOpenGLTextureCacheRelease(textureCache);
    objc_setAssociatedObject(movie, &textureCacheKey, NULL, OBJC_ASSOCIATION_RETAIN);
    
    [movie release];
    [QTMovie exitQTKitOnThread];
    
	[pool release];
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTKitSource_frameImageAtTime
    (JNIEnv *env, jobject jthis, jlong movieAddress,
     jlong timeValue, jint timeScale, jint texture, jint width, jint height,
     jint alphaType, jfloat colorMatteR, jfloat colorMatteG, jfloat colorMatteB, jboolean flipVertical)
{
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    
    QTMovie* movie = POINTER(QTMovie*, movieAddress);
    
    CVOpenGLTextureCacheRef textureCache = NULL;
    CVOpenGLTextureRef textureRef = NULL;
    
    NSDictionary* attrs = [NSDictionary dictionaryWithObjectsAndKeys:
                           [NSNumber numberWithBool:NO], QTMovieFrameImageDeinterlaceFields,
                           [NSNumber numberWithBool:NO], QTMovieFrameImageSingleField,
                           [NSNumber numberWithBool:YES], QTMovieFrameImageSessionMode,
                           QTMovieFrameImageTypeCVPixelBufferRef, QTMovieFrameImageType,
                           NULL];
    NSError* error = NULL;
    CVPixelBufferRef pixelBuffer = [movie frameImageAtTime:QTMakeTime(timeValue, timeScale) withAttributes:attrs error:&error];
    if (error != NULL) {
        NSLog(@"QTMovie frameImageAtTime: error %ld, %@", error.code, error.description);
    } else if (pixelBuffer == NULL) {
        NSLog(@"QTMovie frameImageAtTime: null");
    } else {
        textureCache = [objc_getAssociatedObject(movie, &textureCacheKey) pointerValue];
        CVReturn cvret = CVOpenGLTextureCacheCreateTextureFromImage(
                            kCFAllocatorDefault, textureCache, pixelBuffer, NULL, &textureRef);
        if (cvret != kCVReturnSuccess) {
            NSLog(@"CVOpenGLTextureCacheCreateTextureFromImage: error %d", cvret);
            [pool release];
            return cvret;
        }
    }
    
	glPushAttrib(GL_COLOR_BUFFER_BIT | GL_ENABLE_BIT | GL_TEXTURE_BIT | GL_CURRENT_BIT);
	
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, texture, 0);
	glDrawBuffer(GL_COLOR_ATTACHMENT0_EXT);
	
	switch (alphaType) {
        // AlphaType.IGNORE
		case 0:
			glClearColor(0, 0, 0, 1);
			glClear(GL_COLOR_BUFFER_BIT);
			
			glEnable(GL_BLEND);
			glBlendFuncSeparate(GL_ONE, GL_ZERO, GL_ZERO, GL_ONE);
			glBlendEquation(GL_FUNC_ADD);
			break;
			
        // AlphaType.STRAIGHT
		case 1:
			glClearColor(0, 0, 0, 0);
			glClear(GL_COLOR_BUFFER_BIT);
            
			glEnable(GL_BLEND);
			glBlendFuncSeparate(GL_SRC_ALPHA, GL_ZERO, GL_ONE, GL_ZERO);
			glBlendEquation(GL_FUNC_ADD);
			break;
			
        // AlphaType.PREMULTIPLIED
		case 2:
			if (colorMatteR == 0 && colorMatteG == 0 && colorMatteB == 0) {
				glDisable(GL_BLEND);
                
			} else {
				glClearColor(colorMatteR, colorMatteG, colorMatteB, 1);
				glClear(GL_COLOR_BUFFER_BIT);
				
				glEnable(GL_BLEND);
				glBlendFuncSeparate(GL_ONE, GL_ONE_MINUS_SRC_ALPHA, GL_ONE, GL_ZERO);
				glBlendEquation(GL_FUNC_SUBTRACT);
			}
			break;
	}
    
    if (textureRef != NULL) {
        GLenum target = CVOpenGLTextureGetTarget(textureRef);
        GLint name = CVOpenGLTextureGetName(textureRef);
        
        GLfloat topLeft[2], topRight[2], bottomRight[2], bottomLeft[2];
        CVOpenGLTextureGetCleanTexCoords(textureRef, bottomLeft, bottomRight, topRight, topLeft);
        
        glViewport(0, 0, width, height);
        
        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        if (flipVertical) {
            gluOrtho2D(0, width, height, 0);
        } else {
            gluOrtho2D(0, width, 0, height);
        }
        
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        
        glBindTexture(target, name);
        glEnable(target);
        
        glColor4d(1, 1, 1, 1);
        
        glBegin(GL_QUADS);
        glTexCoord2fv(topLeft);     glVertex2i(0, 0);
        glTexCoord2fv(topRight);    glVertex2i(width, 0);
        glTexCoord2fv(bottomRight); glVertex2i(width, height);
        glTexCoord2fv(bottomLeft);  glVertex2i(0, height);
        glEnd();
        
        CVOpenGLTextureCacheFlush(textureCache, 0);
        CVOpenGLTextureRelease(textureRef);
    }
	
	glFinish();
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, 0, 0);
	
	glPopAttrib();
	
    [pool release];
	return noErr;
}

static void closeall()
{
    DIR* dir = opendir("/dev/fd");
    if (dir == NULL) {
        perror("opendir");
        return;
    }
    for (struct dirent* e; (e = readdir(dir)) != NULL; ) {
        int fd = atoi(e->d_name);
        if (fd >= 3) {
            close(fd);
        }
    }
}

static int getAudioSocket(NSString* filename, long long info[4]) {
    @synchronized (audioSocketHolder) {
        static pid_t audioServerPid;
        static char socketPath[64];
        
        if (audioServerPid != 0) {
            int status;
            if (waitpid(audioServerPid, &status, WNOHANG) != 0) {
                audioServerPid = 0;
                objc_removeAssociatedObjects(audioSocketHolder);
            }
        }
        
        if (audioServerPid == 0) {
            strcpy(socketPath, "/tmp/JavieAudio-XXXXXX");
            if (mkdtemp(socketPath) == NULL) {
                perror("mkdtemp");
                return -1;
            }
            strcat(socketPath, "/socket");
            
            pid_t pid = fork();
            if (pid == -1) {
                perror("fork");
                return -1;
            } else if (pid == 0) {
                closeall();
                char procName[64];
                snprintf(procName, sizeof(procName), "JavieAudioServer-(%d)", getppid());
                char* argv[] = { procName, socketPath, NULL };
                execv("JavieAudioServer", argv);
                perror("execv");
                exit(1);
            }
            
            struct stat st;
            for (int i = 0; i < 30; ++i) {
                usleep(100*1000);
                if (stat(socketPath, &st) == 0) {
                    audioServerPid = pid;
                    break;
                }
            }
            if (audioServerPid == 0) {
                kill(pid, SIGINT);
                return -1;
            }
        }
        
        int sock = [objc_getAssociatedObject(audioSocketHolder, filename) intValue];
        if (sock == 0) {
            sock = socket(AF_UNIX, SOCK_STREAM, 0);
            if (sock == -1) {
                perror("socket");
                return -1;
            }
            
            struct sockaddr_un addr = {0};
            addr.sun_family = AF_UNIX;
            strcpy(addr.sun_path, socketPath);
            
            if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) == -1) {
                perror("connect");
                close(sock);
                return -1;
            }
            
            const char* utf8 = [filename UTF8String];
            size_t len = strlen(utf8);
            char sendbuf[4+len];
            *((int*)sendbuf) = (int)len;
            memcpy(sendbuf+4, utf8, len);
            if (send(sock, sendbuf, 4+len, 0) == -1) {
                perror("send");
                close(sock);
                return -1;
            }
            
            long long recvbuf[4];
            if (recv(sock, recvbuf, sizeof(recvbuf), MSG_WAITALL) == -1) {
                perror("recv");
                close(sock);
                return -1;
            }
            if (recvbuf[0] == 0) {
                close(sock);
                return -1;
            }
            if (info) {
                info[0] = JLONG(filename);
                info[1] = recvbuf[1];
                info[2] = recvbuf[2];
                info[3] = recvbuf[3];
            }
            
            objc_setAssociatedObject(audioSocketHolder, filename,
                                     [NSNumber numberWithInt:sock],
                                     OBJC_ASSOCIATION_RETAIN);
            [filename retain];
        }
        return sock;
    }
}

JNIEXPORT jlongArray JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTKitSource_openAudio
    (JNIEnv *env, jobject jthis, jstring jfilename)
{
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    
    jlongArray result = NULL;
    
    jlong info[4];
    if (getAudioSocket(jstringToNSString(env, jfilename), info) != -1) {
        result = (*env)->NewLongArray(env, 4);
        (*env)->SetLongArrayRegion(env, result, 0, 4, info);
    }
    
    [pool release];
    return result;
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTKitSource_closeAudio
    (JNIEnv *env, jobject jthis, jlong filenameAddress)
{
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];

    NSString* filename = POINTER(NSString*, filenameAddress);
    
    int sock = getAudioSocket(filename, NULL);
    if (sock != -1) {
        shutdown(sock, SHUT_RDWR);
        close(sock);
    }
    
    objc_setAssociatedObject(audioSocketHolder, filename, NULL, OBJC_ASSOCIATION_RETAIN);
    [filename release];
    
    [pool release];
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTKitSource_audioChunkFromTime
    (JNIEnv *env, jobject jthis, jlong filenameAddress,
     jint channels, jint sampleRate, jint sampleSize, jboolean floatingPoint,
     jlong timeValue, jint timeScale, jobject jbuffer, jint offset, jint frameCount)
{
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    
    NSString* filename = POINTER(NSString*, filenameAddress);
    
    int sock = getAudioSocket(filename, NULL);
    if (sock != -1) {
        int buf[8] = { channels, sampleRate, sampleSize, floatingPoint };
        *((long long *)(buf+4)) = timeValue;  // buf[5],buf[6]
        buf[6] = timeScale;
        buf[7] = frameCount;
        if (send(sock, buf, sizeof(buf), 0) != -1) {
            void* buffer = (*env)->GetPrimitiveArrayCritical(env, jbuffer, NULL);
            
            int bytesPerFrame = sampleSize * channels;
            char* p = (char*)buffer + bytesPerFrame * offset;
            size_t len = bytesPerFrame * frameCount;
            
            if (recv(sock, p, len, MSG_WAITALL) == -1) {
                perror("recv");
                sock = -1;
            }
            
            (*env)->ReleasePrimitiveArrayCritical(env, jbuffer, buffer, 0);
        }
    }
    
    [pool release];
    return (sock != -1) ? noErr : -1;
}
