/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "stdafx.h"
#include "MediaFoundationSourceJNI.h"
#include "MediaFoundationSource.h"


class PrimitiveArrayCritical
{
public:
	PrimitiveArrayCritical(JNIEnv* env, jarray array) : mEnv(env), mArray(array), mBuffer(NULL) { }
	~PrimitiveArrayCritical() { Release(); }

	void* Get()
	{
		if (mBuffer == NULL) {
			mBuffer = mEnv->GetPrimitiveArrayCritical(mArray, NULL);
		}
		return mBuffer;
	}

	void Release()
	{
		if (mBuffer != NULL) {
			mEnv->ReleasePrimitiveArrayCritical(mArray, mBuffer, 0);
		}
	}

private:
	JNIEnv*	mEnv;
	jarray	mArray;
	void*	mBuffer;
};

class JStringToWSTR
{
public:
	JStringToWSTR(JNIEnv* env, jstring jstr) : mWSTR(NULL)
	{
		jsize jstrLen = env->GetStringLength(jstr);
		mWSTR = (LPWSTR) malloc(sizeof(jchar) * (jstrLen + 1));

		const jchar* jchars = env->GetStringChars(jstr, NULL);
		memcpy(mWSTR, jchars, sizeof(jchar) * jstrLen);
		mWSTR[jstrLen] = 0;

		env->ReleaseStringChars(jstr, jchars);
	}

	~JStringToWSTR()
	{
		if (mWSTR != NULL) {
			free(mWSTR);
			mWSTR = NULL;
		}
	}

	operator LPWSTR () { return mWSTR; }

private:
	LPWSTR	mWSTR;
};


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
	return JNI_VERSION_1_4;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_initDXVA2
  (JNIEnv* env, jclass cls, jlong hwnd)
{
	return VideoSource::InitDXVA2(reinterpret_cast<HWND>(hwnd));
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_openVideo
  (JNIEnv* env, jclass cls, jstring filename, jboolean sysMem, jlongArray outResult)
{
	HRESULT hr;
	VideoSource* videoSource = NULL;

	hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
	if (FAILED(hr)) {
		return hr;
	}

	hr = MFStartup(MF_VERSION, MFSTARTUP_NOSOCKET);
	if (FAILED(hr)) {
		CoUninitialize();
		return hr;
	}

	videoSource = new VideoSource(sysMem);
	if (videoSource == NULL) {
		hr = E_OUTOFMEMORY;
		goto bail;
	}

	hr = videoSource->InitWithFile(JStringToWSTR(env, filename));
	if (FAILED(hr)) {
		goto bail;
	}

	const MFRatio& frameRate = videoSource->GetFrameRate();
	jlong buf[7] = { reinterpret_cast<jlong>(videoSource),
					videoSource->GetWidth(), videoSource->GetHeight(), videoSource->IsTopDown(),
					videoSource->GetDuration(), frameRate.Numerator, frameRate.Denominator };
	env->SetLongArrayRegion(outResult, 0, 7, buf);

	hr = S_OK;

bail:
	if (FAILED(hr)) {
		delete videoSource;
		MFShutdown();
		CoUninitialize();
	}
	return hr;
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_closeVideo
  (JNIEnv* env, jclass cls, jlong videoSourceAddress)
{
	VideoSource* videoSource = reinterpret_cast<VideoSource*>(videoSourceAddress);
	delete videoSource;

	MFShutdown();
	CoUninitialize();
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_frameSurfaceAtTime
  (JNIEnv* env, jclass cls, jlong videoSourceAddress, jlongArray ioTime, jlong limitTime, jlongArray outSurface)
{
	VideoSource* videoSource = reinterpret_cast<VideoSource*>(videoSourceAddress);

	jlong time;
	env->GetLongArrayRegion(ioTime, 0, 1, &time);

	IDirect3DSurface9* surface;
	HRESULT hr = videoSource->FrameSurfaceAtTime(
								&time, limitTime, (outSurface != NULL) ? &surface : NULL);
	if (SUCCEEDED(hr)) {
		env->SetLongArrayRegion(ioTime, 0, 1, &time);
		if (outSurface != NULL) {
			env->SetLongArrayRegion(outSurface, 0, 1, reinterpret_cast<jlong*>(&surface));
		}
	}
	return hr;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_lockSurface
  (JNIEnv* env, jclass cls, jlong videoSourceAddress, jlong surfaceAddress, jobjectArray outBuffer)
{
	VideoSource* videoSource = reinterpret_cast<VideoSource*>(videoSourceAddress);
	IDirect3DSurface9* surface = reinterpret_cast<IDirect3DSurface9*>(surfaceAddress);

	D3DLOCKED_RECT lockedRect;
	HRESULT hr = videoSource->LockSurface(surface, &lockedRect);
	if (SUCCEEDED(hr)) {
		jobject buffer = env->NewDirectByteBuffer(lockedRect.pBits, lockedRect.Pitch*videoSource->GetHeight());
		env->SetObjectArrayElement(outBuffer, 0, buffer);
	}
	return hr;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_unlockSurface
  (JNIEnv* env, jclass cls, jlong videoSourceAddress, jlong surfaceAddress)
{
	VideoSource* videoSource = reinterpret_cast<VideoSource*>(videoSourceAddress);
	IDirect3DSurface9* surface = reinterpret_cast<IDirect3DSurface9*>(surfaceAddress);
	return videoSource->UnlockSurface(surface);
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_releaseSurface
  (JNIEnv* env, jclass cls, jlong videoSourceAddress, jlong surfaceAddress)
{
	VideoSource* videoSource = reinterpret_cast<VideoSource*>(videoSourceAddress);
	IDirect3DSurface9* surface = reinterpret_cast<IDirect3DSurface9*>(surfaceAddress);
	return videoSource->ReleaseSurface(surface);
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_openAudio
  (JNIEnv* env, jclass cls, jstring filename, jlongArray outResult)
{
	HRESULT hr;
	AudioSource* audioSource = NULL;

	hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
	if (FAILED(hr)) {
		return hr;
	}

	hr = MFStartup(MF_VERSION, MFSTARTUP_NOSOCKET);
	if (FAILED(hr)) {
		CoUninitialize();
		return hr;
	}

	audioSource = new AudioSource();
	if (audioSource == NULL) {
		hr = E_OUTOFMEMORY;
		goto bail;
	}

	hr = audioSource->InitWithFile(JStringToWSTR(env, filename));
	if (FAILED(hr)) {
		goto bail;
	}

	jlong buf[6] = { reinterpret_cast<jlong>(audioSource), audioSource->GetDuration(),
					audioSource->GetSampleRate(), audioSource->GetSampleSizeInBits(),
					audioSource->GetChannels(), audioSource->IsFloat() };
	env->SetLongArrayRegion(outResult, 0, 6, buf);

	hr = S_OK;

bail:
	if (FAILED(hr)) {
		delete audioSource;
		MFShutdown();
		CoUninitialize();
	}
	return hr;
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_closeAudio
  (JNIEnv* env, jclass cls, jlong audioSourceAddress)
{
	AudioSource* audioSource = reinterpret_cast<AudioSource*>(audioSourceAddress);
	delete audioSource;

	MFShutdown();
	CoUninitialize();
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_readAudioBufferAtTime
  (JNIEnv* env, jclass cls, jlong audioSourceAddress, jlongArray ioTime, jlongArray outMedbuf, jintArray outLen)
{
	AudioSource* audioSource = reinterpret_cast<AudioSource*>(audioSourceAddress);

	jlong time;
	env->GetLongArrayRegion(ioTime, 0, 1, &time);

	IMFMediaBuffer* medbuf = NULL;
	DWORD len;
	HRESULT hr = audioSource->ReadBufferAtTime(&time, &medbuf, &len);
	if (SUCCEEDED(hr)) {
		env->SetLongArrayRegion(ioTime, 0, 1, &time);
		env->SetLongArrayRegion(outMedbuf, 0, 1, reinterpret_cast<jlong*>(&medbuf));
		env->SetIntArrayRegion(outLen, 0, 1, reinterpret_cast<jint*>(&len));
	}
	return hr;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_copyAudioBuffer
  (JNIEnv* env, jclass cls, jlong medbufAddress, jint srcOff, jbyteArray dst, jint dstOff, jintArray ioLen)
{
	IMFMediaBuffer* medbuf = reinterpret_cast<IMFMediaBuffer*>(medbufAddress);

	jint dstLen;
	env->GetIntArrayRegion(ioLen, 0, 1, &dstLen);

	BYTE* srcBuf;
	DWORD srcLen;
	HRESULT hr = medbuf->Lock(&srcBuf, NULL, &srcLen);
	if (FAILED(hr)) return hr;

	jint len = min((jint)srcLen-srcOff, dstLen);

	PrimitiveArrayCritical dstBuf(env, dst);

	CopyMemory((BYTE*)dstBuf.Get()+dstOff, srcBuf+srcOff, len);

	dstBuf.Release();

	hr = medbuf->Unlock();
	if (FAILED(hr)) return hr;

	env->SetIntArrayRegion(ioLen, 0, 1, &len);
	return S_OK;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsMediaFoundationSource_releaseAudioBuffer
  (JNIEnv* env, jclass cls, jlong medbufAddress)
{
	IMFMediaBuffer* medbuf = reinterpret_cast<IMFMediaBuffer*>(medbufAddress);
	medbuf->Release();
	return S_OK;
}
