/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.BoxBlur", category=Categories.BLUR_AND_SHARPEN)
public class BoxBlur {

	@Property(min="0", max="500")
	private IAnimatableDouble blurriness;

	@Property(value="1", min="1", max="50")
	private IAnimatableInteger iterations;

	@Property
	private IAnimatableEnum<BlurDimensions> blurDimensions;

	@Property
	private IAnimatableBoolean repeatEdgePixels;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IBlurSupport blurSupport;

	@Inject
	public BoxBlur(IVideoEffectContext context, IBlurSupport blurSupport) {
		this.context = context;
		this.blurSupport = blurSupport;
	}

	public VideoBounds getVideoBounds() {
		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		double blur = context.value(blurriness);

		// blurが50より大きい場合はfastプロパティの値にかかわらず高速モード
		boolean fast = (blur > 50) || context.value(this.fast);

		// 解像度に合わせてblurの値を変換
		blur = context.getVideoResolution().scale(blur);
		if (blur == 0) {
			return bounds;
		}

		if (context.value(this.repeatEdgePixels)) {
			return bounds;
		}

		BlurDimensions dimensions = context.value(blurDimensions);
		int iterations = context.value(this.iterations);

		return blurSupport.calcBoxBlurredBounds(bounds, blur, dimensions, fast, iterations);
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();
		if (input.getBounds().isEmpty()) {
			return input;
		}

		double blur = context.value(blurriness);

		// blurが50より大きい場合はfastプロパティの値にかかわらず高速モード
		boolean fast = (blur > 50) || context.value(this.fast);

		// 解像度に合わせてblurの値を変換
		blur = context.getVideoResolution().scale(blur);
		if (blur == 0) {
			return input;
		}

		BlurDimensions dimensions = context.value(blurDimensions);
		boolean repeatEdgePixels = context.value(this.repeatEdgePixels);
		int iterations = context.value(this.iterations);

		try {
			return blurSupport.boxBlur(input, blur, dimensions, repeatEdgePixels, fast, iterations);
		} finally {
			input.dispose();
		}
	}

}
