package jp.co.powerbeans.jdbcdebug.util;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Properties;
import java.util.regex.Pattern;

import jp.co.powerbeans.jdbcdebug.notify.NotifyUtil;
import jp.co.powerbeans.jdbcdebug.sql.Driver;


/**
 * <p>タイトル: Log</p>
 * <p>説明: </p>
 * <p>Created on 2003/10/01</p>
 * @author 門田明彦
 * @version $Revision: 1.1 $
 */
public class Log {

	/** ログ 接頭辞 */
	private static String prefix;

	/** ログ 出力タイプ stdout */
	private static final int OUTTYPE_STDOUT = 1;

	/** ログ 出力タイプ file */
	private static final int OUTTYPE_FILE = 2;

	/** ログ 出力タイプ Log4J */
	private static final int OUTTYPE_LOG4J = 3;

	/** ログ 出力タイプ */
	private static int outType;

	/** カレンダーフォーマット */
	private static SimpleDateFormat df =
		new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

	// ログファイル名
	private static final String LOG_FILE = "jdbcdebug.log";

	/** Log4J */
  private static Object cat = null;
  static {
      try {
        Class.forName("org.apache.log4j.Category");
        
        // find log4j
        cat = org.apache.log4j.Category.getInstance("jdbcdebug");
    } catch (ClassNotFoundException e) {
        // can't find log4j
    }
  }

  /** SQL 正規表現マッチング文字列 */
    private static String sql_regexp;
    
    private static Pattern ptn;

    private static String CRLF = "\r\n";

	static {
//		ResourceBundle bundle =
//			ResourceBundle.getBundle(Driver.JDBC_DEBUG_PROPERTY);

		int otype = OUTTYPE_STDOUT;
//		try {
//			otype =
//				Integer.parseInt(bundle.getString(Driver.KEY_OUTPUT_TYPE));
//		} catch (RuntimeException e) {
//		}
		outType = otype;
		if (outType == OUTTYPE_LOG4J) {
		    CRLF = "";
		    df = new SimpleDateFormat("");
		}
		
		String l_prefix = "";
//		try {
//            l_prefix = bundle.getString(Driver.KEY_OUTPUT_PREFIX);
//        } catch (RuntimeException e1) {
//        }
    prefix = l_prefix;
        
    String l_sql_regexp = null;
		
//		try {
//		    l_sql_regexp = bundle.getString(Driver.KEY_SQL_CHECK_REGEXP);
//        } catch (RuntimeException e2) {
//        }
        sql_regexp = l_sql_regexp;
		
		if (sql_regexp != null && sql_regexp.length() > 0) {
		    ptn = Pattern.compile(sql_regexp, Pattern.CASE_INSENSITIVE);
		}

//		 println("start");

	}

	/**
	 * コンストラクタ
	 */
	private Log() {
		super();
	}

	/**
	 * println<BR>
	 * ログに出力
	 * @param string 出力文字
	 */
	public static void println(String string) {
		String outtext =
			prefix
				+ " "
				+ df.format(Calendar.getInstance().getTime())
				+ " "
				+ string
				+ CRLF;

		output(outtext);
	}

	/**
	 * output<BR>
	 * @param outtext
	 */
	private static void output(String outtext) {

		switch (outType) {
			case OUTTYPE_FILE :
				outputFile(outtext);
				break;
			case OUTTYPE_LOG4J :
			  outputLog4j(outtext);
			  break;
			case OUTTYPE_STDOUT :
			default :
				outputStdout(outtext);
				break;
		}
	}

	/**
	 * output by log4j
     * @param outtext
     * @author  amonden
     */
    private static void outputLog4j(String outtext) {
        if (cat != null) {
            org.apache.log4j.Category cati = (org.apache.log4j.Category)cat;
            cati.debug(outtext);
        }
    }

    /**
	 * outputStdout<BR>
	 * @param outtext
	 */
	private static void outputStdout(String outtext) {
		System.out.print(outtext);
	}

	/**
	 * outputFile<BR>
	 * @param outtext
	 */
	private static void outputFile(String outtext) {

		FileWriter fr = null;
		try {
			fr = new FileWriter(new File(LOG_FILE), true);
			fr.write(outtext);
			fr.close();
		} catch (IOException e) {
		} finally {
		}
	}

	/**
	 * println<BR>
   * sqlと実行時間 をログに出力
	 * @param sql sql
	 * @param survey 実行時間
	 */
	public static void printlnSqlLog(String sql, Survey survey) {

		String outtext =
			prefix
				+ " "
				+ df.format(Calendar.getInstance().getTime())
				+ " "
				+ (survey.getEndMs() - survey.getStartMs())
				+ "ms "
				+ sql
				+ CRLF;

		output(outtext);
		
		// sql_regexp が設定されていれば正規表現マッチングを行う
		// 一致すればトレース出力
		if (sql_regexp != null && sql_regexp.length() > 0 &&
		    sql != null && sql.length() > 0) {
		    sqlRegexpMatch(sql);
		}
		
		// 解析ログに出力
		NotifyUtil.onExecuteSql(Driver.getDriverInstance(), sql, survey);
	}


    /**
     * SQL正規表現マッチング
     * @param sql
     * @author  amonden
     */
    private static void sqlRegexpMatch(String sql) {
        if (ptn != null && ptn.matcher(sql).matches()) {
            // 一致したのでトレース出力
        		StackTraceElement[] el = new Exception().getStackTrace();
        		String trace = 
        		    prefix + "[SQL Regexp match] " +
        		    ConnectionDtl.formatStackTrace(el);
        		output(trace);
        }
    }

    /**
	 * println<BR>
	 * SQLException をログに出力
	 * @param string
	 * @param e
	 */
	public static void printlnSqlExceptionLog(String string, SQLException e) {
		
		String outtext =
			prefix
				+ " "
				+ df.format(Calendar.getInstance().getTime())
				+ " "
				+ string
				+ " [SQLException] "
				+ e.getMessage() + "(" + e.getErrorCode() + ")"
				+ CRLF;

		output(outtext);
		
		
		// 解析ログに出力
		NotifyUtil.onExecuteSqlException(Driver.getDriverInstance(), string, e);
	}

  /**
   * println<BR>
   * テキストと実行時間とパラメータをログに出力。
   * string 内の ? を パラメータに置換し出力する
   * @param string
   * @param survey
   * @param paramMap
   */
  public static void printlnSqlLog(String string, Survey survey, HashMap paramMap) {
    
    String wstr = string;
    int q = 0;
    int paramIndex = 1;
    while ((q = wstr.indexOf('?', q)) != -1) {
      
      // 値を取得
      String paramIndexS = Integer.toString(paramIndex);
      if (paramMap.containsKey(paramIndexS)) {
        // 値がある場合は置換
        wstr = wstr.substring(0, q) + 
            encodeSqlField(paramMap.get(paramIndexS)) +
            wstr.substring(q + 1);
      }
      
      paramIndex++;
    }
    
    printlnSqlLog(wstr, survey);
  }

  /**
   * encodeSqlField<BR>
   * @param object
   * @return
   */
  private static String encodeSqlField(Object object) {

		if (object == null) {
			return null;
		}   
    else if (object instanceof String ||
            object instanceof java.util.Date) {
      return "'" + object + "'";
    } else {
      return object.toString();
    }
  }

	/**
	 * URL プロパティをLogの設定に反映
	 * @param prop
	 */
	public static void applyProperties(Properties prop) {

		// 1. 出力タイプ
		String otype = prop.getProperty(Driver.KEY_OUTPUT_TYPE);
		if (otype != null) {
			try {
				outType = Integer.parseInt(otype);
			} catch (NumberFormatException e) {
				e.printStackTrace();
			}
		}

		if (outType == OUTTYPE_LOG4J) {
			CRLF = "";
			df = new SimpleDateFormat("");
		}

		// 2. ログ接頭辞
		String l_prefix = prop.getProperty(Driver.KEY_OUTPUT_PREFIX);
		if (l_prefix != null) {
			prefix = l_prefix;
		}
	}

}
