/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/core/EditorConfiguration.java,v 1.1 2005/01/29 03:24:19 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2005/01/29 03:24:19 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.core;

import jp.sourceforge.jeextension.styleeditor.entry.IPartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntryList;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextDoubleClickStrategy;
import org.eclipse.jface.text.presentation.IPresentationReconciler;
import org.eclipse.jface.text.presentation.PresentationReconciler;
import org.eclipse.jface.text.rules.DefaultDamagerRepairer;
import org.eclipse.jface.text.rules.RuleBasedScanner;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;

/**
 * Configuration is defined editor configuration object.
 */
class EditorConfiguration extends SourceViewerConfiguration {

    /** ColorManager object. */
    private ColorManager fColorManager;

    /** DoubleClickStrategy object. */
    private DoubleClickStrategy fDoubleClickStrategy;

    /** Define style tools. */
    private EditorTools fEditorTools;

    /** Current StyleEditor. */
    private StyleEditor fStyleEditor;

    /**
     * Iniciate Configuration object.
     * @param styleEditor   Current Editor.
     */
    EditorConfiguration(StyleEditor styleEditor) {
        this.fColorManager = new ColorManager();
        this.fStyleEditor = styleEditor;
    }

    /**
     * Return ColorManager.
     * @return ColorManager
     */
    public ColorManager getColorManager() {
        return this.fColorManager;
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getConfiguredContentTypes(org.eclipse.jface.text.source.ISourceViewer)
     */
    public String[] getConfiguredContentTypes(ISourceViewer sourceViewer) {
        // get current editting file extension.
        return getTool().getContentTypes(fStyleEditor.getFileExtension());
    }

    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getDoubleClickStrategy(org.eclipse.jface.text.source.ISourceViewer, java.lang.String)
     */
    public ITextDoubleClickStrategy getDoubleClickStrategy(
            ISourceViewer sourceViewer, String contentType) {
        if (fDoubleClickStrategy == null) {
            fDoubleClickStrategy = new DoubleClickStrategy();
        }
        return fDoubleClickStrategy;
    }

    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.source.SourceViewerConfiguration#getPresentationReconciler(org.eclipse.jface.text.source.ISourceViewer)
     */
    public IPresentationReconciler getPresentationReconciler(
            ISourceViewer sourceViewer) {
        PresentationReconciler reconciler = new PresentationReconciler();
        
        // get current editting file extension.
        String ext = fStyleEditor.getFileExtension();
        
        // for default content type.
        EditorTools tools = getTool();
        {
            RuleBasedScanner scanner =
                    tools.createScanner(IDocument.DEFAULT_CONTENT_TYPE, ext);
            scanner.setDefaultReturnToken(tools.getDefaultToken());
            DefaultDamagerRepairer dr = new DefaultDamagerRepairer(scanner);
            reconciler.setDamager (dr, IDocument.DEFAULT_CONTENT_TYPE);
            reconciler.setRepairer(dr, IDocument.DEFAULT_CONTENT_TYPE);
        }
        {
            PartitionEntryList list =
                    EditorTools.getPartitionEntryList(ext, false);
            for (int i = 0; i < list.size(); i++) {
                IPartitionEntry pe = list.getPartitionEntry(i);
                String partName = pe.getName();
                
                // create scanner.
                RuleBasedScanner scanner = tools.createScanner(partName, ext);
                scanner.setDefaultReturnToken(tools.getDefaultToken(pe));
                
                // create damager repairer and set one to.
                DefaultDamagerRepairer dr =
                        new DefaultDamagerRepairer(scanner);
                reconciler.setDamager (dr, partName);
                reconciler.setRepairer(dr, partName);
            }
        }
        
        return reconciler;
    }
    
    /**
     * Return DefinedStyleTools. if not initiated, create one instance.
     * @return DefinedStyleTools instance.
     */
    private EditorTools getTool() {
        if (this.fEditorTools == null) {
            this.fEditorTools = new EditorTools(this);
        }
        return this.fEditorTools;
    }
}