/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/preferences/StyleEditorPreferencePage.java,v 1.2 2005/01/30 06:02:40 kohnosuke Exp $
 * $Revision: 1.2 $
 * $Date: 2005/01/30 06:02:40 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.preferences;

import java.util.Iterator;

import jp.sourceforge.jeextension.common.plugin.DefaultPreferencePage;
import jp.sourceforge.jeextension.common.xml.XMLFile;
import jp.sourceforge.jeextension.styleeditor.StyleEditorPlugin;
import jp.sourceforge.jeextension.styleeditor.core.ColorManager;
import jp.sourceforge.jeextension.styleeditor.core.EditorTools;
import jp.sourceforge.jeextension.styleeditor.entry.DefinedStyleEntry;
import jp.sourceforge.jeextension.styleeditor.entry.DependsEntryList;
import jp.sourceforge.jeextension.styleeditor.entry.IDependsEntry;
import jp.sourceforge.jeextension.styleeditor.entry.IPartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntryList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbench;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;

/**
 * NextPreferencePage
 */
public class StyleEditorPreferencePage extends DefaultPreferencePage {
    
    /** Button for Add. */
    private Button fButtonAdd;
    
    /** Button for Remove. */
    private Button fButtonDel;
    
    /** Button for Down. */
    private Button fButtonDn;
    
    /** Button for Export. */
    private Button fButtonExport;
    
    /** Button for Import. */
    private Button fButtonImport;
    
    /** Button for Modify. */
    private Button fButtonMod;
    
    /** Button for Add Extension. */
    private Button fButtonExt;
    
    /** Button for Add Partition. */
    private Button fButtonPrt;
    
    /** Button for Up. */
    private Button fButtonUp;
    
    /** ColorManager for this Dialog. */
    private ColorManager fColorManager = new ColorManager();
    
    /** Combobox for partition entries */
    private Combo fComboPrt;
    
    /** Combobox for extension entries */
    private Combo fComboExt;
    
    /** Table for Entry view. */
    private Table fTable;
    
    /**
     * 
     */
    public StyleEditorPreferencePage() {
        super();
        setTitle(getMessage("pref.title"));
        setDescription(getMessage("pref.description"));
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.preference.PreferencePage#createContents(org.eclipse.swt.widgets.Composite)
     */
    public Control createContents(Composite parent) {

        Composite container = new Composite(parent, SWT.NULL);
        {
            final GridLayout gridLayout = new GridLayout();
            gridLayout.numColumns = 3;
            container.setLayout(gridLayout);
        }
        // Area 1.
        {
            final Label label = new Label(container, SWT.NONE);
            label.setText(getMessage("pref.main.extension.label"));
            
            final Combo combo = new Combo(container, SWT.READ_ONLY);
            combo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
            combo.addListener(SWT.Selection, this);
            fComboExt = combo;
            
            final Button button = new Button(container, SWT.NONE);
            button.setText(getMessage("pref.main.extension.edit"));
            button.addListener(SWT.Selection, this);
            fButtonExt = button;
        }
        // Area 2.
        {
            final Label label = new Label(container, SWT.NONE);
            label.setText(getMessage("pref.main.partition.label"));
            
            final Combo combo = new Combo(container, SWT.READ_ONLY);
            combo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
            combo.addListener(SWT.Selection, this);
            fComboPrt = combo;
            
            final Button button = new Button(container, SWT.NONE);
            button.setText(getMessage("pref.main.partition.edit"));
            button.addListener(SWT.Selection, this);
            fButtonPrt = button;
        }
        
        // Area 3.
        final Composite composite = new Composite(container, SWT.NONE);
        {
            final GridData gridData = new GridData(GridData.FILL_BOTH);
            gridData.horizontalSpan = 3;
            composite.setLayoutData(gridData);
            final GridLayout gridLayout = new GridLayout();
            gridLayout.numColumns = 2;
            composite.setLayout(gridLayout);
        }
        
        final Table table =
                new Table(composite, SWT.FULL_SELECTION | SWT.BORDER);
        {
            final GridData gridData = new GridData(GridData.FILL_BOTH);
            gridData.verticalSpan = 5;
            table.setLayoutData(gridData);
            table.setLinesVisible(true);
            table.setHeaderVisible(true);
            createTableColumn(table, getMessage("pref.com.colmn.type"));
            createTableColumn(table, getMessage("pref.com.colmn.rule"));
        }
        fTable = table;
        
        fButtonAdd = createButtonA(composite, getMessage("pref.com.btn.add"));
        fButtonMod = createButtonA(composite, getMessage("pref.com.btn.mod"));
        fButtonDel = createButtonA(composite, getMessage("pref.com.btn.del"));
        fButtonUp  = createButtonA(composite, getMessage("pref.com.btn.up"));
        fButtonDn  = createButtonA(composite, getMessage("pref.com.btn.dn"));
        
        final Composite subComposite = new Composite(composite, SWT.NONE);
        {
            final GridData gridData = new GridData(
                GridData.HORIZONTAL_ALIGN_END | GridData.VERTICAL_ALIGN_FILL);
            gridData.horizontalSpan = 2;
            subComposite.setLayoutData(gridData);
            final GridLayout gridLayout = new GridLayout();
            gridLayout.numColumns = 2;
            subComposite.setLayout(gridLayout);
        }
        
        fButtonImport = createButtonB(subComposite, getMessage("pref.import"));
        fButtonExport = createButtonB(subComposite, getMessage("pref.export"));
        //
        return container;
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
     */
    public void dispose() {
        fColorManager.dispose();
        super.dispose();
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
     */
    public void handleEvent(Event event) {
        if (event.widget == fComboExt) {
            refleshComboPrt();
            refleshTableEntry();
        }
        if (event.widget == fComboPrt) {
            refleshTableEntry();
        }
        if (event.widget == fButtonPrt) {
            showDisplayPartDlg();
        }
        if (event.widget == fButtonExt) {
            showDisplayExtDlg();
        }
        if (event.widget == fButtonAdd) {
            fireAddEntry();
        }
        if (event.widget == fButtonMod) {
            fireModifyEntry();
        }
        if (event.widget == fButtonDel) {
            fireDeleteEntry();
        }
        if (event.widget == fButtonUp) {
            fireMoveUpEntry();
        }
        if (event.widget == fButtonDn) {
            fireMoveDownEntry();
        }
        if (event.widget == fButtonImport) {
            fireImport();
        }
        if (event.widget == fButtonExport) {
            fireExport();
        }
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.ui.IWorkbenchPreferencePage#init(org.eclipse.ui.IWorkbench)
     */
    public void init(IWorkbench workbench) {
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.preference.IPreferencePage#performOk()
     */
    public boolean performOk() {
        StyleEditorPlugin.save();
        return super.performOk();
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.dialogs.IDialogPage#setVisible(boolean)
     */
    public void setVisible(boolean visible) {
        super.setVisible(visible);
        refleshComboEntry();
        refleshTableEntry();
    }
    
    /**
     * Add Objects into target Table.
     * @param   entry   IEntry instance.
     */
    private void addTableItem(IDependsEntry entry) {
        /* add table item. */
        TableItem item =
                new TableItem(fTable, SWT.NONE);
        item.setText(0, entry.getName());
        if (entry.getName().equals("endOfLine")) {
            item.setText(1, entry.getRule().getWords()[0]);
        }
        if (entry.getName().equals("singleLine")) {
            item.setText(1,
                    entry.getRule().getWords()[0] + " "
                    + entry.getRule().getWords()[1]);
        }
        if (entry.getName().equals("multiLine")) {
            item.setText(1,
                    entry.getRule().getWords()[0] + " "
                    + entry.getRule().getWords()[1]);
        }
        if (entry.getName().equals("keyword")) {
            item.setText(1, entry.getRule().getWords()[0]);
        }
        item.setData(entry);
        item.setForeground(fColorManager.getColor(entry.getRGB()));
    }
    
    /**
     * Create Button type A.
     * @param composite parent composite.
     * @param str       Button's String.
     * @return created Button type A.
     */
    private Button createButtonA(Composite composite, String str) {
        final Button button = new Button(composite, SWT.NONE);
        button.setLayoutData(new GridData(
            GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_BEGINNING)
        );
        button.setText(str);
        button.addListener(SWT.Selection, this);
        return button;
    }
    
    /**
     * Create Button type B.
     * @param composite parent composite.
     * @param str       Button's String.
     * @return created Button type B.
     */
    private Button createButtonB(Composite composite, String str) {
        final Button button = new Button(composite, SWT.NONE);
        button.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
        button.setText(str);
        button.addListener(SWT.Selection, this);
        return button;
    }
    
    /**
     * Create TableColumn.
     * @param table parent composite (Table).
     * @param str column's string.
     * @return created TableColumn.
     */
    private TableColumn createTableColumn(Table table, String str) {
        final TableColumn tableColumn = new TableColumn(table, SWT.NONE);
        tableColumn.setWidth(100);
        tableColumn.setText(str);
        return tableColumn;
    }
    
    /**
     * Add new entry.
     */
    private void fireAddEntry() {
        IDependsEntry[] container = new IDependsEntry[1];
        String partName = fComboPrt.getItem(fComboPrt.getSelectionIndex());
        String extName  = fComboExt.getItem(fComboExt.getSelectionIndex());
        AddDependsDialog d =
                new AddDependsDialog(getShell(), container, partName, extName);
        if (d.open() == IDialogConstants.OK_ID) {
            // add table item.
            addTableItem(container[0]);
                
            // add to preference store.
            DefinedStyleEntry styleEntry =
                    StyleEditorPlugin.getDefinedStyleEntry();
            styleEntry.add(container[0], extName);
        }
    }
        
    /**
     * Remove selection entry.
     */
    private void fireDeleteEntry() {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        TableItem item = fTable.getItem(sel);
        IDependsEntry entry = (IDependsEntry) item.getData();
        
        // remove from table. and remove from conf list.
        fTable.remove(sel);
        String extName  = fComboExt.getItem(fComboExt.getSelectionIndex());
        StyleEditorPlugin.getDefinedStyleEntry().remove(entry, extName);
    }
        
    /**
     * Export DefinedEditro Configurations to XML file.
     */
    private void fireExport() {
        
        FileDialog d = new FileDialog(getShell(), SWT.SAVE);
        d.setText(getMessage("pref.export.message"));
        d.setFilterExtensions(new String[] {
            getMessage("pref.export.extension"),
        });
        d.setFileName(getMessage("pref.export.filename"));
        String path = d.open();
        
        if (path == null) {
            return;
        }
        StyleEditorPlugin.save(new XMLFile(path));
            
        refleshTableEntry();
    }
        
    /**
     * Import DefinedEditro Configurations from XML file.
     */
    private void fireImport() {
            
        FileDialog d = new FileDialog(getShell());
        d.setText(getMessage("pref.import.message"));
        d.setFilterExtensions(new String[] {
            getMessage("pref.import.extension"),
        });
        String path = d.open();
        if (path == null) {
            return;
        }
        StyleEditorPlugin.load(new XMLFile(path));
            
        refleshTableEntry();
    }
    
    /**
     * Modify selection entry.
     */
    private void fireModifyEntry() {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        TableItem item = fTable.getItem(sel);
        IDependsEntry entry = (IDependsEntry) item.getData();
        IDependsEntry[] container = new IDependsEntry[1];
        container[0] = entry;
        String partName = fComboPrt.getItem(fComboPrt.getSelectionIndex());
        String extName  = fComboExt.getItem(fComboExt.getSelectionIndex());
        AddDependsDialog d =
                new AddDependsDialog(getShell(), container, partName, extName);
        if (d.open() == IDialogConstants.OK_ID) {
            item.setText(0, entry.getName());
            if (entry.getName().equals("endOfLine")) {
                item.setText(1, entry.getRule().getWords()[0]);
            }
            if (entry.getName().equals("singleLine")) {
                item.setText(1,
                        entry.getRule().getWords()[0] + " "
                        + entry.getRule().getWords()[1]);
            }
            if (entry.getName().equals("keyword")) {
                item.setText(1, entry.getRule().getWords()[0]);
            }
            item.setData(entry);
            item.setForeground(fColorManager.getColor(entry.getRGB()));
        }
    }
        
    /**
     * Down Selected row of target Table entry.
     */
    private void fireMoveDownEntry() {
        fireMoveEntry(1);
    }
        
    /**
     * Up Selected row of target Table entry.
     */
    private void fireMoveUpEntry() {
        fireMoveEntry(-1);
    }
        
    /**
     * Up / Down Selected row of target Table entry.
     * @param increment Up=-1 / Down=1
     */
    private void fireMoveEntry(int increment) {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        Object obj = fTable.getSelection()[0].getData();
        
        // get current partition.
        String partName = fComboPrt.getItem(fComboPrt.getSelectionIndex());
        String extName  = fComboExt.getItem(fComboExt.getSelectionIndex());
        
        IPartitionEntry partEntry =
                EditorTools.getPartitionEntry(partName, extName, false);
        DependsEntryList list = partEntry.getDependsEntryList();
        
        int index = list.indexOf(obj);
        Object removeObj = list.remove(index);
        list.add(index + increment, removeObj);
            
        refleshTableEntry();
        fTable.setSelection(sel + increment);
        refleshButtonEnable();
    }
    
    /**
     * Change button press enabled when table entry change.
     */
    private void refleshButtonEnable() {
        int sel = fTable.getSelectionIndex();
        if (fTable.getItemCount() == 1) {
            fButtonUp.setEnabled(false);
            fButtonDn.setEnabled(false);
        }
        else
        if (sel == 0) {
            fButtonUp.setEnabled(false);
            fButtonDn.setEnabled(true);
        }
        else
        if (sel == fTable.getItemCount() - 1) {
            fButtonUp.setEnabled(true);
            fButtonDn.setEnabled(false);
        }
        else {
            fButtonUp.setEnabled(true);
            fButtonDn.setEnabled(true);
        }
    }
        
    /**
     * Reflesh ComboBox Entry from partition management list.
     */
    private void refleshComboEntry() {
        
        // reflesh Extension entry.
        refleshComboExt();
        // reflesh Partition entry.
        refleshComboPrt();
    }
    
    /**
     * reflesh Extension entry.
     */
    private void refleshComboExt() {
        fComboExt.removeAll();
        
        Iterator iterator =
                EditorTools.getPartitionEntryMap().keySet().iterator();
        while (iterator.hasNext()) {
            String extName = (String) iterator.next();
            fComboExt.add(extName);
        }
        fComboExt.select(0);
    }
    
    /**
     * reflesh Partition entry.
     */
    private void refleshComboPrt() {
        fComboPrt.removeAll();
        
        String extName  = fComboExt.getItem(fComboExt.getSelectionIndex());
        PartitionEntryList list;
        list = EditorTools.getPartitionEntryList(extName, false);
        for (int i = 0; i < list.size(); i++) {
            IPartitionEntry entry = list.getPartitionEntry(i);
            fComboPrt.add(entry.getName());
        }
        fComboPrt.select(0);
    }
    
    /**
     * Reflesh table entries.
     */
    private void refleshTableEntry() {
        // clear all table content.
        fTable.removeAll();
            
        // get current partition.
        if (fComboPrt.getItems().length == 0) {
            return;
        }
        String partName = fComboPrt.getItem(fComboPrt.getSelectionIndex());
        String extName  = fComboExt.getItem(fComboExt.getSelectionIndex());
        
        // add entry depends current partition to table.
        IPartitionEntry partEntry =
                EditorTools.getPartitionEntry(partName, extName, false);
        DependsEntryList entryList = partEntry.getDependsEntryList();
                        
        for (int i = 0; i < entryList.size(); i++) {
            IDependsEntry entry = entryList.getDependsEntry(i);
            addTableItem(entry);
        }
    }
    
    /**
     * Open dialog for partition entries.
     */
    private void showDisplayPartDlg() {
        String extName  = fComboExt.getItem(fComboExt.getSelectionIndex());
        PartitionEntryDialog d =
                new PartitionEntryDialog(getShell(), extName);
        d.open();
        refleshComboPrt();
        refleshTableEntry();
    }
    
    /**
     * Open dialog for partition entries.
     */
    private void showDisplayExtDlg() {
        ExtensionEntryDialog d = new ExtensionEntryDialog(getShell());
        d.open();
        refleshComboEntry();
        refleshTableEntry();
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.preference.PreferencePage#performDefaults()
     */
    protected void performDefaults() {
        StyleEditorPlugin.loadDefault();
        refleshComboEntry();
        refleshTableEntry();
    }
}
