/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/core/StyleEditorColorManager.java,v 1.1 2007/09/03 08:59:39 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2007/09/03 08:59:39 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.core;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.jdt.ui.text.IColorManager;
import org.eclipse.jdt.ui.text.IColorManagerExtension;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Display;

/**
 * ColorManager is managing Color and RGB Objects.
 * @since 1.0.0
 */
public class StyleEditorColorManager
        implements IColorManager, IColorManagerExtension {
    
    /**
     * RGB Object managing table.
     */
    protected Map fKeyTable = new HashMap(10);
    
    /**
     * Color Object managing table.
     */
    protected Map fDisplayTable = new HashMap(2);

    /**
     * Initiate ColorManager.
     */
    StyleEditorColorManager() {
        super();
    }
    
    /**
     * dispose Color for current Display.
     * @param display Display
     */
    private void dispose(Display display) {
        Map colorTable = (Map) fDisplayTable.get(display);
        if (colorTable != null) {
            Iterator e = colorTable.values().iterator();
            while (e.hasNext()) {
                Color color = (Color) e.next();
                if (color != null && !color.isDisposed()) {
                    color.dispose();
                }
            }
        }
    }

    /**
     * dispose Color objects. and Clear managing tables.
     */
    public void dispose() {
    }

    /**
     * Return Color object form RGB.
     * @param rgb org.eclipse.swt.graphics.RGB
     * @return org.eclipse.swt.graphics.Color
     */
    public Color getColor(RGB rgb) {
        
        if (rgb == null) {
            return null;
        }
        
        final Display display = Display.getCurrent();
        Map colorTable = (Map) fDisplayTable.get(display);
        if (colorTable == null) {
            colorTable = new HashMap(10);
            fDisplayTable.put(display, colorTable);
            display.disposeExec(new Runnable() {
                public void run() {
                    dispose(display);
                }
            });
        }
        
        Color color = (Color) colorTable.get(rgb);
        if (color == null) {
            color = new Color(Display.getCurrent(), rgb);
            colorTable.put(rgb, color);
        }
        
        return color;
    }
    
    /**
     * {@inheritDoc}
     */
    public Color getColor(String key) {
        
        if (key == null) {
            return null;
        }
        
        RGB rgb = (RGB) fKeyTable.get(key);
        return getColor(rgb);
    }
    
    /**
     * {@inheritDoc}
     */
    public void bindColor(String key, RGB rgb) {
        Object value = fKeyTable.get(key);
        if (value != null) {
            throw new UnsupportedOperationException();
        }
        fKeyTable.put(key, rgb);
    }
    
    /**
     * {@inheritDoc}
     */
    public void unbindColor(String key) {
        fKeyTable.remove(key);
    }
}
