/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/entry/DependsEntry.java,v 1.6 2007/09/03 08:59:39 kohnosuke Exp $
 * $Revision: 1.6 $
 * $Date: 2007/09/03 08:59:39 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.entry;

import jp.sourceforge.jeextension.common.xml.XMLNode;
import jp.sourceforge.jeextension.styleeditor.core.StyleEditorTools;
import jp.sourceforge.jeextension.styleeditor.text.IExtRule;

import org.eclipse.swt.graphics.RGB;

/**
 * DependsEntry
 * @since 1.0.0
 * <!-- #IntID:00004 Style Maximization -->
 * <!-- #ReqID:10017 RegExp -->
 */
public class DependsEntry implements IDependsEntry {
    
    /** entry type name */
    private String fName = "";
    
    /** depends PartitionEntry. */
    private IPartitionEntry fPartitionEntry;
    
    /** defined RGB color. */
    private RGB fRGB = new RGB(0, 0, 0);
    
    /** defined RGB color. */
    private RGB fBgRGB = new RGB(255, 255, 255);
    
    /** defined enable BackgroundColor */
    private boolean fBgColor = false;
    
    /** defined style */
    private boolean fItalic = false;
    
    /** defined style */
    private boolean fBold = false;
    
    /** defined style */
    private boolean fStrikeThrough = false;
    
    /** defined style */
    private boolean fUnderline = false;
    
    /** defined rule object. */
    private IExtRule fRule;
    
    /**
     * Initiate this object. for default constructor.
     */
    public DependsEntry() {
        super();
    }
    
    /**
     * Initiate this object. using XMLNode and IPartitionEntry.
     * @param node  XMLNode
     * @param ext   extension name. (ex. java, means *.java)
     */
    public DependsEntry(XMLNode node, String ext) {
        super();
        this.fPartitionEntry = StyleEditorTools.getPartitionEntry(
                node.getParentNode().getAttribute("name"), ext, true);
        this.fName = node.getNodeName();
        this.fRGB.red   = Integer.parseInt(node.getAttribute("red"));
        this.fRGB.green = Integer.parseInt(node.getAttribute("green"));
        this.fRGB.blue  = Integer.parseInt(node.getAttribute("blue"));
        this.fBgColor = Boolean.valueOf(
                node.getAttribute("bgcolor")).booleanValue();
        if (fBgColor) {
            this.fBgRGB.red     = Integer.parseInt(
                    node.getAttribute("bred"));
            this.fBgRGB.green   = Integer.parseInt(
                    node.getAttribute("bgreen"));
            this.fBgRGB.blue    = Integer.parseInt(
                    node.getAttribute("bblue"));
        }
        this.fItalic = Boolean.valueOf(
                node.getAttribute("italic")).booleanValue();
        this.fBold = Boolean.valueOf(
                node.getAttribute("bold")).booleanValue();
        this.fStrikeThrough = Boolean.valueOf(
                node.getAttribute("strikethrough")).booleanValue();
        this.fUnderline = Boolean.valueOf(
                node.getAttribute("underline")).booleanValue();
        
        this.fRule = StyleEditorTools.createExtRule(node, ext);
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IDependsEntry#createNode()
     */
    public XMLNode createNode() {
        XMLNode xmlNode = new XMLNode(this.fName);
        xmlNode.setAttribute("red"  , String.valueOf(this.fRGB.red));
        xmlNode.setAttribute("green", String.valueOf(this.fRGB.green));
        xmlNode.setAttribute("blue" , String.valueOf(this.fRGB.blue));
        if (fBgColor) {
            xmlNode.setAttribute("bgcolor", "true");
            xmlNode.setAttribute("bred"  , String.valueOf(this.fBgRGB.red));
            xmlNode.setAttribute("bgreen", String.valueOf(this.fBgRGB.green));
            xmlNode.setAttribute("bblue" , String.valueOf(this.fBgRGB.blue));
        }
        else {
            xmlNode.setAttribute("bgcolor", "false");
            xmlNode.setAttribute("bred"  , "");
            xmlNode.setAttribute("bgreen", "");
            xmlNode.setAttribute("bblue" , "");
        }
        xmlNode.setAttribute("italic",        String.valueOf(fItalic));
        xmlNode.setAttribute("bold",          String.valueOf(fBold));
        xmlNode.setAttribute("strikethrough", String.valueOf(fStrikeThrough));
        xmlNode.setAttribute("underline",     String.valueOf(fUnderline));
        
        if (this.fName.equals("multiLine")) {
            xmlNode.setAttribute("start", fRule.getWords()[0]);
            xmlNode.setAttribute("end"  , fRule.getWords()[1]);
        }
        if (this.fName.equals("singleLine")) {
            xmlNode.setAttribute("start", fRule.getWords()[0]);
            xmlNode.setAttribute("end"  , fRule.getWords()[1]);
        }
        if (this.fName.equals("endOfLine")) {
            xmlNode.setAttribute("start", fRule.getWords()[0]);
        }
        if (this.fName.equals("keyword")) {
            xmlNode.setAttribute("word", fRule.getWords()[0]);
            xmlNode.setAttribute("case", fRule.getWords()[1]);
        }
        if (this.fName.equals("regexp")) {
            xmlNode.setAttribute("regexp", fRule.getWords()[0]);
        }
        return xmlNode;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IDependsEntry#getDepends()
     */
    public IPartitionEntry getDepends() {
        return this.fPartitionEntry;
    }

    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IEntry#getName()
     */
    public String getName() {
        return this.fName;
    }

    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IEntry#getRGB()
     */
    public RGB getRGB() {
        return this.fRGB;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IEntry#getRule()
     */
    public IExtRule getRule() {
        return this.fRule;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IDependsEntry#getWords()
     */
    public String[] getWords() {
        return this.fRule.getWords();
    }

    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IDependsEntry#setDepends(jp.sourceforge.jeextension.editor.entry.IPartitionEntry)
     */
    public void setDepends(IPartitionEntry depends) {
        this.fPartitionEntry = depends; 
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IEntry#setRGB(org.eclipse.swt.graphics.RGB)
     */
    public void setRGB(RGB pRGB) {
        this.fRGB = pRGB;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#setRule(jp.sourceforge.jeextension.styleeditor.text.IExtRule)
     */
    public void setRule(IExtRule rule) {
        fRule = rule;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.entry.IDependsEntry#setWords(java.lang.String[])
     */
    public void setWords(String[] words) {
        this.fRule.setWords(words);
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#getBgRGB()
     */
    public RGB getBgRGB() {
        return fBgRGB;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#isBgColor()
     */
    public boolean isBgColor() {
        return fBgColor;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#isBold()
     */
    public boolean isBold() {
        return fBold;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#isItalic()
     */
    public boolean isItalic() {
        return fItalic;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#isStrikeThrough()
     */
    public boolean isStrikeThrough() {
        return fStrikeThrough;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#isUnderline()
     */
    public boolean isUnderline() {
        return fUnderline;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#setBgColor(boolean)
     */
    public void setBgColor(boolean b) {
        fBgColor = b;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#setBgRGB(org.eclipse.swt.graphics.RGB)
     */
    public void setBgRGB(RGB rgb) {
        fBgRGB = rgb;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#setBold(boolean)
     */
    public void setBold(boolean b) {
        fBold = b;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#setItalic(boolean)
     */
    public void setItalic(boolean b) {
        fItalic = b;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#setStrikeThrough(boolean)
     */
    public void setStrikeThrough(boolean b) {
        fStrikeThrough = b;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.styleeditor.entry.IEntry#setUnderline(boolean)
     */
    public void setUnderline(boolean b) {
        fUnderline = b;
    }
}
