/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/StyleEditorPlugin.java,v 1.10 2007/09/22 04:04:24 kohnosuke Exp $
 * $Revision: 1.10 $
 * $Date: 2007/09/22 04:04:24 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.ui.text.IColorManager;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import jp.sourceforge.jeextension.common.xml.XMLException;
import jp.sourceforge.jeextension.common.xml.XMLFile;
import jp.sourceforge.jeextension.common.xml.XMLNode;
import jp.sourceforge.jeextension.common.xml.XMLNodeList;
import jp.sourceforge.jeextension.common.xml.XMLReader;
import jp.sourceforge.jeextension.styleeditor.core.StyleEditorCore;
import jp.sourceforge.jeextension.styleeditor.core.StyleEditorTools;
import jp.sourceforge.jeextension.styleeditor.entry.DefinedStyleEntry;
import jp.sourceforge.jeextension.styleeditor.entry.DependsEntry;
import jp.sourceforge.jeextension.styleeditor.entry.DependsEntryList;
import jp.sourceforge.jeextension.styleeditor.entry.FontEntry;
import jp.sourceforge.jeextension.styleeditor.entry.IDependsEntry;
import jp.sourceforge.jeextension.styleeditor.entry.IPartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntryList;

/**
 * StyleEditorPlugin.
 * @since 1.0.0
 * <!-- #IntId:000000 Eclipse 3.0 Conversion. -->
 */
public class StyleEditorPlugin extends AbstractUIPlugin {

    /** The shared instance. */
    private static StyleEditorPlugin fPlugin;

    /** Configuration Entry List. */
    private static DefinedStyleEntry styleEntry = new DefinedStyleEntry();

    /**
     * Returns the plugin's configuration list.
     * @return configuration entry list
     */
    public static DefinedStyleEntry getDefinedStyleEntry() {
        return styleEntry;
    }

    /**
     * Load the plugin's configuration form xml file.
     */
    public static void load() {
        
        IPath path = getDefault().getStateLocation();
        path = path.append("configuration.xml");
        XMLFile file = new XMLFile(path.toFile(), "ISO-8859-1");
        
        load(file);
    }

    /**
     * Load the plugin's configuration form xml file.
     * @param file XMLFile object.
     */
    public static void load(XMLFile file) {
        styleEntry.clearAll();
        
        if (file.getFile().exists() == false) {
            loadDefault();
            return;
        }
        XMLNode root;
        try {
            root = file.readNode();
        }
        catch (XMLException e) {
            log(e);
            return;
        }
        internalLoad(root);
    }

    /**
     * Load the plugin's default configuration form xml file.
     */
    public static void loadDefault() {
        styleEntry.clearAll();
        
        InputStream stream =
            StyleEditorPlugin.class.getResourceAsStream(
                "default-configuration.xml");
        XMLReader reader = new XMLReader("ISO-8859-1");
        XMLNode root;
        try {
            root = reader.readAsXMLNode(stream);
        }
        catch (XMLException e) {
            log(e);
            return;
        }
        internalLoad(root);
    }

    /**
     * log the plugin error log using Throwable.
     * @param e Throwable object.
     */
    public static void log(Throwable e) {
        ILog log = getDefault().getLog();
        String pluginId = getDefault().getBundle().getSymbolicName();
        log.log(new Status(IStatus.ERROR, pluginId, 10001,
                "internal_error", e));
    }

    /**
     * Save the plugin's configuration to xml file.
     */
    public static void save() {
        IPath path = getDefault().getStateLocation();
        path = path.append("configuration.xml");
        
        XMLFile file = new XMLFile(path.toFile(), "ISO-8859-1");
        
        save(file);
    }

    /**
     * Save the plugin's configuration to xml file.
     * @param file XMLFile object.
     */
    public static void save(XMLFile file) {
        
        XMLNode root = new XMLNode("define");
        
        Map partEntryListMap = styleEntry.getPartitionEntryMap();
        Iterator iterator = partEntryListMap.keySet().iterator();
        
        while (iterator.hasNext()) {
            String ext = (String) iterator.next();
            
            // create 'extension' node.
            XMLNode extNode = new XMLNode("extension");
            extNode.setAttribute("name", ext);
            root.addChild(extNode);
            
            // #ReqID:6168 Set Editor Font.
            FontEntry fontEntry = styleEntry.getFontEntry(ext);
            if (fontEntry != null) {
                extNode.setAttribute("font-name", fontEntry.getName());
                extNode.setAttribute("font-size",
                        String.valueOf(fontEntry.getSize()));
            }
            
            // get entry list for extension.
            PartitionEntryList partEntryList =
                    styleEntry.getPartitionEntryList(ext, false);
            
            for (int i = 0; i < partEntryList.size(); i++) {
                
                // create 'partition' node.
                IPartitionEntry partEntry = partEntryList.getPartitionEntry(i);
                XMLNode pNode = partEntry.createNode();
                extNode.addChild(pNode);
                
                // create 'partition' child nodes.
                DependsEntryList depdEntryList =
                        partEntry.getDependsEntryList();
                for (int j = 0; j < depdEntryList.size(); j++) {
                    IDependsEntry depdEntry = depdEntryList.getDependsEntry(j);
                    XMLNode dNode = depdEntry.createNode();
                    pNode.addChild(dNode);
                }
            }
        }
        
        try {
            file.writeNode(root);
        }
        catch (XMLException e) {
            log(e);
            return;
        }
    }
    
    /**
     * Return version of org.eclipse.core.runtime plugin.
     * @return  version
     */
    public static String getRuntimeVersion() {
        String version = (String) Platform.getBundle(
                "org.eclipse.core.runtime").getHeaders().get(
                        org.osgi.framework.Constants.BUNDLE_VERSION);
        return version;
    }

    /**
     * Returns the shared instance.
     * @return DefinedEditorPlugin
     */
    public static StyleEditorPlugin getDefault() {
        return fPlugin;
    }

    /**
     * Load the plugin's configuration form XMLNode of the root.
     * @param root XMLNode of the root.
     */
    private static void internalLoad(XMLNode root) {
        
        // load any extensions entries.
        XMLNodeList extNodeList = root.getChildNodes();
        for (int i = 0; i < extNodeList.size(); i++) {
            XMLNode extNode = extNodeList.getNode(i);
            
            // get partition entry for current extension.
            String ext = extNode.getAttribute("name");
            PartitionEntryList partitionEntryList =
                    styleEntry.getPartitionEntryList(ext, true);
            
            // #ReqID:6168 Set Editor Font.
            String fontName = extNode.getAttribute("font-name");
            String fontSize = extNode.getAttribute("font-size");
            if (fontName != null && fontSize != null) {
                FontEntry fontEntry = new FontEntry(fontName, fontSize);
                styleEntry.addFontEntry(fontEntry, ext);
            }
            
            // load any partitions entries.
            XMLNodeList partNodeList = extNode.getChildNodes();
            for (int j = 0; j < partNodeList.size(); j++) {
                
                // get a partition node.
                XMLNode partNode = partNodeList.getNode(j);
                
                // add partition entry.
                IPartitionEntry partEntry = new PartitionEntry(partNode, ext);
                partitionEntryList.add(partEntry);
                
                
                // load a partition entries.
                DependsEntryList dependsEntryList =
                        partEntry.getDependsEntryList();
                XMLNodeList entryNodeList = partNode.getChildNodes();
                
                for (int k = 0; k < entryNodeList.size(); k++) {
                    XMLNode entryNode = entryNodeList.getNode(k);
                    dependsEntryList.add(new DependsEntry(entryNode, ext));
                }
            }
        }
    }
    
    /**
     * Jeex Style Editor's Singleton Core Object.
     */
    private StyleEditorCore fCore;
    
    /**
     * Default Constructor.
     */
    public StyleEditorPlugin() {
        super();
        fPlugin = this;
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.BundleContext)
     */
    public void start(BundleContext context) throws Exception {
        super.start(context);
        if (styleEntry.isSizeZero()) {
            load();
        }
    }
    
    /**
     * Return Jeex Style Editor's Tool.
     * @return EditorTools
     */
    public synchronized StyleEditorTools getEditorTools() {
        return getCore().getEditorTools();
    }
    
    /**
     * Return Jeex Style Editor's ColorManager.
     * @return ColorManager
     */
    public synchronized IColorManager getColorManager() {
        return getCore().getColorManager();
    }
    
    /**
     * {@inheritDoc}
     */
    public void stop(BundleContext context) throws Exception {
        getCore().dispose();
        super.stop(context);
    }
    
    /**
     * Return Jeex Style Editor's Singleton Core Object.
     * @return StyleEditorCore
     */
    public synchronized StyleEditorCore getCore() {
        if (fCore == null) {
            fCore = new StyleEditorCore();
        }
        return fCore;
    }
}