/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/entry/DefinedStyleEntry.java,v 1.4 2007/09/22 04:10:45 kohnosuke Exp $
 * $Revision: 1.4 $
 * $Date: 2007/09/22 04:10:45 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.entry;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * DefinedStyleEntry
 * @since 1.0.0
 */
public class DefinedStyleEntry {

    /** default extension. */
    private static final String DEFAULT_EXT = "*";

    /** partition entry list map of extensions. */
    private final Map fExtPartitionEntryMap = new LinkedHashMap();

    /** font map of extensions. */
    private final Map fExtFontMap = new LinkedHashMap();

    /**
     * Initiate this object. for default constructor.
     */
    public DefinedStyleEntry() {
        super();
    }

    /**
     * Add IEntry to Managing list.
     * @param entry IEntry.
     * @param ext   extension name. (ex. java, means *.java)
     */
    public void add(IEntry entry, String ext) {
        this.getEntryList(entry, ext).add(entry);
    }

    /**
     * Clear all managing lists.
     */
    public void clearAll() {
        this.fExtPartitionEntryMap.clear();
    }

    /**
     * Returns IPartitionEntry. if partName not found, return null.
     * @param partName partition name.
     * @param ext   extension name. (ex. java, means *.java)
     * @param create if not exist, create one.
     * @return IPartitionEntry
     */
    public IPartitionEntry getPartitionEntry(String partName, String ext,
            boolean create) {
        PartitionEntryList list = getPartitionEntryList(ext, create);
        for (int i = 0; i < list.size(); i++) {
            IPartitionEntry entry = list.getPartitionEntry(i);
            if (entry.getName().equals(partName)) {
                return entry;
            }
        }
        if (fExtPartitionEntryMap.containsKey(ext) == false) {
            return getPartitionEntry(partName, DEFAULT_EXT, create);
        }
        else {
            return new PartitionEntry();
        }
    }

    /**
     * Returns PartitionEntryMap.
     * @return PartitionEntryMap.
     */
    public Map getPartitionEntryMap() {
        return this.fExtPartitionEntryMap;
    }

    /**
     * Returns PartitionEntryList.
     * @param ext   extension name. (ex. java, means *.java)
     * @param create if not exist, create one.
     * @return PartitionEntryList.
     */
    public PartitionEntryList getPartitionEntryList(String ext,
            boolean create) {
        PartitionEntryList list =
                (PartitionEntryList) this.fExtPartitionEntryMap.get(ext);
        if (list == null && !create) {
            // get default
            list = (PartitionEntryList)
                    this.fExtPartitionEntryMap.get(DEFAULT_EXT);
        }
        if (list == null && create) {
            createNewParitionEntryMap(ext);
            return getPartitionEntryList(ext, false);
        }
        return list;
    }

    /**
     * Create new partition entry list.
     * @param ext   extension name. (ex. java, means *.java)
     */
    public void createNewParitionEntryMap(String ext) {
        this.fExtPartitionEntryMap.put(ext, new PartitionEntryList());
    }

    /**
     * Remove IEntry from Managing list.
     * @param entry IEntry.
     * @param ext   extension name. (ex. java, means *.java)
     */
    public void remove(IEntry entry, String ext) {
        this.getEntryList(entry, ext).remove(entry);
    }

    /**
     * Size of partition entry.
     * @return partition entry size.
     */
    public boolean isSizeZero() {
        return fExtPartitionEntryMap.size() == 0;
    }

    /**
     * Returns target's managing list.
     * @param entry IEntry.
     * @param ext   extension name. (ex. java, means *.java)
     * @return target's managing list.
     */
    private List getEntryList(IEntry entry, String ext) {
        if (entry instanceof IPartitionEntry) {
            return getPartitionEntryList(ext, false);
        }
        else {
            IDependsEntry depdEntry = (IDependsEntry) entry;
            String partName = depdEntry.getDepends().getName();
            
            IPartitionEntry partEntry =
                    getPartitionEntry(partName, ext, false);
            return partEntry.getDependsEntryList();
        }
    }
    
    /**
     * Add new FontEntry.
     * @param entry new FontEntry.
     * @param ext   name of file's extension.
     */
    public void addFontEntry(FontEntry entry, String ext) {
        fExtFontMap.put(ext, entry);
    }
    
    /**
     * Return FontEntry object for extension.
     * @param ext   name of file's extension.
     * @return  FontEntry object.
     */
    public FontEntry getFontEntry(String ext) {
        return (FontEntry) fExtFontMap.get(ext);
    }
    
    /**
     * Return FontEntryMap object.
     * @return  FontEntryMap object.
     */
    public Map getFontEntryMap() {
        return fExtFontMap;
    }
}