=begin rdoc
 =jimpachi_list.rb - jimpachi list generater

 Author:: Kuzuno Hiroki
 Copyright:: Copyright (c) 2004 Kuzuno Hiroki
 License:: GNU General Public License version 2, or any later.

 == Example
  % ./jimpachi_list.rb -D "Directory path" -o "list filename"
  % cat "list filename"
  hoge.mp3,/dirpath/hoge.mp3
  hogehoge.mp3,/dirpath/hogehoge.mp3
=end

require 'getopts'

# Jimpachi list generater class
#
# read directory to generate jimpachi list
class JimpachiList

  def initialize
    @jimpachi_list= []
    @list_fname= ""
    @recursive_num= 0
    parse_option
    parse_args
    output_list
  end

  def print_help
    $stderr.print <<HELP
Usage: jimpachi_list [options] [--] [arguments]
    -D              directory name
    -r              recursive directory
    -o              list output filename
    -h, --help      show this usage message
HELP
  end

  def parse_option
    unless getopts("h", "r", "D:","o:","help")
      $stderr.puts("jimpachi_list: illegal option")
      print_usage
      exit(2)
    end
  end
  
  def parse_args()
    if $OPT_help || $OPT_h || !$OPT_o
      print_help
      exit(2)
    end
    
    if $OPT_D
      p $OPT_D
      read_dir( $OPT_D )
    end
    
    if $OPT_o
      set_list_fname( $OPT_o )
    end
    
    if $OPT_r
      set_recursive_num( $OUT_r )
    end
  end

  def print_usage
    $stderr.puts("Usage: jimpachi_list [options] [--] [arguments]")
    $stderr.puts("Try 'jimpachi_list -h or --help' for more information.")
  end

  # add file to list
  # ===Parameters:
  # * file_name: file name string
  # * file_path: absolute file path
  def add_file( file_name, file_path="" )
    if file_path =~ /mp3\z/
      @jimpachi_list.push( "#{file_name},#{file_path}" )
    end
  end
  
  # read directory, if set -r then recursive search
  # call add_file( file_name, file_path)
  # ===Parameters:
  # * dir_path: directory path
  def read_dir( dir_path )
    absolute_path= File.expand_path( dir_path )
    Dir.foreach( absolute_path ) do |file_name|
      new_path= absolute_path + "/" + file_name
      recursive_read_dir( new_path ) if $OPT_r
      add_file( file_name, new_path )
    end
  end

  def output_list
    begin
      list_fname_fd= File.open( @list_fname, "w+" )
      @jimpachi_list.each do |file_path|
        list_fname_fd.print file_path, "\n"
      end
    ensure
      list_fname_fd.close
    end
  end

  private
  def set_list_fname( list_fname ) 
    @list_fname= list_fname
  end

  def set_recursive_num( recursive_num )
    @recursive_num= recursive_num
  end

  def recursive_read_dir( new_path="" )
    unless new_path  =~ /\.\z/ || new_path =~ /\.\.\z/ 
      if "directory" ==  File.ftype( new_path )
        read_dir( new_path )
      end
    end
  end

end

# main start here
if $0 == __FILE__
  jimpachi_list= JimpachiList.new()
end
