/*
 * Account accountPanel
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: AccountPanel.java 3 2008-06-11 15:08:13Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.border.EmptyBorder;

/**
 * ログインパネル。
 */
@SuppressWarnings("serial")
public class AccountPanel
        extends JFrame
        implements ActionListener, ItemListener{

    private static final String frameTitle = "アカウント管理 - " + Jindolf.title;

    private final LandsModel landsModel;
    private final Map<Land, String> landUserIDMap = new HashMap<Land, String>();
    private final Map<Land, char[]> landPasswordMap = new HashMap<Land, char[]>();

    private final JComboBox landBox = new JComboBox();
    private final JTextField idField = new JTextField(15);
    private final JPasswordField pwField = new JPasswordField(15);
    private final JButton loginButton = new JButton("ログイン");
    private final JButton logoutButton = new JButton("ログアウト");
    private final JButton closeButton = new JButton("閉じる");
    private final JTextField status = new JTextField();

    /**
     * アカウントパネルを生成。
     * @param landsModel 国モデル
     * @throws java.lang.NullPointerException 引数がnull
     */
    public AccountPanel(LandsModel landsModel) throws NullPointerException{
        super(frameTitle);

        if(landsModel == null) throw new NullPointerException();
        this.landsModel = landsModel;
        this.landsModel.loadLandList();
        for(Land land : this.landsModel.getLandList()){
            String userID = "";
            char[] password = {};
            landUserIDMap.put(land, userID);
            landPasswordMap.put(land, password);
            landBox.addItem(land);
        }

        setResizable(true);
        Toolkit kit = getToolkit();
        kit.setDynamicLayout(false);
        setIconImage(GUIUtils.getWindowIconImage());
        setLocationByPlatform(true);

        landBox.setEditable(false);
        landBox.addItemListener(this);

        pwField.setEchoChar('●');

        status.setEditable(false);

        loginButton.addActionListener(this);
        logoutButton.addActionListener(this);
        closeButton.addActionListener(this);

        JPanel accountPanel = new JPanel();
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints cst = new GridBagConstraints();
        accountPanel.setLayout(layout);

        cst.insets = new Insets(5,5,5,5);
        cst.fill = GridBagConstraints.NONE;

        cst.anchor = GridBagConstraints.EAST;
        accountPanel.add(new JLabel("国名 :"), cst);
        cst.anchor = GridBagConstraints.WEST;
        accountPanel.add(landBox, cst);

        cst.gridy = 1;
        cst.anchor = GridBagConstraints.EAST;
        accountPanel.add(new JLabel("ID :"), cst);
        cst.anchor = GridBagConstraints.WEST;
        accountPanel.add(idField, cst);

        cst.gridy = 2;
        cst.anchor = GridBagConstraints.EAST;
        accountPanel.add(new JLabel("パスワード :"), cst);
        cst.anchor = GridBagConstraints.WEST;
        accountPanel.add(pwField, cst);

        accountPanel.setBorder(new EmptyBorder(10,10,10,10));

        JPanel buttonPanel = new JPanel();
        BoxLayout boxLayout;
        boxLayout = new BoxLayout(buttonPanel, BoxLayout.X_AXIS);
        buttonPanel.setLayout(boxLayout);
        buttonPanel.add(loginButton);
        buttonPanel.add(logoutButton);
        buttonPanel.add(closeButton);

        Container content = getContentPane();
        boxLayout = new BoxLayout(content, BoxLayout.Y_AXIS);
        content.setLayout(boxLayout);

        content.add(accountPanel);
        content.add(new JSeparator());
        content.add(status);
        content.add(new JSeparator());
        content.add(buttonPanel);

        preSelectActiveLand();

        updateGUI();

        return;
    }

    /**
     * 現在コンボボックスで選択中の国を返す。
     * @return 現在選択中のLand
     */
    public Land getSelectedLand(){
        Land land = (Land)( this.landBox.getSelectedItem() );
        return land;
    }

    /**
     * ACTIVEな最初の国がコンボボックスで既に選択されている状態にする。
     */
    private void preSelectActiveLand(){
        for(int index=0; index < this.landBox.getItemCount(); index++){
            Object item = this.landBox.getItemAt(index);
            Land land = (Land) item;
            Land.Type type = land.getType();
            if(type == Land.Type.ACTIVE){
                this.landBox.setSelectedItem(land);
                return;
            }
        }
        return;
    }

    /**
     * 指定された国のユーザIDを返す。
     * @param land 国
     * @return ユーザID
     */
    private String getUserID(Land land){
        return this.landUserIDMap.get(land);
    }

    /**
     * 指定された国のパスワードを返す。
     * @param land 国
     * @return パスワード
     */
    private char[] getPassword(Land land){
        return this.landPasswordMap.get(land);
    }

    /**
     * ネットワークエラーを通知するモーダルダイアログを表示する。
     * OKボタンを押すまでこのメソッドは戻ってこない。
     * @param e ネットワークエラー
     */
    protected void showNetworkError(IOException e){
        Land land = getSelectedLand();
        ServerAccess server = land.getServerAccess();
        String message =
                land.getLandName()
                +"を運営するサーバとの間の通信で何らかのトラブルが発生しました。\n"
                +"相手サーバのURLは [ " + server.getBaseURL() + " ] だよ。\n"
                +"Webブラウザでも遊べないか確認してみてね!\n";

        JOptionPane pane = new JOptionPane(message,
                                           JOptionPane.WARNING_MESSAGE,
                                           JOptionPane.DEFAULT_OPTION );

        JDialog dialog = pane.createDialog(this, "通信異常発生 - " + Jindolf.title);

        dialog.pack();
        dialog.setVisible(true);
        dialog.dispose();

        return;
    }

    /**
     * アカウントエラーを通知するモーダルダイアログを表示する。
     * OKボタンを押すまでこのメソッドは戻ってこない。
     */
    protected void showIllegalAccountDialog(){
        Land land = getSelectedLand();
        String message =
                land.getLandName()
                +"へのログインに失敗しました。\n"
                +"ユーザ名とパスワードは本当に正しいかな？\n"
                +"あなたは本当に [ " + getUserID(land) + " ] さんかな？\n"
                +"WebブラウザによるID登録手続きは本当に完了してるかな？\n"
                +"Webブラウザでもログインできないか試してみて！\n"
                +"…ユーザ名やパスワードにある種の特殊文字を使っている人は問題があるかも。";

        JOptionPane pane = new JOptionPane(message,
                                           JOptionPane.WARNING_MESSAGE,
                                           JOptionPane.DEFAULT_OPTION );

        JDialog dialog = pane.createDialog(this, "ログイン認証失敗 - " + Jindolf.title);

        dialog.pack();
        dialog.setVisible(true);
        dialog.dispose();

        return;
    }

    /**
     * 入力されたアカウント情報を基に現在選択中の国へログインする。
     * @return ログインに成功すればtrueを返す。
     */
    protected boolean login(){
        Land land = getSelectedLand();
        ServerAccess server = land.getServerAccess();

        String id = this.idField.getText();
        char[] password = this.pwField.getPassword();
        landUserIDMap.put(land, id);
        landPasswordMap.put(land, password);

        boolean result = false;
        try{
            result = server.login(id, password);
        }catch(IOException e){
            showNetworkError(e);
            return false;
        }
        
        if( ! result ){
            showIllegalAccountDialog();
        }

        return result;
    }

    /**
     * 現在選択中の国からログアウトする。
     */
    protected void logout(){
        try{
            logoutInternal();
        }catch(IOException e){
            showNetworkError(e);
        }
        return;
    }

    /**
     * 現在選択中の国からログアウトする。
     * @throws java.io.IOException ネットワークエラー
     */
    protected void logoutInternal() throws IOException{
        Land land = getSelectedLand();
        ServerAccess server = land.getServerAccess();
        server.logout();
        return;
    }

    /**
     * 現在選択中の国のログイン状態に合わせてGUIを更新する。
     */
    protected void updateGUI(){
        Land land = getSelectedLand();
        Land.Type type = land.getType();
        ServerAccess server = land.getServerAccess();
        boolean hasLoggedIn = server.hasLoggedIn();

        if(type != Land.Type.ACTIVE){
            status.setText("この国は既に滅びました。ログインは無意味です");
            idField.setEnabled(false);
            pwField.setEnabled(false);
            loginButton.setEnabled(false);
            logoutButton.setEnabled(false);
        }else if(hasLoggedIn){
            status.setText("ユーザ [ " + getUserID(land) + " ] として現在ログイン中です");
            idField.setEnabled(false);
            pwField.setEnabled(false);
            loginButton.setEnabled(false);
            logoutButton.setEnabled(true);
        }else{
            status.setText("現在ログインしていません");
            idField.setEnabled(true);
            pwField.setEnabled(true);
            loginButton.setEnabled(true);
            logoutButton.setEnabled(false);
        }

        return;
    }

    /**
     * ボタン操作のリスナ
     * @param event イベント
     */
    public void actionPerformed(ActionEvent event){
        Object source = event.getSource();

        if(source == this.closeButton){
            setVisible(false);
            dispose();
            return;
        }

        if(source == this.loginButton){
            login();
        }else if(source == this.logoutButton){
            logout();
        }

        updateGUI();

        return;
    }

    /**
     * コンボボックス操作のリスナ
     * @param event イベント
     */
    public void itemStateChanged(ItemEvent event){
        Object source = event.getSource();
        if(source != this.landBox) return;

        Land land = (Land) event.getItem();
        String id;
        char[] password;

        switch(event.getStateChange()){
        case ItemEvent.SELECTED:
            id = getUserID(land);
            password = getPassword(land);
            this.idField.setText(id);
            this.pwField.setText(new String(password));
            updateGUI();
            break;
        case ItemEvent.DESELECTED:
            id = this.idField.getText();
            password = this.pwField.getPassword();
            landUserIDMap.put(land, id);
            landPasswordMap.put(land, password);
            break;
        default:
            assert false;
            return;
        }

        return;
    }
}
