/*
 * MVC controller
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Controller.java 3 2008-06-11 15:08:13Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Container;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.net.URL;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.LookAndFeel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.tree.TreePath;

/**
 * いわゆるMVCでいうとこのコントローラ
 */
public class Controller
        implements ActionListener,
                   TreeWillExpandListener,
                   TreeSelectionListener,
                   ChangeListener            {

    private MenuManager menuManager;
    private TopFrameView topFrame;
    private LandsModel model;
    private JFrame helpFrame;
    private FilterPanel filterFrame;
    private AccountPanel accountFrame;

    /**
     * コントローラの生成。
     * @param menuManager アクション管理
     * @param topFrame 最上位ビュー
     * @param model 最上位データモデル
     */
    public Controller(MenuManager menuManager,
                       TopFrameView topFrame,
                       LandsModel model){
        super();

        this.menuManager = menuManager;
        this.topFrame = topFrame;
        this.model = model;

        this.menuManager.addActionListener(this);

        JTree treeView = topFrame.getLandTree();
        treeView.setModel(model);
        treeView.addTreeWillExpandListener(this);
        treeView.addTreeSelectionListener(this);
        
        topFrame.getTabBrowser().addChangeListener(this);
        topFrame.getReloadButton().addActionListener(this);
        topFrame.getSearchButton().addActionListener(this);

        this.filterFrame = new FilterPanel();
//        this.filterFrame.setAlwaysOnTop(true);
        this.filterFrame.addChangeListener(this);
        this.filterFrame.pack();
        this.filterFrame.setVisible(false);

        return;
    }

    /**
     * 検索ボタン押下処理。
     */
    private void searchPeriod(){
        updatePeriod(false);
        return;
    }
    
    /**
     * 現在選択中のの検索パターンを得る。
     * @return
     */
    private Pattern currentPattern(){
        JComboBox box = this.topFrame.getFindBox();
        Object selected = box.getSelectedItem();
        if(selected == null) return null;
        String searchString = selected.toString();
        if(searchString.length() <= 0) return null;
        
        Pattern pattern;
        try{
            pattern= Pattern.compile(searchString, Pattern.DOTALL);
        }catch(PatternSyntaxException e){
            return null;
        }

        return pattern;
    }

    /**
     * Period表示の更新処理。
     * @param force trueならPeriodデータを強制再読み込み。
     */
    private void updatePeriod(final boolean force){
        final Village village = this.topFrame.currentVillage();
        if(village == null) return;
        this.topFrame.setFrameTitle(village.getVillageName());
        
        final TabBrowser tabBrowser = this.topFrame.getTabBrowser();
        final JdfBrowser browser = this.topFrame.currentJdfBrowser();
        if(browser == null) return;
        final Period period = browser.getPeriod();
        if(period == null) return;
        
        new Thread(){
            @Override
            public void run(){
                topFrame.setEnabled(false);
                
                topFrame.setBusy("1日分のデータを読み込んでいます…", true);
                boolean wasHot = period.isHot();
                period.loadPeriod(force);
                topFrame.setBusy("1日分のデータを読み終わりました", false);
                Thread.yield();
                
                if(wasHot && ! period.isHot() ){
                    village.updatePeriodList();
                    tabBrowser.setVillage(village);
                }
                Thread.yield();
                
                topFrame.setBusy("フィルタリング中…", true);
                browser.applyFilter(filterFrame, force);
                browser.highlightRegex(currentPattern());
                topFrame.setBusy("フィルタリング完了", false);
                
                topFrame.setEnabled(true);
            }
        }.start();
        
        Thread.yield();
        
        return;
    }

    /**
     * 発言フィルタの操作による更新処理。
     */
    private void filterChanged(){
        final JdfBrowser jdfBrowser = this.topFrame.currentJdfBrowser();
        if(jdfBrowser == null) return;

        new Thread(){
            @Override
            public void run(){
                topFrame.setEnabled(false);

                topFrame.setBusy("フィルタリング中…", true);
                jdfBrowser.applyFilter(filterFrame, false);
                jdfBrowser.highlightRegex(currentPattern());
                topFrame.setBusy("フィルタリング完了", false);

                topFrame.setEnabled(true);
            }
        }.start();

        return;
    }
    
    /**
     * L&Fの変更を行う。
     */
    private void changeLandF(){
        String className = menuManager.getSelectedLookAndFeel();

        LookAndFeel lnf;
        try{
            Class lnfClass = Class.forName(className);
            Object lnfObj = lnfClass.newInstance();
            if(!(lnfObj instanceof LookAndFeel)){
                return;
            }
            lnf = (LookAndFeel) lnfObj;
        }catch(ClassNotFoundException e){
            return;
        }catch(InstantiationException e){
            return;
        }catch(IllegalAccessException e){
            return;
        }

        try{
            UIManager.setLookAndFeel(lnf);
        }catch(UnsupportedLookAndFeelException e){
            return;
        }

        SwingUtilities.updateComponentTreeUI(topFrame);
        topFrame.validate();

        if(helpFrame != null){
            SwingUtilities.updateComponentTreeUI(helpFrame);
            helpFrame.validate();
        }
        if(filterFrame != null){
            SwingUtilities.updateComponentTreeUI(filterFrame);
            filterFrame.validate();
            filterFrame.pack();
        }
        if(accountFrame != null){
            SwingUtilities.updateComponentTreeUI(accountFrame);
            accountFrame.validate();
            accountFrame.pack();
        }
        
        return;
    }

    /**
     * アプリ終了。
     */
    private void doExit(){
        Jindolf.exit(0);
    }

    /**
     * フレーム表示のトグル処理
     * @param frame フレーム
     */
    // TODO アイコン化からの復帰もサポートしたい
    private void toggleFrame(JFrame frame){
        if(frame == null) return;
        if(frame.isVisible()){
            frame.setVisible(false);
            frame.dispose();
        }else{
            frame.setVisible(true);
        }
        return;
    }
    
    /**
     * 発言フィルタ画面を表示する。
     */
    private void showFilter(){
        toggleFrame(filterFrame);
        return;
    }
    
    /**
     * アカウント管理画面を表示する。
     */
    private void showAccount(){
        if(accountFrame != null){                 // show Toggle
            toggleFrame(accountFrame);
            return;
        }

        accountFrame = new AccountPanel(model);
        accountFrame.pack();
        accountFrame.setVisible(true);

        return;
    }

    /**
     * Help画面を表示する。
     */
    private void showHelp(){
        if(helpFrame != null){                 // show Toggle
            toggleFrame(helpFrame);
            return;
        }

        helpFrame = new JFrame(Jindolf.title + " ヘルプ");

        helpFrame.setResizable(true);
        Toolkit kit = helpFrame.getToolkit();
        kit.setDynamicLayout(false);
        helpFrame.setIconImage(GUIUtils.getWindowIconImage());
        helpFrame.setLocationByPlatform(true);

        HTMLEditorKit editorKit = new UnshareStyleEditorKit();

        JEditorPane edit = new JEditorPane();
        edit.setEditorKit(editorKit);
        edit.setEditable(false);
        edit.setContentType("text/html");
        edit.putClientProperty(JEditorPane.W3C_LENGTH_UNITS, Boolean.TRUE);
        edit.setBorder(new EmptyBorder(0,0,0,0));

        URL htmlurl = getClass().getResource("resources/help.html");
        try{
            edit.setPage(htmlurl);
        }catch(IOException e){
            helpFrame = null;
            return;
        }

        JScrollPane sc = new JScrollPane(edit);

        Container content = helpFrame.getContentPane();
        content.add(sc);

        helpFrame.pack();
        helpFrame.setSize(450, 450);
        helpFrame.setVisible(true);

        return;
    }

    /**
     * About画面を表示する。
     */
    private void showAbout(){
        String message =
                Jindolf.title
                + "   Version " + Jindolf.version + "\n"
                + Jindolf.copyright + "\n"
                + "ライセンス: " + Jindolf.license + "\n"
                + "連絡先: " + Jindolf.contact;

        JOptionPane pane = new JOptionPane(message,
                                           JOptionPane.INFORMATION_MESSAGE,
                                           JOptionPane.DEFAULT_OPTION,
                                           GUIUtils.getLogoIcon());

        JDialog dialog = pane.createDialog(topFrame, Jindolf.title + "について");

// JRE1.6 only
//      dialog.setIconImage(GUIUtils.getWindowIconImage());
        
        dialog.pack();
        dialog.setVisible(true);
        dialog.dispose();

        return;
    }

    /**
     * ツリーリストで何らかの要素（国、村）がクリックされたときの処理。
     * @param event イベント
     */
    public void valueChanged(TreeSelectionEvent event){
        TreePath path = event.getNewLeadSelectionPath();
        Object selobj = path.getLastPathComponent();

        if( selobj instanceof Land ){
            Land land = (Land)selobj;
            this.topFrame.showLandInfo(land);
            return;
        }

        if( selobj instanceof Village ){  // TODO 別スレッド化&ビジー処理
            Village village = (Village)selobj;
            village.updatePeriodList();
            this.topFrame.showVillageInfo(village);
        }

        return;
    }

    /**
     * Periodがタブ選択されたときもしくは発言フィルタが操作されたときの処理。
     * @param event イベント
     */
    public void stateChanged(ChangeEvent event){
        Object source = event.getSource();
        
        if(source == filterFrame){
            filterChanged();
        }else if(source instanceof TabBrowser){
            updatePeriod(false);
        }
        return;
    }

    /**
     * アクションイベント処理。
     * 主にメニュー選択やボタン押下など。
     * @param e イベント
     */
    public void actionPerformed(ActionEvent e){
        String cmd = e.getActionCommand();
        if(cmd.equals(MenuManager.COMMAND_ABOUT)){
            showAbout();
        }else if(cmd.equals(MenuManager.COMMAND_EXIT)){
            doExit();
        }else if(cmd.equals(MenuManager.COMMAND_LANDF)){
            changeLandF();
        }else if(cmd.equals(MenuManager.COMMAND_HELPDOC)){
            showHelp();
        }else if(cmd.equals(MenuManager.COMMAND_FILTER)){
            showFilter();
        }else if(cmd.equals(MenuManager.COMMAND_ACCOUNT)){
            showAccount();
        }else if(cmd.equals(MenuManager.COMMAND_RELOAD)){
            updatePeriod(true);
        }else if(cmd.equals(MenuManager.COMMAND_SEARCH)){
            searchPeriod();
        }
        return;
    }
    
    /**
     * 村選択ツリーリストが畳まれるとき呼ばれる。
     * @param event ツリーイベント
     */
    public void treeWillCollapse(TreeExpansionEvent event){
        return;
    }

    /**
     * 村選択ツリーリストが展開されるとき呼ばれる。
     * @param event ツリーイベント
     */
    public void treeWillExpand(TreeExpansionEvent event){
        if(!(event.getSource() instanceof JTree)){
            return;
        }

        TreePath path = event.getPath();
        Object lastObj = path.getLastPathComponent();
        if(!(lastObj instanceof Land)){
            return;
        }
        final Land land = (Land) lastObj;
        if(land.getVillageCount() > 0){
            return;
        }

        new Thread(){
            @Override
            public void run(){
                topFrame.setEnabled(false);
                topFrame.setBusy("村一覧を読み込み中…", true);
                model.loadVillageList(land);
                topFrame.setBusy("村一覧の読み込み完了", false);
                topFrame.setEnabled(true);
            }
        }.start();

        return;
    }
}
