/*
 * Filter panel
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: FilterPanel.java 3 2008-06-11 15:08:13Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.LayoutManager;
import java.awt.Toolkit;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.BitSet;
import java.util.HashMap;
import java.util.Map;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;

/**
 * 発言フィルタ GUI
 */
@SuppressWarnings("serial")
public class FilterPanel extends JFrame
        implements ItemListener, TopicFilter{

    private JCheckBox checkPublic = new JCheckBox("公開", true);
    private JCheckBox checkWolf = new JCheckBox("狼", true);
    private JCheckBox checkPrivate = new JCheckBox("独り言", true);
    private JCheckBox checkGrave = new JCheckBox("墓下", true);

    private Map<Avatar, JCheckBox> cbMap = new HashMap<Avatar, JCheckBox>();

    private EventListenerList listeners;

    /**
     * 発言フィルタを生成する。
     */
    public FilterPanel(){
        super("発言フィルタ - " + Jindolf.title);

        setResizable(true);
        Toolkit kit = getToolkit();
        kit.setDynamicLayout(false);
        setIconImage(GUIUtils.getWindowIconImage());
        setLocationByPlatform(true);

        LayoutManager layout;

        JPanel avatarPanel = new JPanel();
        layout = new GridBagLayout();
        GridBagConstraints cst = new GridBagConstraints();
        cst.weightx = 0.25;
        cst.anchor = GridBagConstraints.WEST;
        avatarPanel.setLayout(layout);

        int ct = 0;
        for(Avatar avatar : Avatar.getPredefinedAvatars()){
            JCheckBox checkBox = new JCheckBox(avatar.getName(), true);
            if(ct >= 3){
                cst.gridwidth = GridBagConstraints.REMAINDER;
                ct = 0;
            }else{
                cst.gridwidth = 1;
                ct++;
            }
            avatarPanel.add(checkBox, cst);
            checkBox.addItemListener(this);
            cbMap.put(avatar, checkBox);
        }
        JScrollPane avatarsc = new JScrollPane(avatarPanel);
        avatarsc.setHorizontalScrollBarPolicy(
                ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER
        );

        JPanel topicPanel = new JPanel();
        layout = new BoxLayout(topicPanel, BoxLayout.Y_AXIS);
        topicPanel.setLayout(layout);
        topicPanel.add(checkPublic);
        topicPanel.add(checkWolf);
        topicPanel.add(checkPrivate);
        topicPanel.add(checkGrave);
        checkPublic.addItemListener(this);
        checkWolf.addItemListener(this);
        checkPrivate.addItemListener(this);
        checkGrave.addItemListener(this);

        Container content = getContentPane();
        layout = new BoxLayout(content, BoxLayout.X_AXIS);
        content.setLayout(layout);
        content.add(topicPanel);
        content.add(avatarsc);

        listeners = new EventListenerList();

        return;
    }

    /**
     * リスナを登録する。
     * @param listener リスナ
     */
    public void addChangeListener(ChangeListener listener){
        listeners.add(ChangeListener.class, listener);
    }

    /**
     * リスナを削除する。
     * @param listener リスナ
     */
    public void removeChangeListener(ChangeListener listener){
        listeners.remove(ChangeListener.class, listener);
    }

    /**
     * 全リスナを取得する。
     * @return リスナの配列
     */
    public ChangeListener[] getChangeListeners(){
        return listeners.getListeners(ChangeListener.class);
    }

    /**
     * 全リスナへチェックボックス操作を通知する。
     */
    protected void fireCheckChanged(){
        ChangeEvent changeEvent = new ChangeEvent(this);
        for(ChangeListener listener : getChangeListeners()){
            listener.stateChanged(changeEvent);
        }
    }

    /**
     * チェックボックス操作時にリスナとして呼ばれる。
     * @param event イベント
     */
    public void itemStateChanged(ItemEvent event){
        fireCheckChanged();
    }

    /**
     * フィルタ判定を行う。
     * @param topic 判定対象のTopic
     * @return フィルタ対象ならtrue
     */
    public boolean isFiltered(Topic topic){
        Talk talk;
        if(topic instanceof Talk){
            talk = (Talk) topic;
        }else{
            return false;
        }

        JCheckBox cbox;

        Talk.Type type = talk.getTalkType();
        switch(type){
        case PUBLIC:
            cbox = checkPublic;
            break;
        case WOLFONLY:
            cbox = checkWolf;
            break;
        case PRIVATE:
            cbox = checkPrivate;
            break;
        case GRAVE:
            cbox = checkGrave;
            break;
        default:
            assert false;
            return true;
        }
        if( ! cbox.isSelected()){
            return true;
        }

        Avatar avatar = talk.getAvatar();
        cbox = cbMap.get(avatar);
        if( ! cbox.isSelected()){
            return true;
        }

        return false;
    }

    /**
     * カスタム化されたフィルタ状態。
     */
    private class FilterPanelContext implements FilterContext{
        private final BitSet context = new BitSet();
        public FilterPanelContext(){
            int index = 0;
            context.set(index++, checkPublic.isSelected());
            context.set(index++, checkWolf.isSelected());
            context.set(index++, checkPrivate.isSelected());
            context.set(index++, checkGrave.isSelected());

            for(Avatar avatar : Avatar.getPredefinedAvatars()){
                JCheckBox checkBox = cbMap.get(avatar);
                context.set(index++, checkBox.isSelected());
            }

            return;
        }
    }

    /**
     * フィルタ状態を取得する。
     * @return フィルタ状態
     */
    public FilterContext getFilterContext(){
        return new FilterPanelContext();
    }

    /**
     * 与えられたフィルタ状態と今等しい状態にあるかを判定する。
     * @param context フィルタ状態
     * @return 等しければtrue
     */
    public boolean isSame(FilterContext context){
        if(context == null) return false;
        if( ! (context instanceof FilterPanelContext) ) return false;
        FilterPanelContext argContext = (FilterPanelContext) context;
        FilterPanelContext thisContext = (FilterPanelContext) getFilterContext();

        return thisContext.context.equals(argContext.context);
    }

    // TODO 占い結果、狩人の護衛先もフィルタできたほうがいいか？
}
