/*
 * dialogs in game
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Talk.java 3 2008-06-11 15:08:13Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.text.DateFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * プレイヤーの発言
 */
public class Talk implements Topic{
    
    /**
     * 発言種別
     */
    public static enum Type{
        /** 公開発言 */  // white say
        PUBLIC,
        /** 狼発言 */   // red whisper
        WOLFONLY,
        /** 独り言 */   // gray think
        PRIVATE,
        /** 墓下発言 */       // blue gloan
        GRAVE,
    }

    private static DateFormat dform =
            DateFormat.getDateTimeInstance(DateFormat.MEDIUM,
                                           DateFormat.MEDIUM);

    private final Period homePeriod;
    private final Type talkType;
    private final Avatar avatar;
    private final String messageID;
    private final int hour;
    private final int minute;
    private final String dialog;

    /**
     * Talkの生成
     * @param homePeriod 発言元Period
     * @param talkType 発言種別
     * @param avatar Avatar
     * @param messageID メッセージID
     * @param hour 発言時
     * @param minute 発言分
     * @param dialog 会話データ
     */
    public Talk(Period homePeriod,
                 Type talkType,
                 Avatar avatar,
                 String messageID,
                 int hour,
                 int minute,
                 String dialog){
        if(   homePeriod == null
           || talkType   == null
           || avatar     == null
           || messageID  == null
           || dialog     == null ) throw new NullPointerException();
        if(hour   < 0 || 23 < hour  ) throw new IllegalArgumentException();
        if(minute < 0 || 59 < minute) throw new IllegalArgumentException();

        this.homePeriod = homePeriod;
        this.talkType = talkType;
        this.avatar = avatar;
        this.messageID = messageID;
        this.hour = hour;
        this.minute = minute;
        this.dialog = dialog.replace("<br />", "\n");
        return;
    }

    /**
     * 発言種別を得る。
     * @return 種別
     */
    public Type getTalkType(){
        return this.talkType;
    }

    /**
     * 発言元Avatarを得る。
     * @return 発言元Avatar
     */
    public Avatar getAvatar(){
        return this.avatar;
    }

    /**
     * メッセージIDを取得する。
     * @return メッセージID
     */
    public String getMessageID(){
        return this.messageID;
    }

    /**
     * メッセージIDからエポック秒(ms)に変換する。
     * @return GMT 1970-01-01 00:00:00 からのエポック秒(ms)
     */
    private long getTimeFromID(){
        String epoch = this.messageID.replace("mes", "");
        long result = Long.parseLong(epoch) * 1000;
        return result;
    }

    /**
     * 発言時を取得する。
     * @return 発言時
     */
    public int getHour(){
        return this.hour;
    }

    /**
     * 発言分を取得する。
     * @return 発言分
     */
    public int getMinute(){
        return this.minute;
    }

    /**
     * 会話データを取得する。
     * 会話データには実体参照やBRタグが含まれる。
     * @return 会話データ
     */
    public String getDialog(){
        return this.dialog.replace("\n", "<br></br>");
    }

    /**
     * この会話を識別するためのアンカー文字列を生成する。
     * 例えば「3d09:56」など。
     * @return アンカー文字列
     */
    private String getAnchor(){
        int day = this.homePeriod.getDay();

        String hstr = "0"+this.hour;
        hstr = hstr.substring(hstr.length()-2);
        String mstr = "0"+this.minute;
        mstr = mstr.substring(mstr.length()-2);

        return String.valueOf(day) + "d" + hstr + ":" + mstr;
    }

    /**
     * 「午後 6時 12分」の形式で発言時刻の文字列表記を取得する。
     * 空白に&nbsp;を使うので注意。
     * @return 発言時刻の文字列表記
     */
    public String getTime(){
        StringBuilder result = new StringBuilder();

        if(hour <= 11) result.append("午前&nbsp;").append(hour);
        else           result.append("午後&nbsp;").append(hour-12);

        result.append("時&nbsp;").append(minute).append("分");

        return result.toString();
    }

    /**
     * 人狼BBS出力HTML中のdivクラス名から会話種別への変換を行う。
     * @param encoded divクラス名
     * @return 会話種別
     */
    public static Type decodeType(String encoded){
        Talk.Type type;
        if     ( encoded.equals("say")    ) type = Type.PUBLIC;
        else if( encoded.equals("think")  ) type = Type.PRIVATE;
        else if( encoded.equals("whisper")) type = Type.WOLFONLY;
        else if( encoded.equals("groan")  ) type = Type.GRAVE;
        else return null;

        return type;
    }

    /**
     * 内部ブラウザ用に最深divのHTML文字列を生成する。
     * 会話に含まれるアンカー参照を抽出、展開する。
     * @return 内部ブラウザ用のHTML文字列
     */
    private CharSequence buildJdfHTMLDiv(){
        StringBuilder div = new StringBuilder();

        String mesClass;
        switch(getTalkType()){
        case PUBLIC:   mesClass = "public";   break;
        case PRIVATE:  mesClass = "private";  break;
        case WOLFONLY: mesClass = "wolfonly"; break;
        case GRAVE:    mesClass = "grave";    break;
        default:       return null;
        }

        String content = getDialog();
        Pattern anchorRegex =
                Pattern.compile("(([0-9]?[0-9])(?:[dD]|日|日目))?"
                +"([0-2]?[0-9]):([0-5]?[0-9])",
                Pattern.DOTALL);
        Matcher matcher = anchorRegex.matcher(content);

        StringBuffer newContent = new StringBuffer();
        while(matcher.find()){
            String anchor = matcher.group();
            String dd = matcher.group(2);
            String hh = matcher.group(3);
            String mm = matcher.group(4);

            if(dd == null) dd = Integer.toString(homePeriod.getDay());

            StringBuilder href = new StringBuilder();
            href.append(dd).append("d");
            href.append(hh).append(":");
            href.append(mm);

            anchor = "<a href=\"" + href + "\">" + anchor + "</a>";
            matcher.appendReplacement(newContent, anchor);
        }
        matcher.appendTail(newContent);

        div.append("<div class=\"");
        div.append(mesClass);
        div.append("\">");
        div.append(newContent);
        div.append("</div>");

        return div;
    }

    /**
     * 内部ブラウザ向けに3種の時刻表記をまとめてHTML出力する。
     * @return 時刻表記
     */
    // TODO 利用者が選択できるようにしたい
    private CharSequence buildTimeSpans(){
        StringBuilder result = new StringBuilder(" ");

        String simple  = getTime();
        String anchor  = getAnchor();
        long epoch     = getTimeFromID();
        String decoded = dform.format(epoch);

        result.append( HTMLUtils.elemented("span", "time", simple) )
              .append(" ");
        result.append( HTMLUtils.elemented("span", "time", anchor) )
              .append(" ");
        result.append( HTMLUtils.elemented("span", "time", decoded))
              .append(" ");

        return result;
    }

    /**
     * 内部ブラウザ用のHTML文字列を生成する。
     * @return 内部ブラウザ用のHTML文字列
     */
    public CharSequence buildJdfHTML(){
        StringBuilder html = new StringBuilder();

        CharSequence aElem = HTMLUtils.elemented("a",
                                                 "name",
                                                 getMessageID(),
                                                 this.avatar.getFullName() );

        CharSequence timeInfo = buildTimeSpans();

        CharSequence imgElem;
        if(getTalkType() == Type.GRAVE){
            // TODO we need special name
            imgElem = HTMLUtils.elemented("img", "avatar", "GRAVE", "");
        }else{
            imgElem = this.avatar.buildJdfHTMLImg();
        }
        imgElem = HTMLUtils.elemented("td", imgElem);

        CharSequence content = buildJdfHTMLDiv();
        content = HTMLUtils.elemented("td", content);

        StringBuilder table = new StringBuilder();
        table = table.append(imgElem).append(content);
        table = HTMLUtils.elemented("tr", table);
        table = HTMLUtils.elemented("table", table);

        StringBuilder msgDiv = new StringBuilder();
        msgDiv = msgDiv.append(aElem).append(timeInfo).append(table);
        msgDiv = HTMLUtils.elemented("div", "message", msgDiv);

        html.append(msgDiv).append("\n<!-- -->\n"); // MEMO バグ回避のため！

        return html;
    }

    /**
     * 会話のString表現を返す。
     * 実体参照やHTMLタグも含まれる。
     * @return 会話のString表現
     */
    @Override
    public String toString(){
        StringBuilder result = new StringBuilder();

        result.append(avatar.getFullName());

        if     (talkType == Type.PUBLIC)   result.append(" says ");
        else if(talkType == Type.PRIVATE)  result.append(" think ");
        else if(talkType == Type.WOLFONLY) result.append(" howl ");
        else if(talkType == Type.GRAVE)    result.append(" groan ");

        result.append(dialog);

        return result.toString();
    }

    // TODO &amp; 以外にどんな実体参照が出現しうるか要調査
    // <br /> を <br></br> にしてるのは MacJRE1.5対策だよ
    // TODO Stringコンストラクタ呼び出しと文字列コピーが多すぎ。要再考。
}
