/*
 * Top frame
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: TopFrameView.java 3 2008-06-11 15:08:13Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.text.DateFormat;
import java.util.Date;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.border.BevelBorder;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;

/**
 * 最上位ビュー。
 * 最初に表示されるフレーム。
 * あらゆるコンポーネント、フレームの親。
 */
@SuppressWarnings("serial")
public class TopFrameView extends JFrame{
    
    private final JSplitPane split;
    private final JTree landSelectView;
    private final JTextArea landInfo;
    private final JScrollPane llss;
    private final JTextField statusBar;
    private final JProgressBar progressBar;
    private final JButton reloadButton;
    private final TabBrowser tabBrowser;
    private final JPanel browsePanel;
    private final JComboBox findBox;
    private final JButton searchButton;
    
    /**
     * トップフレームを生成する。
     */
    public TopFrameView(){
        super();
        setFrameTitle(null);

        Border border;

        this.landSelectView = new JTree();
        this.landSelectView.setRootVisible(false);
        border = new EmptyBorder(5,5,5,5);
        this.landSelectView.setBorder(border);
        JScrollPane scrollVList = new JScrollPane(this.landSelectView);

        JLabel label = new JLabel("村を選択してください");
        label.setHorizontalAlignment(SwingConstants.CENTER);
        label.setVerticalAlignment(SwingConstants.CENTER);

        this.split = new JSplitPane();
        this.split.setLeftComponent(scrollVList);
        this.split.setRightComponent(label);
        this.split.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
        this.split.setContinuousLayout(false);
        this.split.setOneTouchExpandable(true);
        this.split.setDividerLocation(200);

        this.statusBar = new JTextField();
        this.statusBar.setText(Jindolf.title + " を使ってくれてありがとう !");
        this.statusBar.setEditable(false);
        border = new BevelBorder(BevelBorder.LOWERED);
        border = new CompoundBorder(border, new EmptyBorder(2,5,2,2));
        this.statusBar.setBorder(border);

        this.progressBar = new JProgressBar();
        this.progressBar.setIndeterminate(false);
        this.progressBar.setOrientation(SwingConstants.VERTICAL);
        
        Container content = this.getContentPane();
        LayoutManager layout = new BorderLayout();
        content.setLayout(layout);
        content.add(this.split, BorderLayout.CENTER);
        content.add(this.statusBar, BorderLayout.SOUTH);
        content.add(this.progressBar, BorderLayout.EAST);
        
        this.landInfo = new JTextArea();
        this.llss = new JScrollPane(this.landInfo);
        border = new EmptyBorder(5,5,5,5);
        this.landInfo.setBorder(border);

        this.reloadButton = new JButton("Reload");
        this.reloadButton.setActionCommand(MenuManager.COMMAND_RELOAD);
        this.findBox = new JComboBox();
        this.findBox.setEditable(true);
        this.findBox.addItem("");
        this.findBox.addItem("【[^】]*】");
        this.findBox.addItem("[■●▼]");
        this.searchButton = new JButton("Search");
        this.searchButton.setActionCommand(MenuManager.COMMAND_SEARCH);
        
        JComponent buttons = designButtons();
        
        this.tabBrowser = new TabBrowser();
        this.browsePanel = new JPanel();
        layout = new BorderLayout();
        this.browsePanel.setLayout(layout);
        this.browsePanel.add(buttons, BorderLayout.NORTH);
        this.browsePanel.add(tabBrowser, BorderLayout.CENTER);
        
        return;
    }

    /**
     * ボタン群のデザインを行う。
     * @return ボタンの集合パネル
     */
    private JComponent designButtons(){
        JPanel panel = new JPanel();
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints cst = new GridBagConstraints();
        panel.setLayout(layout);

        cst.insets = new Insets(1,1,1,1);
        cst.fill = GridBagConstraints.NONE;
        
        cst.anchor = GridBagConstraints.WEST;
        panel.add(this.reloadButton, cst);
        
        cst.anchor = GridBagConstraints.EAST;
        cst.weightx = 1.0;
        panel.add(this.findBox, cst);

        cst.weightx = 0.0;
        panel.add(this.searchButton, cst);

        return panel;
    }

    /**
     * 国村選択ツリービューを返す。
     * @return 国村選択ツリービュー
     */
    public JTree getLandTree(){
        return this.landSelectView;
    }
    
    /**
     * タブビューを返す。
     * @return タブビュー
     */
    public TabBrowser getTabBrowser(){
        return this.tabBrowser;
    }
    
    /**
     * リロードボタンを返す。
     * @return リロードボタン
     */
    public JButton getReloadButton(){
        return this.reloadButton;
    }

    /**
     * 検索コンボボックスを返す。
     * @return 検索コンボボックス
     */
    public JComboBox getFindBox(){
        return this.findBox;
    }
    
    /**
     * 検索ボタンを返す。
     * @return 検索ボタン
     */
    // TODO 本当に必要?ボタン群にリスナー登録できればそれで十分じゃない？
    public JButton getSearchButton(){
        return this.searchButton;
    }
    
    /**
     * 現在タブ選択中の内部ブラウザを返す。
     * @return 内部ブラウザ
     */
    // TODO なくても支障なし
    public JdfBrowser currentJdfBrowser(){
        JdfBrowser result = this.tabBrowser.currentJdfBrowser();
        return result;
    }
    
    /**
     * 現在選択中の村を返す。
     * @return 村
     */
    // TODO なくても支障なし
    public Village currentVillage(){
        return this.tabBrowser.getVillage();
    }
    
    /**
     * フレームタイトルを設定する。
     * タイトルは指定された村名 + " - Jindolf"
     * @param villageName 村の名前
     */
    public void setFrameTitle(String villageName){
        String title = Jindolf.title;
        
        if(villageName != null && villageName.length() > 0){
            title = villageName + " - " + title;
        }
        
        setTitle(title);
        
        return;
    }
    
    /**
     * ステータスバーとプログレスバーとカーソルの設定を行う。
     * @param message ステータスバーへ表示する文字列
     * @param busy trueならプログレスバーのアニメ開始&WAITカーソル。
     *              falseなら停止&通常カーソル。
     */
    public void setBusy(String message, boolean busy){
        updateStatusBar(message);
        if(busy){
            setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
            progressBar.setIndeterminate(true);
        }else{
            setCursor(Cursor.getDefaultCursor());
            progressBar.setIndeterminate(false);
        }
        return;
    }
    
    /**
     * ステータスバーの更新
     * @param message 更新文字列
     */
    protected void updateStatusBar(String message){
        if(message == null) return;
        if(message.length() <= 0) message = " ";
        statusBar.setText(message);
        return;
    }

    /**
     * 村情報を表示する。
     * @param village 村
     */
    public void showVillageInfo(Village village){
        setFrameTitle(village.getVillageName());
        
        this.tabBrowser.setVillage(village);

        int loc = this.split.getDividerLocation();
        this.split.setRightComponent(this.browsePanel);
        this.split.setDividerLocation(loc);

        return;
    }

    /**
     * 国情報を表示する。
     * @param land 国
     */
    public void showLandInfo(Land land){
        setFrameTitle(land.getLandName());
        
        StringBuilder info = new StringBuilder();

        info.append("国名 :\t")
            .append(land.getLandName()).append('\n');
        info.append("識別名 :\t")
            .append(land.getLandIdentifier()).append('\n');
        info.append("ベースURL :\t")
            .append(land.getServerAccess().getBaseURL()).append('\n');

        DateFormat dform = DateFormat.getDateTimeInstance(DateFormat.FULL,
                                                          DateFormat.FULL);
        Date start = land.getStartDate();
        Date end = land.getEndDate();
        info.append("建国 :\t").append(dform.format(start)).append('\n');
        info.append("亡国 :\t").append(dform.format(end)).append('\n');

        info.append("状態 :\t");
        switch(land.getType()){
        case CLOSED:
            info.append("サービス終了");
            break;
        case HISTORICAL:
            info.append("過去ログ提供のみ");
            break;
        case ACTIVE:
            info.append("稼動中");
            break;
        default:
            break;
        }
        info.append('\n');
        info.append("ロケール :\t").append(land.getLocale()).append('\n');
        info.append("時間帯 :\t")
            .append(land.getTimeZone().getDisplayName())
            .append('\n');
        info.append("連絡先 :\t").append(land.getContactInfo()).append('\n');
        info.append("説明 :\t").append(land.getDescription()).append('\n');

        this.landInfo.setText(info.toString());

        int loc = this.split.getDividerLocation();
        this.split.setRightComponent(this.llss);
        this.split.setDividerLocation(loc);

        return;
    }
    
    // TODO setEnabled()を全子フレームにも波及させるべきか
}
