/*
 * Village
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Village.java 3 2008-06-11 15:08:13Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Image;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * いわゆる「村」
 */
public class Village implements Comparable<Village>{

    /**
     * 村同士を比較するためのComparator
     */
    private static class VillageComparator implements Comparator<Village>{
        /**
         * コレクションのために村同士を比較する。
         * @param v1 比較元の村
         * @param v2 比較先の村
         * @return 正負による順序関係
         */
        public int compare(Village v1, Village v2){
            if     ( v1 == null && v2 == null ) return 0;
            else if( v1 == null && v2 != null ) return -1;
            else if( v1 != null && v2 == null ) return +1;

            String v1id = v1.getVillageID();
            String v2id = v2.getVillageID();
            if( v1id.equals(v2id) ) return 0;

            boolean v1idIsNum = true;
            boolean v2idIsNum = true;
            int v1Num = -1;
            int v2Num = -1;
            try{
                v1Num = Integer.parseInt(v1id);
            }catch(NumberFormatException e){
                v1idIsNum = false;
            }
            try{
                v2Num = Integer.parseInt(v2id);
            }catch(NumberFormatException e){
                v2idIsNum = false;
            }

            if     ( ! v1idIsNum && ! v2idIsNum ) return v1id.compareTo(v2id);
            else if(   v1idIsNum && ! v2idIsNum ) return -1;
            else if( ! v1idIsNum &&   v2idIsNum ) return +1;

            return v1Num - v2Num;
        }
    }

    /**
     * 村の状態。
     * 滅んだ国の村は全てGAMEOVERとする。
     */
    public static enum State{
        /** プロローグ進行中 */
        PROLOGUE,
        /** ゲーム進行中 */
        PROGRESS,
        /** エピローグ進行中 */
        EPILOGUE,
        /** ゲーム終了 */
        GAMEOVER,
        /** 状況不明だが村の存在は確認 */
        UNKNOWN
    }

    private static final Comparator<Village> vComparator
            = new VillageComparator();

    private static final Pattern anchorRegex;
    private static final Pattern titleRegex;
    private static final Pattern villageInfoRegex;

    static{
        String Spchar = "\u0020\\t\\n\\r";
        String Sp = "[" +Spchar+ "]";
        String Sp_n = Sp + "*";
        String Sp_m = Sp + "+";

        String prologue = "プロローグ";
        String epilogue = "エピローグ";
        String gameover = "終了";
        String progress = "([1-9][0-9]*)日目";
        String anchorStart = "<a" +Sp_m+ "href=\"([^\"]*)\"" +Sp_n+ ">" +Sp_n;
        String anchorClose = Sp_n+ "</a>";
        String content = "("+
                            prologue +
                         "|"+
                            epilogue +
                         "|"+
                            progress +
                         "|"+
                            gameover +
                         ")";
        String anchor = anchorStart + content + anchorClose;
        String title = "<title>" +Sp_n+ "([^<]*)" +Sp_n+ "</title>";
        String villageInfo =
                ">" +Sp_n+ "([^<]+?)" +Sp_n+
                "<strong>" +Sp_n+
                    "[\\(（]" +Sp_n+
                    "([1-9][0-9]?)/([1-9][0-9]?)" +Sp_n+
                    "(午前|午後)" +Sp_n+
                    "([0-9][0-9]?)時" +Sp_n+
                    "([0-9][0-9]?)分" +Sp_n+
                    "に更新" +Sp_n+
                    "[\\)）]" +Sp_n+
                "</strong>";

        anchorRegex      = Pattern.compile(anchor,      Pattern.DOTALL);
        titleRegex       = Pattern.compile(title,       Pattern.DOTALL);
        villageInfoRegex = Pattern.compile(villageInfo, Pattern.DOTALL);
    }

    /**
     * 村同士を比較するためのComparatorを返す。
     * @return Comparatorインスタンス
     */
    public static Comparator<Village> comparator(){
        return vComparator;
    }

    private final Land parentLand;
    private final String villageID;
    private String villageName;
    private State state = State.UNKNOWN;
    private LinkedList<Period> periodList = new LinkedList<Period>();
    private List<Period> unmodList = Collections.unmodifiableList(periodList);
    private Image tombImage = null;

    private Map<String, Avatar> avatarMap = new HashMap<String,Avatar>();
    private Map<Avatar, Image> faceImageMap = new HashMap<Avatar, Image>();

    /**
     * Villageを生成する。
     * @param parentLand Villageの所属する国
     * @param villageID 村のID
     * @param villageName 村の名前
     */
    public Village(Land parentLand, String villageID, String villageName) {
        this.parentLand    = parentLand;
        this.villageID   = villageID;
        this.villageName = villageName;

        return;
    }

    /**
     * 所属する国を返す。
     * @return 村の所属する国（Land）
     */
    public Land getParentLand(){
        return this.parentLand;
    }

    /**
     * 村のID文字列を返す。
     * @return 村ID
     */
    public String getVillageID(){
        return this.villageID;
    }

    /**
     * 村の名前を返す。
     * @return 村の名前
     */
    public String getVillageName(){
        return this.villageName;
    }

    /**
     * 村の状態を返す。
     * @return 村の状態
     */
    public State getState(){
        return this.state;
    }

    /**
     * 村の状態を設定する。
     * @param state 村の状態
     */
    public void setState(State state){
        this.state = state;
        return;
    }

    /**
     * プロローグを返す。
     * @return プロローグ
     */
    public Period getPrologue(){
        for(Period period : this.periodList){
            if(period.isPrologue()) return period;
        }
        return null;
    }

    /**
     * エピローグを返す。
     * @return エピローグ
     */
    public Period getEpilogue(){
        for(Period period : this.periodList){
            if(period.isEpilogue()) return period;
        }
        return null;
    }

    /**
     * 指定された日付の進行日を返す。
     * @param day 日付
     * @return Period
     */
    public Period getProgress(int day){
        for(Period period : this.periodList){
            if(   period.isProgress()
               && period.getDay() == day ) return period;
        }
        return null;
    }

    /**
     * 指定されたPeriodインデックスのPeriodを返す。
     * プロローグやエピローグへのアクセスも可能。
     * @param day Periodインデックス
     * @return Period
     */
    public Period getPeriod(int day){
        return this.periodList.get(day);
    }

    /**
     * エピローグでない最後の日付を返す。
     * @return 最後の日付。プロローグしかない場合は負の値。
     */
    public int getMaxProgressDay(){
        int maxday = -1;

        for(Period period : this.periodList){
            if( period.isProgress() ) continue;
            int day = period.getDay();
            if(maxday < day) maxday = day;
        }

        return maxday;
    }

    /**
     * Period総数を返す。
     * @return Period総数
     */
    public int getPeriodSize(){
        return this.periodList.size();
    }

    /**
     * Periodへのリストを返す。
     * @return Periodのリスト。
     */
    public List<Period> getPeriodList(){
        return this.unmodList;
    }

    /**
     * 指定した名前で村に登録されているAvatarを返す。
     * @param fullName Avatarの名前
     * @return Avatar
     */
    public Avatar getAvatar(String fullName){
        Avatar avatar;

        avatar = Avatar.getPredefinedAvatar(fullName);
        if( avatar != null ){
            return avatar;
        }

        avatar = this.avatarMap.get(fullName);
        if( avatar != null ){
            return avatar;
        }

        return null;
    }

    /**
     * Avatarを村に登録する。
     * @param avatar Avatar
     */
    // 未知のAvatar出現時の処理が不完全
    public void addAvatar(Avatar avatar){
        if(avatar == null) return;
        String fullName = avatar.getFullName();
        this.avatarMap.put(fullName, avatar);
        return;
    }

    /**
     * Avatarの顔イメージを村に登録する。
     * @param avatar Avatar
     * @param faceImage 顔イメージ
     */
    // デカキャラ対応
    public void putAvatarFaceImage(Avatar avatar, Image faceImage){
        this.faceImageMap.put(avatar, faceImage);
        return;
    }

    /**
     * 村に登録されたAvatarの顔イメージを返す。
     * @param avatar Avatar
     * @return 顔イメージ
     */
    public Image getAvatarFaceImage(Avatar avatar){
        return this.faceImageMap.get(avatar);
    }

    /**
     * 村に登録された墓イメージを返す。
     * @return 墓イメージ
     */
    public Image getGraveImage(){
        return this.tombImage;
    }

    /**
     * 村に墓イメージを登録する。
     * @param image 墓イメージ
     * @throws java.lang.NullPointerException 引数がnullだった場合。
     */
    public void setGraveImage(Image image) throws NullPointerException{
        if(image == null) throw new NullPointerException();
        this.tombImage = image;
    }

    /**
     * 村にアクセスするためのCGIクエリーを返す。
     * @return CGIクエリー
     */
    public CharSequence getCGIQuery(){
        StringBuilder result = new StringBuilder();
        result.append("?vid=").append(getVillageID());
        return result;
    }

    /**
     * 人狼BBSサーバからPeriod一覧情報が含まれたHTMLを取得し、
     * Periodリストを更新する。
     */
    public void updatePeriodList(){
        CharSequence html = null;
        Land land = getParentLand();
        ServerAccess server = land.getServerAccess();
        try{
            if(land.getType() == Land.Type.ACTIVE){
                html = server.getHTMLBoneHead(this);
            }else{
                html = server.getHTMLVillage(this);
            }
        }catch(IOException e){
            System.out.println("error!");
            // もうすこしちゃんと...
            return;
        }

        parseTitle(html);
        updatePeriodList(html);

        return;
    }

    /**
     * 与えられたHTMLのリンク情報を抽出してPeriodリストを更新する。
     * まだPeriodデータのロードは行われない。
     * ゲーム進行中の村で更新時刻をまたいで更新が行われた場合、
     * 既存のPeriodリストが伸張する場合がある。
     * @param html HTMLデータ
     */
    private void updatePeriodList(CharSequence html){
        boolean hasPrologue = false;
        boolean hasEpilogue = false;
        boolean hasProgress = false;
        boolean hasDone = false;
        int maxProgress = 0;

        Matcher matcher = anchorRegex.matcher(html);
        for(;;){
            if( ! matcher.find() ) break;
            String hrefValue  = matcher.group(1);
            String anchorText = matcher.group(2);
            String dayString  = matcher.group(3);

            if( anchorText.equals("プロローグ") ){
                hasPrologue = true;
            }else if( anchorText.endsWith("日目") ){
                hasProgress = true;
                int day = Integer.parseInt(dayString);
                maxProgress = day;
            }else if( anchorText.equals("エピローグ") ){
                hasEpilogue = true;
            }else if( anchorText.equals("終了") ){
                hasDone = true;
                break;
            }else{
                continue;
            }
        }

        if(getParentLand().getType() != Land.Type.ACTIVE){
            setState(State.GAMEOVER);
        }else if(hasDone){
            setState(State.GAMEOVER);
        }else if(hasEpilogue){
            setState(State.EPILOGUE);
        }else if(hasProgress){
            setState(State.PROGRESS);
        }else if(hasPrologue){
            setState(State.PROLOGUE);
        }else{
            setState(State.UNKNOWN);
            this.periodList.clear();
            assert false;
            return;
        }

        Period lastPeriod = null;
        if(hasPrologue){
            Period prologue = getPrologue();
            if(prologue == null){
                lastPeriod = new Period(this, Period.Type.PROLOGUE, 0);
                setPeriod(0, lastPeriod);
            }else{
                lastPeriod = prologue;
            }
        }
        if(hasProgress){
            for(int day=1; day<=maxProgress; day++){
                Period progress = getProgress(day);
                if(progress == null){
                    lastPeriod = new Period(this, Period.Type.PROGRESS, day);
                    setPeriod(day, lastPeriod);
                }else{
                    lastPeriod = progress;
                }
            }
        }
        if(hasEpilogue){
            Period epilogue = getEpilogue();
            if(epilogue == null){
                lastPeriod = new Period(this,
                                        Period.Type.EPILOGUE,
                                        maxProgress +1);
                setPeriod(maxProgress +1, lastPeriod);
            }else{
                lastPeriod = epilogue;
            }
        }

        assert getPeriodSize() > 0;
        assert lastPeriod != null;

        // 念のためチョップ。
        // リロードで村が縮むわけないじゃん。みんな大げさだなあ
        while(this.periodList.getLast() != lastPeriod){
            this.periodList.removeLast();
        }

        if(getState() != State.GAMEOVER){
            lastPeriod.setHot(true);
        }

        return;
    }

    /**
     * Periodリストの指定したインデックスにPeriodを上書きする。
     * リストのサイズと同じインデックスを指定する事が許される。
     * その場合の動作はList.addと同じ。
     * @param index Periodリストのインデックス。
     * @param period 上書きするPeriod
     * @throws java.lang.IndexOutOfBoundsException インデックスの指定がおかしい
     */
    private void setPeriod(int index, Period period)
            throws IndexOutOfBoundsException{
        int listSize = this.periodList.size();
        if(index == listSize){
            this.periodList.add(period);
        }else if(index < listSize){
            this.periodList.set(index, period);
        }else{
            throw new IndexOutOfBoundsException();
        }
        return;
    }

    /**
     * 村タイトルと更新時刻のパースを行う。
     * 現時点では何も反映されない。
     * @param html HTMLデータ。
     */
    private void parseTitle(CharSequence html){
        Matcher matcher;

        matcher = titleRegex.matcher(html);
        if( matcher.find() ){
            String title = matcher.group(1);
        }

        int upMonth;
        int upDay;
        int upHour;
        int upMinute;
        matcher = villageInfoRegex.matcher(html);
        if( matcher.find() ){
            String vName = matcher.group(1);
            String month = matcher.group(2);
            String day   = matcher.group(3);
            String ampm  = matcher.group(4);
            String hh    = matcher.group(5);
            String mm    = matcher.group(6);

            // TODO どこかへ格納が必要
            upMonth  = Integer.parseInt(month);
            upDay    = Integer.parseInt(day);
            upHour   = Integer.parseInt(hh);
            upMinute = Integer.parseInt(mm);
            // TODO 「午後12時」は存在するか？たぶんない。
            if( ampm.equals("午後") ) upHour += 12;
            upHour = upHour % 24;     // 24時 = 0時
        }

        return;
    }

    /**
     * 二つの村を順序付ける。
     * @param village 比較する村
     * @return thisの方が小さければ負、thisの方が大きければ正、どちらでもなければ0
     */
    @Override
    public int compareTo(Village village){
        int cmpResult = vComparator.compare(this, village);
        return cmpResult;
    }

    /**
     * 二つの村が等しいか調べる。
     * @param obj 比較対象の村
     * @return 等しいときのみtrue
     */
    @Override
    public boolean equals(Object obj){
        if(obj == null) return false;
        if( ! (obj instanceof Village) ) return false;
        Village village = (Village) obj;

        if( this.getParentLand() != village.getParentLand() ) return false;

        int cmpResult = compareTo(village);
        if(cmpResult == 0) return true;
        return false;
    }

    /**
     * ハッシュコードを返す。
     * @return ハッシュコード
     */
    @Override
    public int hashCode(){
        int homeHash = getParentLand().hashCode();
        int vidHash = getVillageID().hashCode();
        int result = homeHash ^ vidHash;
        return result;
    }

    /**
     * 村の文字列表現を返す。
     * 村の名前と等しい。
     * @return 村の名前
     */
    @Override
    public String toString(){
        return getVillageName();
    }
}
