/*
 * Jindolf main class
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Jindolf.java 27 2008-06-20 13:10:58Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Dimension;
import java.awt.GraphicsEnvironment;
import java.awt.Toolkit;
import java.awt.Window;
import java.util.Date;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JWindow;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

/**
 * Jindolf スタートアップクラス。
 * 
 * コンストラクタは無いよ。
 * アプリ開始はstaticメソッド{@link #main(String[])}呼び出しから。
 */
public class Jindolf{
    
    public static final String version = "2.1.8";

    public static final String license = "The MIT License";
    public static final String contact = "http://jindolf.sourceforge.jp/";
    
    public static final Date initDate;
    public static final long initNano;
    
    public static final ClassLoader loader;
    public static final Class selfClass;
    public static final Package selfPackage;
    public static final Package jrePackage;
    
    public static final Runtime runtime;
    public static SecurityManager secManager;
    
    public static final String title;
    public static final String author;
    public static final String copyright;

    private static Dimension initGeometry = new Dimension(800, 600);
    
    static{
        initDate = new Date();
        initNano = System.nanoTime();

        selfClass = Jindolf.class;
        selfPackage = selfClass.getPackage();
        jrePackage = java.lang.Object.class.getPackage();

        ClassLoader thisLoader;
        try{
            thisLoader = selfClass.getClassLoader();
        }catch(SecurityException e){
            thisLoader = null;
        }
        loader = thisLoader;

        runtime = Runtime.getRuntime();
        secManager = System.getSecurityManager();

        String implTitle = selfPackage.getImplementationTitle();
        if(implTitle != null){
            title = implTitle;
        }else{
            title = "Jindolf";
        }

        String implVersion = selfPackage.getImplementationVersion();
        if(implVersion != null){
            if( ! version.equals(implVersion) ){
                throw new ExceptionInInitializerError("unmatch package version");
            }
        }

        String implVendor = selfPackage.getImplementationVendor();
        if(implVendor != null){
            author = implVendor;
        }else{
            author = "olyutorskii";
        }

        copyright = "Copyright(c) 2008 " + author;
    }

    /**
     * 任意のクラス群に対して一括ロード／初期化を単一スレッドで順に行う。
     * どーしてもクラス初期化の順序に依存する障害が発生する場合や
     * クラス初期化のオーバーヘッドでGUIの操作性が損なわれるときなどにどうぞ。
     * 
     * @throws java.lang.LinkageError
     * @throws java.lang.ExceptionInInitializerError
     */
    private static void preLoadClass()
            throws LinkageError, ExceptionInInitializerError{
        String[] classes = {
            "java.lang.Object",
            "jp.sourceforge.jindolf.JdfEditorKit",
            "jp.sourceforge.jindolf.TabBrowser",
            "jp.sourceforge.jindolf.TopFrameView",
            "java.net.HttpURLConnection",
            "java.text.SimpleDateFormat",
            "javax.swing.JEditorPane",
            "javax.swing.text.html.CSS",
            "javax.swing.text.html.StyleSheet",
            "javax.swing.text.html.HTMLDocument",
            "javax.swing.text.html.BlockView",
            "javax.swing.text.html.ParagraphView",
            "javax.swing.text.html.InlineView",
            "javax.swing.text.html.parser.DocumentParser",
        };

        for(String className : classes){
            try{
                if(loader != null){
                    Class.forName(className, true, loader);
                }else{
                    Class.forName(className);
                }
            }catch(ClassNotFoundException e){
                warnMessage(e);
                continue;
            }
        }

        return;
    }

    /**
     * JRE1.5相当のランタイムライブラリ環境が提供されているか判定する。
     * 
     * @return JRE1.5相当ならtrueを返す。
     */
    public static boolean hasRuntime5(){
        boolean result;
        try{
            result = jrePackage.isCompatibleWith("1.5");
        }catch(NumberFormatException e){
            return false;
        }
        return result;
    }

    /**
     * JRE1.6相当のランタイムライブラリ環境が提供されているか判定する。
     * 
     * @return JRE1.6相当ならtrueを返す。
     */
    public static boolean hasRuntime6(){
        boolean result;
        try{
            result = jrePackage.isCompatibleWith("1.6");
        }catch(NumberFormatException e){
            return false;
        }
        return result;
    }

    /**
     * VMごとプログラムを終了する。
     * 
     * @param exitCode 終了コード
     * @throws java.lang.SecurityException
     */
    // TODO トップフレームのクローズボタン押下を捕獲しとかなきゃ…
    static void exit(int exitCode) throws SecurityException{
        System.out.flush();
        System.err.flush();
        runtime.runFinalization();
        runtime.gc();
        Thread.yield();
        try{
            runtime.exit(exitCode);
        }catch(SecurityException e){
            warnMessage(e);
            throw e;
        }
        return;
    }

    /**
     * Jindolf の実行が可能な環境でなければ、即時VM実行を終了する。
     */
    private static void testEnvironment(){
        if(!hasRuntime5()){
            errorMessage(
                    Jindolf.title
                    + " need J2SE5 platform or later.");
            exit(1);
        }

        if(GraphicsEnvironment.isHeadless()){
            errorMessage(
                    Jindolf.title
                    + " need bit-map display, keyboard, & pointing-device.");
            exit(1);
        }

        if(   '狼' != 0x72fc
           || '　' != 0x3000
           || '~'  != 0x007e
           || '\\' != 0x005c
           || '～' != 0xff5e ){
            errorMessage(
                    "ソースコードの文字コードが正しくコンパイルされていないかも。"
                   +"あなたは今、オリジナル開発元の意図しない文字コード環境で"
                   +"コンパイルされたプログラムを起動しようとしているよ。"
                   +"ソースコードの入手に際して"
                   +"どのような文字コード変換が行われたか認識している？"
                   +"コンパイルオプションで正しい文字コードを指定したかな？"
                    );
            exit(1);
        }
        
        return;
    }

    /**
     * Jindolf のスタートアップエントリ
     *  
     * @param args コマンドライン引数
     */
    public static void main(final String[] args){
        testEnvironment();
        
        // Metal L&F でもの凄く日本語が汚くなるのを強制回避
        UIManager.put("swing.boldMetal", Boolean.FALSE);

        SwingUtilities.invokeLater(
                new Runnable(){
                    public void run(){
                        startGUI(args);
                    }
                } );

        Thread.yield();
        
        return;
    }

    /**
     * AWTイベントディスパッチスレッド版スタートアップエントリ
     * 
     * @param args コマンドライン引数
     */
    // TODO コマンドライン解析
    private static void startGUI(String[] args){
        Window splashWindow;
        if(!hasRuntime6()){
            splashWindow = createSplashWindow();
            splashWindow.setVisible(true);
            Thread.yield();
        }else{
            splashWindow = null;
        }

        preLoadClass();

        LandsModel model = new LandsModel();
        model.loadLandList();
        TopFrameView topFrame = buildMVC(model);

        topFrame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        topFrame.setResizable(true);
        topFrame.setIconImage(GUIUtils.getWindowIconImage());
        topFrame.pack();
        topFrame.setSize(initGeometry);
        topFrame.setLocationByPlatform(true);
        Toolkit kit = topFrame.getToolkit();
        kit.setDynamicLayout(false);

        if(splashWindow != null){
            splashWindow.setVisible(false);
            splashWindow.dispose();
            splashWindow = null;
        }

        topFrame.setVisible(true);

        Thread.yield();
        
        return;
    }

    /**
     * スプラッシュウィンドウを作成する。
     * 
     * @return 未表示のスプラッシュウィンドウ。
     */
    private static Window createSplashWindow(){
        JWindow splashWindow = new JWindow();
        JLabel splashLabel = new JLabel(GUIUtils.getLogoIcon());
        splashWindow.add(splashLabel);

        splashWindow.pack();
        splashWindow.setLocationRelativeTo(null); // locate center

        return splashWindow;
    }

    /**
     * モデル・ビュー・コントローラの結合?
     * 
     * @param model 最上位のデータモデル
     * @return 最上位ビュー画面
     */
    private static TopFrameView buildMVC(LandsModel model){
        MenuManager menuManager = new MenuManager();
        TopFrameView topFrame = new TopFrameView();
        topFrame.setJMenuBar(menuManager.createMenuBar());
        new Controller(menuManager, topFrame, model);
        return topFrame;
    }

    /**
     * 警告を出力する。
     * 
     * @param message 警告メッセージ
     */
    private static void warnMessage(Object message){
        System.out.println(message);
        return;
    }

    /**
     * エラーを出力する。
     * 
     * @param message エラーメッセージ
     */
    private static void errorMessage(Object message){
        System.err.println(message);
        return;
    }

    /**
     * hidden default constructor
     */
    private Jindolf(){
        super();
        return;
    }
}
