/*
 * menu manager
 * 
 * Copyright(c) 2008 olyutorskii
 * $Id: MenuManager.java 18 2008-06-15 13:00:34Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.event.ActionListener;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.LookAndFeel;
import javax.swing.UIManager;

/**
 * メニュー、ボタン、その他各種Actionを伴うイベントを生成する
 * コンポーネントの一括管理
 */
public class MenuManager{
    
    public static final String COMMAND_EXIT  = "EXIT";
    public static final String COMMAND_ABOUT = "ABOUT";
    public static final String COMMAND_LANDF = "LANDF";
    public static final String COMMAND_HELPDOC = "HELPDOC";
    public static final String COMMAND_FILTER = "FILTER";
    public static final String COMMAND_ACCOUNT = "ACCOUNT";
    public static final String COMMAND_RELOAD = "RELOAD";
    public static final String COMMAND_SEARCH = "SEARCH";
    
    private Set<AbstractButton> items = new HashSet<AbstractButton>();

    private JMenuItem itemExit;
    private JMenuItem itemAbout;
    private JMenuItem itemHelpDoc;
    private JMenuItem itemFilter;
    private JMenuItem itemAccount;

    private ButtonGroup landfGroup = new ButtonGroup();
    private Map<ButtonModel, String> landfMap
        = new HashMap<ButtonModel, String>();
    
    public MenuManager(){
        itemExit  = new JMenuItem("Exit");
        itemAbout = new JMenuItem("About");
        itemHelpDoc = new JMenuItem("Help document");
        itemFilter = new JMenuItem("Show Filter");
        itemAccount = new JMenuItem("Manage Account");

        itemExit .setActionCommand(COMMAND_EXIT);
        itemAbout.setActionCommand(COMMAND_ABOUT);
        itemHelpDoc.setActionCommand(COMMAND_HELPDOC);
        itemFilter.setActionCommand(COMMAND_FILTER);
        itemAccount.setActionCommand(COMMAND_ACCOUNT);
    
        itemExit    .setMnemonic(java.awt.event.KeyEvent.VK_X);
        itemAbout   .setMnemonic(java.awt.event.KeyEvent.VK_A);
        itemHelpDoc .setMnemonic(java.awt.event.KeyEvent.VK_H);
        itemFilter  .setMnemonic(java.awt.event.KeyEvent.VK_F);
        itemAccount .setMnemonic(java.awt.event.KeyEvent.VK_M);

        items.add(itemExit);
        items.add(itemAbout);
        items.add(itemHelpDoc);
        items.add(itemFilter);
        items.add(itemAccount);

        createLookAndFeelMenuItems();

        return;
    }
    
    /**
     * L&F 一覧メニューを作成する。
     */
    private void createLookAndFeelMenuItems(){
        LookAndFeel currentLookAndFeel = UIManager.getLookAndFeel();
        String currentName = currentLookAndFeel.getClass().getName();
        AbstractButton matchedButton = null;

        UIManager.LookAndFeelInfo[] landfs;
        landfs = UIManager.getInstalledLookAndFeels();
        for(int ct=0; ct<=landfs.length-1; ct++){
            String name      = landfs[ct].getName();
            String className = landfs[ct].getClassName();

            JRadioButtonMenuItem button = new JRadioButtonMenuItem(name);
            button.setActionCommand(COMMAND_LANDF);

            if(className.equals(currentName)) matchedButton = button;

            items.add(button);
            landfGroup.add(button);
            landfMap.put(button.getModel(), className);
        }

        if(matchedButton != null) matchedButton.setSelected(true);

        return;
    }
    
    /**
     * 現在メニューで選択中のL&Fのクラス名を返す。
     * @return L&F クラス名
     */
    public String getSelectedLookAndFeel(){
        ButtonModel selected = landfGroup.getSelection();
        if(selected == null) return null;
        String className = landfMap.get(selected);
        return className;
    }
    
    /**
     * 全てのボタンにリスナー登録する。
     * @param listener リスナー
     */
    public void addActionListener(ActionListener listener){
        for(AbstractButton button : items){
            button.addActionListener(listener);
        }
        return;
    }
    
    /**
     * メニューバーを生成する。
     * @return メニューバー
     */
    public JMenuBar createMenuBar(){
        JMenuBar bar;

        JMenu menuFile;
        JMenu menuLook;
        JMenu menuWindow;
        JMenu menuHelp;
        
        menuFile = new JMenu("File");
        menuLook = new JMenu("Look & Feel");
        menuWindow = new JMenu("Window");
        menuHelp = new JMenu("Help");

        menuFile.setMnemonic(java.awt.event.KeyEvent.VK_F);
        menuLook.setMnemonic(java.awt.event.KeyEvent.VK_L);
        menuWindow.setMnemonic(java.awt.event.KeyEvent.VK_W);
        menuHelp.setMnemonic(java.awt.event.KeyEvent.VK_H);

        menuFile.add(itemExit);
        menuWindow.add(itemFilter);
        menuWindow.add(itemAccount);
        menuHelp.add(itemHelpDoc);
        menuHelp.addSeparator();
        menuHelp.add(itemAbout);

        Enumeration landfButtons = landfGroup.getElements();
        while(landfButtons.hasMoreElements()){
            JMenuItem item = (JMenuItem)( landfButtons.nextElement() );
            menuLook.add(item);
        }
        
        bar = new JMenuBar();
        bar.add(menuFile);
        bar.add(menuWindow);
        bar.add(menuLook);
        bar.add(menuHelp);
         
        return bar;
    }
}
