/*
 * main browser
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: JdfBrowser.java 26 2008-06-20 12:51:53Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.JEditorPane;
import javax.swing.event.HyperlinkListener;
import javax.swing.plaf.TextUI;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultHighlighter;
import javax.swing.text.Document;
import javax.swing.text.EditorKit;
import javax.swing.text.Highlighter;
import javax.swing.text.View;

/**
 * 独自HTMLベースのメインブラウザ。
 * EditorKitはJdfEditorKitwを利用する。
 */
@SuppressWarnings("serial")
public class JdfBrowser extends JEditorPane{

    private static final Highlighter.HighlightPainter highlightPainter =
            new DefaultHighlighter.DefaultHighlightPainter(new Color(0xd02020));
    
    private Village village;
    private Period period;
    private TopicFilter.FilterContext filterContext;
    private boolean hasShown = false;

    /**
     * 初期Periodを指定してメインブラウザを作成する。
     * @param period 初期Period
     */
    public JdfBrowser(Period period){
        super();

        setEditable(false);
        setContentType("text/html");
        putClientProperty(JEditorPane.W3C_LENGTH_UNITS, Boolean.TRUE);

        setPeriod(period);
        
        return;
    }
    
    /**
     * 現在のPeriodを返す。
     * @return 現在のPeriod
     */
    public Period getPeriod(){
        return this.period;
    }
    
    /**
     * Periodを更新する。
     * 古いPeriodの表示内容は消える。
     * 新しいPeriodの表示内容はまだ反映されない。
     * 既存のHyperlinkListenerは登録を外される。
     * @param period 新しいPeriod
     */
    public void setPeriod(Period period){
        if(period == this.period) return;
        
        this.period = period;
        this.hasShown = false;
        this.filterContext = null;
        setText("");

        if(this.period == null) return;
        
        if(period.getVillage() == this.village) return;
        this.village = period.getVillage();

        EditorKit kit = new JdfEditorKit(this.village);
        setEditorKit(kit);

        HyperlinkListener[] listeners = getHyperlinkListeners();
        for(HyperlinkListener listener : listeners){
            removeHyperlinkListener(listener);
        }
        addHyperlinkListener(new AnchorListener(this.village));

        return;
    }
    
    /**
     * フィルタを適用してPeriodの内容を出力する。
     * @param filter フィルタ
     */
    public void showTopics(TopicFilter filter, boolean force){
        if(force || ! this.hasShown){
            CharSequence html = this.period.buildJdfHTML();
            setCharSequence(html);
            this.hasShown = true;
            this.filterContext = null;
        }
        
        filtering(filter);

        return;
    }
    
    /**
     * 発言フィルタを適用する。
     * @param filter フィルタ
     */
    public void filtering(TopicFilter filter){
        if( filter.isSame(this.filterContext) ) return;

        List<TopicView> viewList = getTopicViewList(null, null);
        for(TopicView view : viewList){
            if(filter.isFiltered(view)){
                view.setVisible(false);
            }else{
                view.setVisible(true);
            }
            
            View parentView = view.getParent();
            if(parentView == null) continue;
            parentView.preferenceChanged(view, true, true);
        }

        this.filterContext = filter.getFilterContext();
        
        return;
    }
    
    /**
     * 指定したHTML文書で表示を更新する。
     * @param html HTML文書
     */
    private void setCharSequence(CharSequence html){
        Reader reader;
        if(html != null) reader = new StringReader(html.toString());
        else             reader = new StringReader("");
        
        EditorKit kit = getEditorKit();
        Document doc = kit.createDefaultDocument();

        try{
            kit.read(reader, doc, 0);
        }catch(IOException e){
            assert false;
        }catch(BadLocationException e){
            assert false;
        }

        setDocument(doc);

        return;
    }

    /**
     * 与えられた正規表現にマッチする文字列をハイライト描画する。
     * @param pattern 正規表現
     */
    public void highlightRegex(Pattern pattern){
        if(pattern == null) return;
        
        Highlighter highlighter = getHighlighter();
        Highlighter.Highlight[] highlights = highlighter.getHighlights();
        for(Highlighter.Highlight highlight : highlights){
            if(highlight.getPainter() != highlightPainter) continue;
            highlighter.removeHighlight(highlight);
        }
        
        Document doc = getDocument();
        int docLength = doc.getLength();
        String text;
        try{
            text = doc.getText(0, docLength);
        }catch(BadLocationException e){
            assert false;
            return;
        }
        
        Matcher matcher = pattern.matcher(text);
        matcher.region(0, docLength);
        while( matcher.find() ){
            int startPos = matcher.start();
            int endPos = matcher.end();
            if(startPos == endPos) break;
            matcher.region(endPos, docLength);
            try{
                highlighter.addHighlight(startPos, endPos, highlightPainter);
            }catch(BadLocationException e){
                assert false;
                return;
            }
        }

        return;
    }
    
    /**
     * 描画品質をカスタマイズするためのフック
     * @param g グラフィックスコンテキスト
     */
    // TODO フキダシの角丸だけでもアンチエイリアス指定しときたいなぁ
    @Override
    public void paint(Graphics g){
        Graphics2D g2 = (Graphics2D) g;
        g2.addRenderingHints(GUIUtils.getDefaultHints());
        super.paint(g);
        return;
    }
    
    /**
     * ルートViewを得る。
     * @return ルートView
     */
    public View getRootView(){
        TextUI textUI = getUI();
        View root = textUI.getRootView(this);
        return root;
    }
    
    /**
     * 全てのTopicViewを与えられた親から再帰的に抽出する。
     * @param parent 親View
     * @param list TopicViewのリスト
     * @return TopicViewのリスト
     */
    public List<TopicView> getTopicViewList(View parent,
                                              List<TopicView> list){
        if(parent == null){
            parent = getRootView();
            list = new LinkedList<TopicView>();
        }
        
        int childNo = parent.getViewCount();
        for(int ct=0; ct <= childNo-1; ct++){
            View child = parent.getView(ct);
            if(child instanceof TopicView){
                list.add( (TopicView) child );
            }else{
                getTopicViewList(child, list);
            }
        }
        
        return list;
    }
}
