/*
 * model of lands for JTree view
 * 
 * Copyright(c) 2008 olyutorskii
 * $Id: LandsModel.java 42 2008-06-25 12:35:34Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;
import javax.swing.event.EventListenerList;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

/**
 * 国の集合。あらゆるデータモデルの大元。
 * 国一覧と村一覧を管理。
 * JTreeのモデルも兼用。
 */
public class LandsModel implements TreeModel{ // ComboBoxModelも付けるか？

    private static final String KEY_PREFIX = "land.order.";
    private static final String ROOT = "ROOT";

    private final List<Land> landList = new Vector<Land>();
    private boolean isLandListLoaded = false;
    
    private final EventListenerList listeners = new EventListenerList();

    /**
     * コンストラクタ。
     * この時点ではまだ国一覧が読み込まれない。
     */
    public LandsModel(){
        return;
    }
    
    /**
     * 指定した国の村一覧を読み込む。
     * @param land
     */
    public void loadVillageList(Land land) throws IOException{
        land.updateVillageList();
        List<Village> villageList = land.getVillageList();

        int[] childIndices = new int[villageList.size()];
        for(int ct=0; ct < childIndices.length; ct++){
            childIndices[ct] = ct;
        }
        Object[] children = villageList.toArray();
        
        Object[] path = {ROOT, land};
        TreePath treePath = new TreePath(path);
        TreeModelEvent event = new TreeModelEvent(this,
                                                  treePath,
                                                  childIndices,
                                                  children     );
        fireTreeNodesInserted(event);

        return;
    }
    
    /**
     * リソース上のプロパティファイルから国一覧を読み込む。
     */
    public void loadLandList() throws IOException{
        if(isLandListLoaded) return;
        
        InputStream is = this.getClass()
                        .getResourceAsStream("resources/land.properties");
        Properties properties = new Properties();

        properties.load(is);
        is.close();
        
        String codeCheck = properties.getProperty("codeCheck");
        if(   codeCheck == null
           || codeCheck.length() != 1
           || codeCheck.charAt(0) != '\u72fc'){  // 「狼」
            Jindolf.logger.severe("国定義一覧プロパティファイルの"
                    +"文字コードがおかしいようです。"
                    +"native2ascii は正しく適用しましたか？");
            Jindolf.exit(1);
            return;
        }
        
        Set<Object> keySet = properties.keySet();
             
        SortedSet<Integer> orderSet = new TreeSet<Integer>();
        for(Object keyObj : keySet){
            if(keyObj == null) continue;
            String key = keyObj.toString();
            if( ! key.startsWith(KEY_PREFIX) ) continue;
            key = key.replace(KEY_PREFIX, "");
            Integer order;
            try{
                order = new Integer(key);
            }catch(NumberFormatException e){
                continue;
            }
            orderSet.add(order);
        }
        
        this.landList.clear();
        for(Integer order : orderSet){
            if(order == null) continue;
            String key = KEY_PREFIX + order.toString();
            String landId = properties.getProperty(key);
            if(landId == null || landId.length() <= 0) continue;

            Land land = new Land(landId, properties);
            this.landList.add(land);
        }
        isLandListLoaded = true;

        int[] childIndices = new int[this.landList.size()];
        for(int ct=0; ct < childIndices.length; ct++){
            childIndices[ct] = ct;
        }
        Object[] children = this.landList.toArray();
            
        TreePath treePath = new TreePath(ROOT);
        TreeModelEvent event = new TreeModelEvent(this,
                                                  treePath,
                                                  childIndices,
                                                  children     );
        fireTreeNodesInserted(event);

        return;
    }

    /**
     * リスナー登録
     * @param l リスナー
     */
    public void addTreeModelListener(TreeModelListener l){
        listeners.add(TreeModelListener.class, l);
        return;
    }

    /**
     * リスナー削除
     * @param l リスナー
     */
    public void removeTreeModelListener(TreeModelListener l){
        listeners.remove(TreeModelListener.class, l);
        return;
    }

    /**
     * 登録中のリスナーのリストを得る。
     * @return リスナーのリスト
     */
    private TreeModelListener[] getTreeModelListeners(){
        return listeners.getListeners(TreeModelListener.class);
    }

    /**
     * 全リスナーにイベントを送出する。
     * @param event ツリーイベント
     */
    protected void fireTreeNodesInserted(TreeModelEvent event){
        for(TreeModelListener listener : getTreeModelListeners()){
            listener.treeNodesInserted(event);
        }
        return;
    }

    /**
     * 親の子を返す。
     * @param parent 親
     * @param index 子のインデックス
     * @return 子
     */
    public Object getChild(Object parent, int index){
        if(index < 0)                      return null;
        if(index >= getChildCount(parent)) return null;

        if(parent == ROOT){
            return getLandList().get(index);
        }
        if(parent instanceof Land){
            Land land = (Land)parent;
            return land.getVillage(index);
        }
        return null;
    }

    /**
     * 子の数を返す。
     * @param parent 親
     * @return 子の数
     */
    public int getChildCount(Object parent){
        if(parent == ROOT){
            return getLandList().size();
        }
        if(parent instanceof Land){
            Land land = (Land)parent;
            return land.getVillageCount();
        }
        return 0;
    }

    /**
     * 親の中で子が占める位置のインデックス値を返す。
     * @param parent 親
     * @param child 子
     * @return インデックス値
     */
    public int getIndexOfChild(Object parent, Object child){
        if(child == null) return -1;
        if(parent == ROOT){
            return getLandList().indexOf(child);
        }
        if(parent instanceof Land){
            Land land = (Land)parent;
            return land.getIndexOfVillage(child);
        }
        return -1;
    }

    /**
     * Treeの根元を返す。
     * @return 根本
     */
    public Object getRoot(){
        return ROOT;
    }

    /**
     * 葉要素か否か判定する。
     * @param node 要素
     * @return 葉ならtrue
     */
    public boolean isLeaf(Object node){
        if(node == ROOT)            return false;
        if(node instanceof Land)   return false;
        if(node instanceof Village) return true;
        return true;
    }

    /**
     * Tree項目が変更されたことを通知。
     * ※ たぶん使わないので必ず失敗させている。
     * @param path Treeパス
     * @param newValue 新しい値
     */
    public void valueForPathChanged(TreePath path, Object newValue){
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * 国リストを得る。
     * @return 国のリスト
     */
    // TODO Collection.unmodifiableList()でラップしたい。
    public List<Land> getLandList(){
        return this.landList;
    }
}
