/*
 * period viewer with tab access
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: TabBrowser.java 18 2008-06-15 13:00:34Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Component;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.JViewport;
import javax.swing.ScrollPaneConstants;
import javax.swing.border.EmptyBorder;

/**
 * タブを用いて村情報と各Periodを閲覧するためのコンポーネント。
 */
@SuppressWarnings("serial")
public class TabBrowser extends JTabbedPane{

    private Village village;
    private final JTextArea villageInfo = new JTextArea();

    /**
     * 村が指定されていない状態のタブパネルを生成する。
     */
    public TabBrowser(){
        super();

        setTabPlacement(JTabbedPane.TOP);
        // Mac Aqua L&F ignore WRAP_TAB_LAYOUT
        setTabLayoutPolicy(JTabbedPane.WRAP_TAB_LAYOUT);

        this.villageInfo.setEditable(false);
        
        addTab("村情報", this.villageInfo);
        
        setVillage(null);
        
        return;
    }

    /**
     * 村情報閲覧用のコンポーネントを更新する。
     */
    private void updateVillageInfo(){
        if(getVillage() == null){
            this.villageInfo.setText("no village");
            return;
        }

        StringBuilder info = new StringBuilder();

        Land land = getVillage().getParentLand();
        Village target = getVillage();
        info.append("国名 :\t").append(land.getLandName()).append('\n');
        info.append("村名 :\t").append(target.getVillageName()).append('\n');
        info.append("村ID :\t").append(target.getVillageID()).append('\n');

        info.append("状態 :\t");
        switch(getVillage().getState()){
        case PROLOGUE:
            info.append("プロローグ中");
            break;
        case PROGRESS:
            info.append("ゲーム進行中");
            break;
        case EPILOGUE:
            info.append("エピローグ中");
            break;
        case GAMEOVER:
            info.append("ゲーム終了");
            break;
        case UNKNOWN:
            info.append("不明");
            break;
        default:
            info.append("???");
            assert false;
            break;
        }
        info.append('\n');

        info.append("所要日数 :\t")
            .append(getVillage().getPeriodSize())
            .append('\n');

        this.villageInfo.setText(info.toString());

        return;
    }

    /**
     * 村情報表示タブを選択表示する。
     */
    public void selectVillageInfoTab(){
        setSelectedIndex(0);
        return;
    }
    
    /**
     * 設定された村を返す。
     * @return 設定された村
     */
    public Village getVillage(){
        return this.village;
    }

    /**
     * 新規に村を設定する。
     * @param village 新しい村
     */
    public void setVillage(Village village){
        if(village == null){
            this.village = null;
            selectVillageInfoTab();
            modifyTabCount(0);
            updateVillageInfo();
            return;
        }else if(village != this.village){
            selectVillageInfoTab();
        }

        this.village = village;

        updateVillageInfo();

        int periodNum = this.village.getPeriodSize();
        modifyTabCount(periodNum);
        
        for(int periodDays=0; periodDays < periodNum; periodDays++){
            Period period = this.village.getPeriod(periodDays);
            int tabIndex = periodDaysToTabIndex(periodDays);
            JdfBrowser browser = getJdfBrowser(tabIndex);
            if(browser == null){
                Component tabComponent = createTabComponent(period);
                setComponentAt(tabIndex, tabComponent);
                browser = getJdfBrowser(tabIndex);
            }
            String caption = period.getCaption();
            setTitleAt(tabIndex, caption);
            if(period == browser.getPeriod()) continue;
            browser.setPeriod(period);
        }

        return;
    }

    /**
     * 指定した数のPeriodが収まるよう必要十分なタブ数を用意する。
     * @param periods Periodの数
     */
    private void modifyTabCount(int periods){ // TODO 0でも大丈夫?
        int maxPeriodDays = periods -1;
        
        for(;;){   // 短ければタブ追加
            int maxTabIndex = getTabCount() -1;
            if(tabIndexToPeriodDays(maxTabIndex) >= maxPeriodDays) break;
            String title = "";
            Component component = new JPanel();
            addTab(title, component);
        }
        
        for(;;){   // 長ければ余分なタブ削除
            int maxTabIndex = getTabCount() -1;
            if(tabIndexToPeriodDays(maxTabIndex) <= maxPeriodDays) break;
            remove(maxTabIndex);
        }
        
        return;
    }
    
    /**
     * Period閲覧用のタブコンポーネントを生成する。
     * 指定されたPeriodを閲覧するJdfBrowseインスタンスを、
     * JscrollPaneでラップしたものが返される。
     * @param period 閲覧対象のPeriod
     * @return JScrollPaneのインスタンス
     */
    private Component createTabComponent(Period period){
        JdfBrowser browser = new JdfBrowser(period);
        browser.setBorder(new EmptyBorder(0, 0, 0, 0));

        JScrollPane scrollPane = new JScrollPane(browser);
        scrollPane.setHorizontalScrollBarPolicy(
                ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER
        );
        return scrollPane;
    }

    /**
     * Period日付指定からタブインデックス値への変換。
     * @param days Period日付指定
     * @return タブインデックス
     */
    private int periodDaysToTabIndex(int days){
        int tabIndex = days+1;
        if(tabIndex >= getTabCount()) return -1;
        return tabIndex;
    }

    /**
     * タブインデックス値からPeriod日付指定への変換。
     * @param tabIndex タブインデックス
     * @return Period日付指定
     */
    private int tabIndexToPeriodDays(int tabIndex){
        if(tabIndex >= getTabCount()) return -1;
        int days = tabIndex-1;
        return days;
    }

    /**
     * 現在タブ選択中のJdfBrowseを返す。
     * Periodに関係ないタブが選択されていたらnullを返す。
     * @return 現在選択中のJdfBrowse
     */
    public JdfBrowser currentJdfBrowser(){
        int tabIndex = getSelectedIndex();
        JdfBrowser result = getJdfBrowser(tabIndex);
        return result;
    }

    /**
     * 指定したタブインデックスに関連付けられたJdfBrowseを返す。
     * Periodに関係ないタブが指定されたらnullを返す。
     * @param tabIndex タブインデックス
     * @return 指定されたJdfBrowse
     */
    private JdfBrowser getJdfBrowser(int tabIndex){
        if(tabIndexToPeriodDays(tabIndex) < 0) return null;
        if(tabIndex >= getTabCount()) return null;

        Component component = getComponentAt(tabIndex);
        if(component == null) return null;
        if( ! (component instanceof JScrollPane) ) return null;

        JScrollPane scrollPane = (JScrollPane) component;
        JViewport viewport = scrollPane.getViewport();
        if(viewport == null) return null;

        Component view = viewport.getView();
        if(view == null) return null;
        if( ! (view instanceof JdfBrowser) ) return null;

        JdfBrowser result = (JdfBrowser) view;

        return result;
    }
}
