/*
 * Filter panel
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: FilterPanel.java 74 2008-07-05 04:15:13Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.BitSet;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.SwingConstants;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;

/**
 * 発言フィルタ GUI
 */
@SuppressWarnings("serial")
public class FilterPanel extends JFrame
        implements ActionListener, TopicFilter{

    private final JCheckBox checkPublic = new JCheckBox("公開", true);
    private final JCheckBox checkWolf = new JCheckBox("狼", true);
    private final JCheckBox checkPrivate = new JCheckBox("独り言", true);
    private final JCheckBox checkGrave = new JCheckBox("墓下", true);
    
    private final JButton selAllButton = new JButton("全選択");
    private final JButton selNoneButton = new JButton("全解除");
    private final JButton negateButton = new JButton("反転");

    private final Map<Avatar, JCheckBox> cbMap =
            new HashMap<Avatar, JCheckBox>();
    private final List<JCheckBox> cbList = new LinkedList<JCheckBox>();

    private EventListenerList listeners;

    /**
     * 発言フィルタを生成する。
     */
    public FilterPanel(){
        super("発言フィルタ - " + Jindolf.title);

        setResizable(true);
        Toolkit kit = getToolkit();
        kit.setDynamicLayout(false);
        setIconImage(GUIUtils.getWindowIconImage());
        setLocationByPlatform(true);

        JComponent topicPanel = createTopicPanel();
        JComponent avatarPanel = createAvatarPanel();
        JComponent buttonPanel = createButtonPanel();
        design(topicPanel, avatarPanel, buttonPanel);

        this.listeners = new EventListenerList();

        this.checkPublic.addActionListener(this);
        this.checkWolf.addActionListener(this);
        this.checkPrivate.addActionListener(this);
        this.checkGrave.addActionListener(this);
        
        for(JCheckBox avatarCheckBox : this.cbList){
            avatarCheckBox.addActionListener(this);
        }
        
        this.selAllButton.addActionListener(this);
        this.selNoneButton.addActionListener(this);
        this.negateButton.addActionListener(this);
        
        return;
    }

    /**
     * レイアウトデザインを行う。
     * @param topicPanel システムイベント選択
     * @param avatarPanel キャラ一覧
     * @param buttonPanel ボタン群
     */
    private void design(JComponent topicPanel,
                         JComponent avatarPanel,
                         JComponent buttonPanel){
        Container content = getContentPane();

        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();

        content.setLayout(layout);

        constraints.weightx = 1.0 / 5;
        constraints.weighty = 1.0;
        constraints.gridheight = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.BOTH;
        constraints.anchor = GridBagConstraints.CENTER;
        content.add(topicPanel, constraints);

        constraints.weightx = 0.0;
        constraints.gridheight = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.VERTICAL;
        content.add(new JSeparator(SwingConstants.VERTICAL), constraints);

        constraints.weightx = 4.0 / 5;
        constraints.weighty = 1.0;
        constraints.gridheight = 1;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.BOTH;
        content.add(avatarPanel, constraints);

        constraints.weighty = 0.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.HORIZONTAL;
        content.add(new JSeparator(), constraints);
        
        constraints.insets = new Insets(5,5,5,5);
        content.add(buttonPanel, constraints);
        
        return;
    }
    
    /**
     * システムイベントチェックボックス群パネルを作成
     * @return システムイベントチェックボックス群パネル
     */
    private JComponent createTopicPanel(){
        JPanel topicPanel = new JPanel();
        
        topicPanel.setBorder(new EmptyBorder(2, 2, 2, 2));
        
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();

        topicPanel.setLayout(layout);

        constraints.anchor = GridBagConstraints.WEST;
        constraints.weightx = 1.0;
        constraints.weighty = 1.0;
        
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        topicPanel.add(checkPublic, constraints);

        constraints.gridwidth = GridBagConstraints.REMAINDER;
        topicPanel.add(checkWolf, constraints);

        constraints.gridwidth = GridBagConstraints.REMAINDER;
        topicPanel.add(checkPrivate, constraints);

        constraints.gridwidth = GridBagConstraints.REMAINDER;
        topicPanel.add(checkGrave, constraints);
        
        return topicPanel;
    }
    
    /**
     * キャラ一覧チェックボックス群パネルを作成。
     * @return キャラ一覧チェックボックス群パネル
     */
    private JComponent createAvatarPanel(){
        final int cols = 4;
        
        JPanel avatarPanel = new JPanel();
        
        avatarPanel.setBorder(new EmptyBorder(2, 2, 2, 2));
        
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        
        avatarPanel.setLayout(layout);

        constraints.weightx = 1.0 / cols;
        constraints.weighty = 1.0;
        constraints.anchor = GridBagConstraints.WEST;

        int xPos = 0;
        for(Avatar avatar : Avatar.getPredefinedAvatars()){
            JCheckBox checkBox = new JCheckBox(avatar.getName(), true);
            this.cbList.add(checkBox);
            if(xPos >= cols-1){
                constraints.gridwidth = GridBagConstraints.REMAINDER;
                xPos = 0;
            }else{
                constraints.gridwidth = 1;
                xPos++;
            }
            avatarPanel.add(checkBox, constraints);
            this.cbMap.put(avatar, checkBox);
        }
        
        return avatarPanel;
    }
    
    /**
     * ボタン群パネルを生成。
     * @return ボタン群パネル
     */
    private JComponent createButtonPanel(){
        JPanel buttonPanel = new JPanel();
        
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();

        buttonPanel.setLayout(layout);

        constraints.weightx = 1.0 / 3;
        constraints.insets = new Insets(0, 0, 0, 5);
        buttonPanel.add(this.selAllButton, constraints);
        buttonPanel.add(this.selNoneButton, constraints);
        constraints.insets = new Insets(0, 0, 0, 0);
        buttonPanel.add(this.negateButton, constraints);

        return buttonPanel;
    }
    
    /**
     * リスナを登録する。
     * @param listener リスナ
     */
    public void addChangeListener(ChangeListener listener){
        this.listeners.add(ChangeListener.class, listener);
    }

    /**
     * リスナを削除する。
     * @param listener リスナ
     */
    public void removeChangeListener(ChangeListener listener){
        this.listeners.remove(ChangeListener.class, listener);
    }

    /**
     * 全リスナを取得する。
     * @return リスナの配列
     */
    public ChangeListener[] getChangeListeners(){
        return this.listeners.getListeners(ChangeListener.class);
    }

    /**
     * 全リスナへフィルタ操作を通知する。
     */
    protected void fireCheckChanged(){
        ChangeEvent changeEvent = new ChangeEvent(this);
        for(ChangeListener listener : getChangeListeners()){
            listener.stateChanged(changeEvent);
        }
    }

    /**
     * ボタン状態の初期化
     */
    public void initButtons(){
        this.checkPublic.setSelected(true);
        this.checkWolf.setSelected(true);
        this.checkPrivate.setSelected(true);
        this.checkGrave.setSelected(true);

        this.selAllButton.doClick();

        return;
    }
    
    /**
     * チェックボックスまたはボタン操作時にリスナとして呼ばれる。
     * @param event イベント
     */
    public void actionPerformed(ActionEvent event){
        Object source = event.getSource();

        if(source == this.selAllButton){
            boolean hasChanged = false;
            for(JCheckBox avatarCBox : this.cbList){
                if( ! avatarCBox.isSelected()){
                    avatarCBox.setSelected(true);
                    hasChanged = true;
                }
            }
            if(hasChanged) fireCheckChanged();
        }else if(source == this.selNoneButton){
            boolean hasChanged = false;
            for(JCheckBox avatarCBox : this.cbList){
                if(avatarCBox.isSelected()){
                    avatarCBox.setSelected(false);
                    hasChanged = true;
                }
            }
            if(hasChanged) fireCheckChanged();
        }else if(source == this.negateButton){
            for(JCheckBox avatarCBox : this.cbList){
                if(avatarCBox.isSelected()){
                    avatarCBox.setSelected(false);
                }else{
                    avatarCBox.setSelected(true);
                }
            }
            fireCheckChanged();
        }else if(source instanceof JCheckBox){
            fireCheckChanged();
        }

        return;
    }

    /**
     * フィルタ判定を行う。
     * @param topicView 判定対象のTopicView
     * @return フィルタ対象ならtrue
     */
    public boolean isFiltered(TopicView topicView){
        TalkView talkView;
        if(topicView instanceof TalkView){
            talkView = (TalkView) topicView;
        }else{
            return false;
        }

        JCheckBox cbox;

        Talk.Type type = talkView.getTalkType();
        switch(type){
        case PUBLIC:
            cbox = this.checkPublic;
            break;
        case WOLFONLY:
            cbox = this.checkWolf;
            break;
        case PRIVATE:
            cbox = this.checkPrivate;
            break;
        case GRAVE:
            cbox = this.checkGrave;
            break;
        default:
            assert false;
            return true;
        }
        if( ! cbox.isSelected()){
            return true;
        }

        Avatar avatar = talkView.getAvatar();
        cbox = this.cbMap.get(avatar);
        if( ! cbox.isSelected()){
            return true;
        }

        return false;
    }

    /**
     * カスタム化されたフィルタ状態。
     */
    private class FilterPanelContext implements FilterContext{
        private final BitSet context = new BitSet();
        public FilterPanelContext(){
            int index = 0;
            this.context.set(index++, checkPublic.isSelected());
            this.context.set(index++, checkWolf.isSelected());
            this.context.set(index++, checkPrivate.isSelected());
            this.context.set(index++, checkGrave.isSelected());

            for(Avatar avatar : Avatar.getPredefinedAvatars()){
                JCheckBox checkBox = cbMap.get(avatar);
                this.context.set(index++, checkBox.isSelected());
            }

            return;
        }
    }

    /**
     * フィルタ状態を取得する。
     * @return フィルタ状態
     */
    public FilterContext getFilterContext(){
        return new FilterPanelContext();
    }

    /**
     * 与えられたフィルタ状態と今等しい状態にあるかを判定する。
     * @param context フィルタ状態
     * @return 等しければtrue
     */
    public boolean isSame(FilterContext context){
        if(context == null) return false;
        if( ! (context instanceof FilterPanelContext) ) return false;
        FilterPanelContext argContext = (FilterPanelContext) context;
        FilterPanelContext thisContext = (FilterPanelContext) getFilterContext();

        return thisContext.context.equals(argContext.context);
    }

    // TODO 占い結果、狩人の護衛先もフィルタできたほうがいいか？
}
