/*
 * main browser
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: JdfBrowser.java 87 2008-07-10 17:12:36Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.JEditorPane;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.HyperlinkListener;
import javax.swing.plaf.TextUI;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultHighlighter;
import javax.swing.text.Document;
import javax.swing.text.EditorKit;
import javax.swing.text.Element;
import javax.swing.text.Highlighter;
import javax.swing.text.StyleConstants;
import javax.swing.text.Utilities;
import javax.swing.text.View;
import javax.swing.text.html.HTML;

/**
 * 独自HTMLベースのメインブラウザ。
 * EditorKitはJdfEditorKitを利用する。
 */
@SuppressWarnings("serial")
public class JdfBrowser
        extends JEditorPane
        implements CaretListener, ActionListener, MouseListener{

    private static final Highlighter.HighlightPainter regexPainter =
            new DefaultHighlighter.DefaultHighlightPainter(Color.YELLOW);

    private final JPopupMenu popup;
    private final JMenuItem menuCopy = new JMenuItem("選択範囲をコピー");
    private final JMenuItem menuSelTalk = new JMenuItem("この発言をコピー");
    private final JMenuItem menuPrint = new JMenuItem("印刷");
    private final JMenuItem menuSelAll = new JMenuItem("全選択");

    private String internalText = "";
    private Village village;
    private Period period;
    private TopicFilter topicFilter = null;
    private TopicFilter.FilterContext filterContext;
    private boolean hasShown = false;
    private Pattern searchRegex = null;
    private Element talkElem;

    /**
     * 初期Periodを指定してメインブラウザを作成する。
     * @param period 初期Period
     */
    public JdfBrowser(Period period){
        super();

        setEditable(false);
        setContentType("text/html");
        putClientProperty(JEditorPane.W3C_LENGTH_UNITS, Boolean.TRUE);

        this.popup = createPopupMenu();

        addMouseListener(this);
        addCaretListener(this);

        setPeriod(period);

        return;
    }

    private JPopupMenu createPopupMenu(){
        JPopupMenu result = new JPopupMenu();

        this.menuCopy.addActionListener(this);
        this.menuSelTalk.addActionListener(this);
        this.menuPrint.addActionListener(this);
        this.menuSelAll.addActionListener(this);

        this.menuCopy.setActionCommand(MenuManager.COMMAND_COPY);
        this.menuSelTalk.setActionCommand(MenuManager.COMMAND_SELTALK);
        this.menuPrint.setActionCommand(MenuManager.COMMAND_PRINT);
        this.menuSelAll.setActionCommand(MenuManager.COMMAND_SELALL);

        result.add(this.menuCopy);
        result.add(this.menuSelTalk);
//      result.add(this.menuPrint);
        result.add(this.menuSelAll);

        return result;
    }

    /**
     * 現在のPeriodを返す。
     * @return 現在のPeriod
     */
    public Period getPeriod(){
        return this.period;
    }

    /**
     * Periodを更新する。
     * 古いPeriodの表示内容は消える。
     * 新しいPeriodの表示内容はまだ反映されない。
     * 既存のHyperlinkListenerは登録を外される。
     * @param period 新しいPeriod
     */
    synchronized public void setPeriod(Period period){
        if(period == this.period) return;

        this.period = period;
        this.hasShown = false;
        this.filterContext = null;
        setText("");

        if(this.period == null) return;

        if(period.getVillage() == this.village) return;
        this.village = period.getVillage();

        EditorKit kit = new JdfEditorKit(this.village);
        setEditorKit(kit);

        HyperlinkListener[] listeners = getHyperlinkListeners();
        for(HyperlinkListener listener : listeners){
            removeHyperlinkListener(listener);
        }
        addHyperlinkListener(new AnchorListener(this.village));

        return;
    }

    /**
     * フィルタを適用してPeriodの内容を出力する。
     * @param force trueなら強制再出力
     */
    synchronized public void showTopics(boolean force){
        if(force || ! this.hasShown){
            CharSequence html = this.period.buildJdfHTML();
            setHTML(html);
            this.hasShown = true;
            this.filterContext = null;
        }

        highlightRegex();
        filtering();

        return;
    }

    /**
     * 指定したHTML文書で表示を更新する。
     * @param html HTML文書
     */
    private void setHTML(CharSequence html){
        Reader reader;
        if(html != null) reader = new StringReader(html.toString());
        else             reader = new StringReader("");

        EditorKit kit = getEditorKit();
        Document doc = kit.createDefaultDocument();

        try{
            kit.read(reader, doc, 0);
        }catch(IOException e){
            assert false;
        }catch(BadLocationException e){
            assert false;
        }

        setDocument(doc);
        int docLength = doc.getLength();
        try{
            this.internalText = doc.getText(0, docLength);
        }catch(BadLocationException e){
            assert false;
        }

        return;
    }

    public void setTopicFilter(TopicFilter filter){
        this.topicFilter = filter;
        return;
    }

    /**
     * 発言フィルタを適用する。
     */
    synchronized public void filtering(){
        if(   this.topicFilter != null
           && this.topicFilter.isSame(this.filterContext)){
            return;
        }

        List<TopicView> viewList = getTopicViewList(null, null);
        for(TopicView view : viewList){
            if(   this.topicFilter != null
               && this.topicFilter.isFiltered(view)){
                view.setVisible(false);
            }else{
                view.setVisible(true);
            }

            View parentView = view.getParent();
            if(parentView == null) continue;
            parentView.preferenceChanged(view, true, true);
        }

        if(this.topicFilter != null){
            this.filterContext = this.topicFilter.getFilterContext();
        }else{
            this.filterContext = null;
        }

        return;
    }

    /**
     * 検索パターンを設定する。
     * @param searchRegex 検索パターン
     */
    public void setSearchRegex(Pattern searchRegex){
        if(this.searchRegex == searchRegex) return;
        this.searchRegex = searchRegex;
        return;
    }

    /**
     * 与えられた正規表現にマッチする文字列をハイライト描画する。
     */
    synchronized public int highlightRegex(){
        clearHighlight();
        if(this.searchRegex == null) return 0;

        int hits = 0;

        Highlighter highlighter = getHighlighter();
        int docLength = this.internalText.length();
        Matcher matcher = this.searchRegex.matcher(this.internalText);
        matcher.region(0, docLength);
        while( matcher.find() ){
            int startPos = matcher.start();
            int endPos = matcher.end();
            if(startPos == endPos) break;
            hits++;
            matcher.region(endPos, docLength);
            try{
                highlighter.addHighlight(startPos, endPos, regexPainter);
            }catch(BadLocationException e){
                assert false;
                return 0;
            }
        }

        return hits;
    }

    /**
     * 検索ハイライトをクリアする。
     */
    private void clearHighlight(){
        Highlighter highlighter = getHighlighter();
        Highlighter.Highlight[] highlights = highlighter.getHighlights();
        for(Highlighter.Highlight highlight : highlights){
            if(highlight.getPainter() != regexPainter) continue;
            highlighter.removeHighlight(highlight);
        }
        return;
    }

    /**
     * ルートViewを得る。
     * @return ルートView
     */
    private View getRootView(){
        TextUI textUI = getUI();
        View root = textUI.getRootView(this);
        return root;
    }

    /**
     * 全てのTopicViewを与えられた親から再帰的に抽出する。
     * @param parent 親View
     * @param list TopicViewのリスト
     * @return TopicViewのリスト
     */
    private List<TopicView> getTopicViewList(View parent,
                                              List<TopicView> list){
        if(parent == null){
            parent = getRootView();
            list = new LinkedList<TopicView>();
        }

        int childNo = parent.getViewCount();
        for(int ct=0; ct <= childNo-1; ct++){
            View child = parent.getView(ct);
            if(child instanceof TopicView){
                list.add( (TopicView) child );
            }else{
                getTopicViewList(child, list);
            }
        }

        return list;
    }

    /**
     * カレット変更イベント（範囲選択操作）の受信に応じて
     * ポップアップメニューの内訳を変更する。
     * @param event カレット変更イベント
     */
    public void caretUpdate(CaretEvent event){
        int dot  = event.getDot();
        int mark = event.getMark();
        if(dot == mark){
            this.menuCopy.setEnabled(false);
        }else{
            this.menuCopy.setEnabled(true);
        }
        return;
    }

    /**
     * ポップアップメニュー選択
     * @param event メニュー選択イベント
     */
    public void actionPerformed(ActionEvent event){
        String cmd = event.getActionCommand();
        if(cmd.equals(MenuManager.COMMAND_COPY)){
            copy();
        }else if(cmd.equals(MenuManager.COMMAND_SELTALK)){
            if(this.talkElem == null) return;
            int startPos = this.talkElem.getStartOffset();
            int endPos = this.talkElem.getEndOffset();
            select(startPos, endPos);
            copy();
        }else if(cmd.equals(MenuManager.COMMAND_PRINT)){
//          print(); // JRE1.6
        }else if(cmd.equals(MenuManager.COMMAND_SELALL)){
            selectAll();
        }

        return;
    }

    /** マウスイベント処理（無視） */
    public void mouseClicked(MouseEvent event){ }

    /** マウスイベント処理（無視） */
    public void mouseEntered(MouseEvent event){ }

    /** マウスイベント処理（無視） */
    public void mouseExited(MouseEvent event){ }

    /**
     * ポップアップメニュー表示。
     * @param event マウスボタン押下イベント
     */
    public void mousePressed(MouseEvent event){
        mouseReleased(event);
        return;
    }

    /**
     * ポップアップメニュー表示。
     * @param event マウスボタン離イベント
     */
    public void mouseReleased(MouseEvent event){
        if( ! event.isPopupTrigger() ) return;
        
        Point point = event.getPoint();
        int offset = viewToModel(point);
        Element element = Utilities.getParagraphElement(this, offset);
        
        this.talkElem = findTalkDivElem(element);
        if(this.talkElem != null){
            menuSelTalk.setEnabled(true);
        }else{
            menuSelTalk.setEnabled(false);
        }
        
        this.popup.show(this, event.getX(), event.getY());
        
        return;
    }

    /**
     * 与えられた子Elementの祖先から会話Elementを探索する。
     * @param child 子Element
     * @return 会話Element。見つからなければnull。
     */
    private Element findTalkDivElem(Element child){
        Element element = child;
        while( element != null && ! isTalkDivElem(element) ){
            element = element.getParentElement();
        }
        return element;
    }
    
    /**
     * 与えられたElementが会話(&lt;div class="message"&gt;...)か否か判定する。
     * @param element 要素
     * @return 会話ならtrue
     */
    private boolean isTalkDivElem(Element element){
        if(element == null) return false;

        AttributeSet attr = element.getAttributes();
        Object obj = attr.getAttribute(StyleConstants.NameAttribute);
        if( ! (obj instanceof HTML.Tag) ) return false;
        HTML.Tag kind = (HTML.Tag) obj;
        if(kind != HTML.Tag.DIV) return false;
        
        Object objClass = attr.getAttribute(HTML.Attribute.CLASS);
        String attrClass = (String) objClass;
        if(attrClass == null) return false;
        if( ! attrClass.equals("message") ) return false;
        
        return true;
    }
    
    /**
     * 描画品質をカスタマイズするためのフック
     * @param g グラフィックスコンテキスト
     */
    // TODO フキダシの角丸だけでもアンチエイリアス指定しときたいなぁ
    @Override
    synchronized public void paint(Graphics g){
        // updateUI() 対策
        if(this.topicFilter != null && this.filterContext == null){
            SwingUtilities.invokeLater(new Runnable(){
                public void run(){
                    filtering();
                    return;
                }
            });
            repaint();
            return;
        }
        
        Graphics2D g2 = (Graphics2D) g;
        g2.addRenderingHints(GUIUtils.getDefaultHints());
        super.paint(g);
        return;
    }

    /**
     * L&F を変更する。
     */
    @Override
    public void updateUI(){
        super.updateUI();

        if(this.popup != null){
            SwingUtilities.updateComponentTreeUI(this.popup);
        }

        this.filterContext = null;

        SwingUtilities.invokeLater(new Runnable(){
            public void run(){
                highlightRegex();
                return;
            }
        });

        return;
    }
}
