/*
 * MVC controller
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Controller.java 117 2008-07-17 15:19:47Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Container;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.net.URL;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.LookAndFeel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.tree.TreePath;

/**
 * いわゆるMVCでいうとこのコントローラ
 */
public class Controller
        implements ActionListener,
                   TreeWillExpandListener,
                   TreeSelectionListener,
                   ChangeListener            {
    
    private static final String HELP_HTML = "resources/help.html";

    private MenuManager menuManager;
    private TopFrameView topFrame;
    private LandsModel model;
    private JFrame helpFrame;
    private FilterPanel filterFrame;
    private AccountPanel accountFrame;
    private LogFrame showlogFrame;

    /**
     * コントローラの生成。
     * @param menuManager アクション管理
     * @param topFrame 最上位ビュー
     * @param model 最上位データモデル
     */
    public Controller(MenuManager menuManager,
                       TopFrameView topFrame,
                       LandsModel model){
        super();

        this.menuManager = menuManager;
        this.topFrame = topFrame;
        this.model = model;

        this.menuManager.addActionListener(this);

        JTree treeView = topFrame.getTreeView();
        treeView.setModel(model);
        treeView.addTreeWillExpandListener(this);
        treeView.addTreeSelectionListener(this);

        topFrame.getTabBrowser().addChangeListener(this);
        topFrame.getReloadButton().addActionListener(this);
        topFrame.getSearchButton().addActionListener(this);
        
        JButton reloadVillageListButton = topFrame
                                         .getLandsTree()
                                         .getReloadVillageListButton();
        reloadVillageListButton.addActionListener(this);
        reloadVillageListButton.setEnabled(false);

        this.filterFrame = new FilterPanel();
//        this.filterFrame.setAlwaysOnTop(true);
        this.filterFrame.addChangeListener(this);
        this.filterFrame.pack();
        this.filterFrame.setVisible(false);
        
        this.showlogFrame = new LogFrame();
        this.showlogFrame.pack();
        this.showlogFrame.setSize(600,250);
        this.showlogFrame.setLocationByPlatform(true);
        this.showlogFrame.setVisible(false);
        Handler newHandler = this.showlogFrame.getHandler();
        Jindolf.logger.addHandler(newHandler);
        Handler[] handlers = Jindolf.logger.getHandlers();
        for(Handler handler : handlers){
            if( ! (handler instanceof PileHandler) ) continue;
            PileHandler pile = (PileHandler) handler;
            pile.delegate(newHandler);
            pile.close();
        }
        
        return;
    }

    /**
     * 村一覧の再読み込み
     */
    private void reloadVillageList(){
        JTree tree = this.topFrame.getTreeView();
        TreePath path = tree.getSelectionPath();
        if(path == null) return;

        Land land = null;
        for(int ct=0; ct<path.getPathCount(); ct++){
            Object obj = path.getPathComponent(ct);
            if(obj instanceof Land){
                land = (Land) obj;
                break;
            }
        }
        if(land == null) return;

        this.topFrame.showInitPanel();
        
        execReloadVillageList(land);
        
        return;
    }

    /**
     * 指定した国の村一覧を読み込む。
     * @param land 国
     */
    private void execReloadVillageList(final Land land){
        final LandsTree treePanel = this.topFrame.getLandsTree();
        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                topFrame.setBusy(true);
                topFrame.updateSysMessage("村一覧を読み込み中…");
                try{
                    try{
                        model.loadVillageList(land);
                    }catch(IOException e){
                        showNetworkError(land, e);
                    }
                    treePanel.expandLand(land);
                }finally{
                    topFrame.updateSysMessage("村一覧の読み込み完了");
                    topFrame.setBusy(false);
                }
                return;
            }
        });
        return;
    }
    
    /**
     * 検索ボタン押下処理。
     */
    private void searchPeriod(){
        JdfBrowser browser = currentJdfBrowser();
        if(browser == null) return;

        Pattern searchRegex = currentPattern();
        browser.setSearchRegex(searchRegex);
        int hits = browser.highlightRegex();  // TODO 別スレッド化
        
        this.topFrame.updateSysMessage("［"+hits+"］件ヒットしました");
        
        return;
    }

    /**
     * 現在選択中のの検索パターンを得る。
     * @return
     */
    private Pattern currentPattern(){
        JComboBox box = this.topFrame.getFindBox();
        Object selected = box.getSelectedItem();
        if(selected == null) return null;
        String searchString = selected.toString();
        if(searchString.length() <= 0) return null;

        Pattern pattern;
        try{
            pattern= Pattern.compile(searchString, Pattern.DOTALL);
        }catch(PatternSyntaxException e){
            return null;
        }

        return pattern;
    }

    /**
     * Period表示の更新処理。
     * @param force trueならPeriodデータを強制再読み込み。
     */
    private void updatePeriod(final boolean force){
        final TabBrowser tabBrowser = this.topFrame.getTabBrowser();
        final Village village = tabBrowser.getVillage();
        if(village == null) return;
        this.topFrame.setFrameTitle(village.getVillageName());

        final JdfScroller scroller = currentJdfScroller();
        final JdfBrowser browser = currentJdfBrowser();
        if(browser == null) return;
        browser.setTopicFilter(filterFrame);
        final Period period = browser.getPeriod();
        if(period == null) return;

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                topFrame.setBusy(true);
                try{
                    boolean wasHot = loadPeriod();

                    if(wasHot && ! period.isHot() ){
                        if(updatePeriodList() != true) return;
                    }

                    renderBrowser();
                }finally{
                    topFrame.setBusy(false);
                }
                return;
            }

            private boolean loadPeriod(){
                topFrame.updateSysMessage("1日分のデータを読み込んでいます…");
                boolean wasHot;
                try{
                    wasHot = period.isHot();
                    try{
                        period.loadPeriod(force);
                    }catch(IOException e){
                        showNetworkError(village, e);
                    }
                }finally{
                    topFrame.updateSysMessage("1日分のデータを読み終わりました");
                }
                return wasHot;
            }

            private boolean updatePeriodList(){
                topFrame.updateSysMessage("村情報を読み直しています…");
                try{
                    village.updatePeriodList();
                }catch(IOException e){
                    showNetworkError(village, e);
                    return false;
                }
                try{
                    SwingUtilities.invokeAndWait(new Runnable(){
                        public void run(){
                            tabBrowser.setVillage(village);
                            return;
                        }
                    });
                }catch(Exception e){
                    Jindolf.logger.log(Level.SEVERE,
                                       "タブ操作で致命的な障害が発生しました",
                                       e);
                }
                topFrame.updateSysMessage("村情報を読み直しました…");
                return true;
            }
            
            private void renderBrowser(){
                topFrame.updateSysMessage("レンダリング中…");
                try{
                    final int lastPos = scroller.getVerticalPosition();
                    try{
                        SwingUtilities.invokeAndWait(new Runnable(){
                            public void run(){
                                browser.showTopics(force);
                                return;
                            }
                        });
                    }catch(Exception e){
                        Jindolf.logger
                               .log(Level.SEVERE,
                                    "ブラウザ表示で致命的な障害が発生しました",
                                    e);
                    }
                    SwingUtilities.invokeLater(new Runnable(){
                        public void run(){
                            scroller.setVerticalPosition(lastPos);
                        }
                    });
                }finally{
                    topFrame.updateSysMessage("レンダリング完了");
                }
                return;
            }
        });

        return;
    }

    /**
     * 発言フィルタの操作による更新処理。
     */
    private void filterChanged(){
        final JdfBrowser browser = currentJdfBrowser();
        if(browser == null) return;
        browser.setTopicFilter(filterFrame);

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                topFrame.setBusy(true);
                topFrame.updateSysMessage("フィルタリング中…");
                try{
                    browser.filtering();
                }finally{
                    topFrame.updateSysMessage("フィルタリング完了");
                    topFrame.setBusy(false);
                }
                return;
            }
        });

        return;
    }

    /**
     * 現在選択中のPeriodを内包するJdfScrollerを返す。
     * @return JdfScroller
     */
    private JdfScroller currentJdfScroller(){
        TabBrowser tb = this.topFrame.getTabBrowser();
        JdfScroller result = tb.currentJdfScroller();
        return result;
    }
    
    /**
     * 現在選択中のPeriodを内包するJdfBrowserを返す。
     * @return JdfScroller
     */
    private JdfBrowser currentJdfBrowser(){
        JdfScroller scroller = currentJdfScroller();
        if(scroller == null) return null;
        JdfBrowser result = scroller.getJdfBrowser();
        return result;
    }
    
    /**
     * L&Fの変更を行う。
     */
    private void changeLandF(){
        String className = menuManager.getSelectedLookAndFeel();

        LookAndFeel lnf;
        try{
            Class lnfClass = Class.forName(className);
            lnf = (LookAndFeel)( lnfClass.newInstance() );
        }catch(Exception e){
            String message = "このLook&Feel["
                            + className
                            + "]を読み込む事ができません。";
            Jindolf.logger.log(Level.WARNING, message, e);
            JOptionPane.showMessageDialog(
                this.topFrame,
                message,
                "Look&Feel - " + Jindolf.title,
                JOptionPane.WARNING_MESSAGE );
            return;
        }

        try{
            UIManager.setLookAndFeel(lnf);
        }catch(UnsupportedLookAndFeelException e){
            String message = "このLook&Feel["
                            + lnf.getName()
                            + "]はサポートされていません。";
            Jindolf.logger.log(Level.WARNING, message, e);
            JOptionPane.showMessageDialog(
                this.topFrame,
                message,
                "Look&Feel - " + Jindolf.title,
                JOptionPane.WARNING_MESSAGE );
            return;
        }

        Jindolf.logger.info(
                "Look&Feelが["
                +lnf.getName()
                +"]に変更されました。");
        
        final Runnable updateUITask = new Runnable(){
            public void run(){
                SwingUtilities.updateComponentTreeUI(topFrame);
                topFrame.validate();

                if(helpFrame != null){
                    SwingUtilities.updateComponentTreeUI(helpFrame);
                    helpFrame.validate();
                }

                if(showlogFrame != null){
                    SwingUtilities.updateComponentTreeUI(showlogFrame);
                    showlogFrame.validate();
                }

                if(filterFrame != null){
                    SwingUtilities.updateComponentTreeUI(filterFrame);
                    filterFrame.validate();
                    filterFrame.pack();
                }

                if(accountFrame != null){
                    SwingUtilities.updateComponentTreeUI(accountFrame);
                    accountFrame.validate();
                    accountFrame.pack();
                }
            }
        };

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                topFrame.setBusy(true);
                topFrame.updateSysMessage("Look&Feelを更新中…");
                try{
                    SwingUtilities.invokeAndWait(updateUITask);
                }catch(Exception e){
                    Jindolf.logger.log(
                            Level.WARNING,
                            "Look&Feelの更新に失敗しました。",
                            e );
                }finally{
                    topFrame.updateSysMessage("Look&Feelが更新されました");
                    topFrame.setBusy(false);
                }
                return;
            }
        });

        return;
    }

    /**
     * アプリ終了。
     */
    private void doExit(){
        Jindolf.exit(0);
    }

    /**
     * フレーム表示のトグル処理
     * @param frame フレーム
     */
    // TODO アイコン化からの復帰もサポートしたい
    private void toggleFrame(JFrame frame){
        if(frame == null) return;
        if(frame.isVisible()){
            frame.setVisible(false);
            frame.dispose();
        }else{
            frame.setVisible(true);
        }
        return;
    }

    /**
     * 発言フィルタ画面を表示する。
     */
    private void showFilter(){
        toggleFrame(filterFrame);
        return;
    }

    /**
     * アカウント管理画面を表示する。
     */
    private void showAccount(){
        if(accountFrame != null){                 // show Toggle
            toggleFrame(accountFrame);
            return;
        }

        accountFrame = new AccountPanel(model);
        accountFrame.pack();
        accountFrame.setVisible(true);

        return;
    }

    private void showLog(){
        toggleFrame(showlogFrame);
        return;
    }
    
    /**
     * Help画面を表示する。
     */
    private void showHelp(){
        if(helpFrame != null){                 // show Toggle
            toggleFrame(helpFrame);
            return;
        }

        helpFrame = new JFrame(Jindolf.title + " ヘルプ");

        helpFrame.setResizable(true);
        Toolkit kit = helpFrame.getToolkit();
        kit.setDynamicLayout(false);
        helpFrame.setIconImage(GUIUtils.getWindowIconImage());
        helpFrame.setLocationByPlatform(true);

        HTMLEditorKit editorKit = new UnshareStyleEditorKit();

        JEditorPane edit = new JEditorPane();
        edit.setEditorKit(editorKit);
        edit.setEditable(false);
        edit.setContentType("text/html");
        edit.putClientProperty(JEditorPane.W3C_LENGTH_UNITS, Boolean.TRUE);
        edit.setBorder(new EmptyBorder(0,0,0,0));

        URL htmlurl = Jindolf.getResource(HELP_HTML);
        try{
            edit.setPage(htmlurl);
        }catch(IOException e){
            Jindolf.logger.log(Level.WARNING,
                               "ヘルプファイルが読み込めません",
                               e);
            helpFrame = null;
            return;
        }

        JScrollPane sc = new JScrollPane(edit);

        Container content = helpFrame.getContentPane();
        content.add(sc);

        helpFrame.pack();
        helpFrame.setSize(450, 450);
        helpFrame.setVisible(true);

        return;
    }

    /**
     * About画面を表示する。
     */
    private void showAbout(){
        String message =
                Jindolf.title
                + "   Version " + Jindolf.version + "\n"
                + Jindolf.copyright + "\n"
                + "ライセンス: " + Jindolf.license + "\n"
                + "連絡先: " + Jindolf.contact;

        JOptionPane pane = new JOptionPane(message,
                                           JOptionPane.INFORMATION_MESSAGE,
                                           JOptionPane.DEFAULT_OPTION,
                                           GUIUtils.getLogoIcon());

        JDialog dialog = pane.createDialog(topFrame, Jindolf.title + "について");

// JRE1.6 only
//      dialog.setIconImage(GUIUtils.getWindowIconImage());

        dialog.pack();
        dialog.setVisible(true);
        dialog.dispose();

        return;
    }

    /**
     * ネットワークエラーを通知するモーダルダイアログを表示する。
     * OKボタンを押すまでこのメソッドは戻ってこない。
     * @param e ネットワークエラー
     */
    public void showNetworkError(Village village, IOException e){
        Land land = village.getParentLand();
        showNetworkError(land, e);
        return;
    }

    /**
     * ネットワークエラーを通知するモーダルダイアログを表示する。
     * OKボタンを押すまでこのメソッドは戻ってこない。
     * @param e ネットワークエラー
     */
    public void showNetworkError(Land land, IOException e){
        Jindolf.logger.log(Level.WARNING, "ネットワークで障害が発生しました", e);

        ServerAccess server = land.getServerAccess();
        String message =
                land.getLandName()
                +"を運営するサーバとの間の通信で何らかのトラブルが発生しました。\n"
                +"相手サーバのURLは [ " + server.getBaseURL() + " ] だよ。\n"
                +"Webブラウザでも遊べないか確認してみてね!\n";

        JOptionPane pane = new JOptionPane(message,
                                           JOptionPane.WARNING_MESSAGE,
                                           JOptionPane.DEFAULT_OPTION );

        JDialog dialog = pane.createDialog(topFrame,
                                           "通信異常発生 - " + Jindolf.title);

        dialog.pack();
        dialog.setVisible(true);
        dialog.dispose();

        return;
    }

    /**
     * ツリーリストで何らかの要素（国、村）がクリックされたときの処理。
     * @param event イベント
     */
    public void valueChanged(TreeSelectionEvent event){
        TreePath path = event.getNewLeadSelectionPath();
        if(path == null) return;

        Object selObj = path.getLastPathComponent();
        
        if( selObj instanceof Land ){
            Land land = (Land)selObj;
            this.topFrame.showLandInfo(land);
        }else if( selObj instanceof Village ){
            final Village village = (Village)selObj;
            
            Executor executor = Executors.newCachedThreadPool();
            executor.execute(new Runnable(){
                public void run(){
                    topFrame.setBusy(true);
                    topFrame.updateSysMessage("村情報を読み込み中…");

                    try{
                        village.updatePeriodList();
                    }catch(IOException e){
                        showNetworkError(village, e);
                        return;
                    }finally{
                        topFrame.updateSysMessage("村情報の読み込み完了");
                        topFrame.setBusy(false);
                    }
                    
                    topFrame.showVillageInfo(village);
                    
                    return;
                }
            });
        }
        
        return;
    }

    /**
     * Periodがタブ選択されたときもしくは発言フィルタが操作されたときの処理。
     * @param event イベント
     */
    public void stateChanged(ChangeEvent event){
        Object source = event.getSource();

        if(source == filterFrame){
            filterChanged();
        }else if(source instanceof TabBrowser){
            final JdfBrowser browser = currentJdfBrowser();
            if(browser == null) return;
            Pattern searchRegex = currentPattern();
            browser.setSearchRegex(searchRegex);
            updatePeriod(false);
        }
        return;
    }

    /**
     * アクションイベント処理。
     * 主にメニュー選択やボタン押下など。
     * @param e イベント
     */
    public void actionPerformed(ActionEvent e){
        String cmd = e.getActionCommand();
        if(cmd.equals(MenuManager.COMMAND_ABOUT)){
            showAbout();
        }else if(cmd.equals(MenuManager.COMMAND_EXIT)){
            doExit();
        }else if(cmd.equals(MenuManager.COMMAND_LANDF)){
            changeLandF();
        }else if(cmd.equals(MenuManager.COMMAND_HELPDOC)){
            showHelp();
        }else if(cmd.equals(MenuManager.COMMAND_FILTER)){
            showFilter();
        }else if(cmd.equals(MenuManager.COMMAND_ACCOUNT)){
            showAccount();
        }else if(cmd.equals(MenuManager.COMMAND_SHOWLOG)){
            showLog();
        }else if(cmd.equals(MenuManager.COMMAND_RELOAD)){
            updatePeriod(true);
        }else if(cmd.equals(MenuManager.COMMAND_SEARCH)){
            searchPeriod();
        }else if(cmd.equals(MenuManager.COMMAND_VILLAGELIST)){
            reloadVillageList();
        }
        return;
    }

    /**
     * 村選択ツリーリストが畳まれるとき呼ばれる。
     * @param event ツリーイベント
     */
    public void treeWillCollapse(TreeExpansionEvent event){
        return;
    }

    /**
     * 村選択ツリーリストが展開されるとき呼ばれる。
     * @param event ツリーイベント
     */
    public void treeWillExpand(TreeExpansionEvent event){
        if(!(event.getSource() instanceof JTree)){
            return;
        }

        TreePath path = event.getPath();
        Object lastObj = path.getLastPathComponent();
        if(!(lastObj instanceof Land)){
            return;
        }
        final Land land = (Land) lastObj;
        if(land.getVillageCount() > 0){
            return;
        }

        execReloadVillageList(land);

        return;
    }
}
