/*
 * アンカー描画
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: AnchorDraw.java 154 2008-09-04 16:15:41Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.font.FontRenderContext;
import java.io.IOException;
import java.util.regex.Pattern;

/**
 * アンカー描画
 */
public class AnchorDraw implements RowsLtoR{

    private static final Color COLOR_ANCHOR   = new Color(0xffff99);
    private static final int UNDER_MARGIN = 5;

    private final Talk talk;
    private final GlyphDraw caption;
    private final GlyphDraw dialog;
    private final Image faceImage;
    private final Point imageOrigin   = new Point();
    private final Point captionOrigin = new Point();
    private final Point dialogOrigin  = new Point();
    private final Rectangle bounds = new Rectangle();
    private Font font;
    private FontRenderContext frc;
    
    /**
     * コンストラクタ
     * @param talk 発言
     * @param font フォント
     * @param frc フォント描画設定
     */
    public AnchorDraw(Talk talk, Font font, FontRenderContext frc){
        super();
        
        this.talk = talk;
        this.font = font;
        this.frc = frc;

        Period period = this.talk.getPeriod();
        Village village = period.getVillage();
        Avatar avatar = this.talk.getAvatar();
        
        Image image;
        if(this.talk.getTalkType() == Talk.Type.GRAVE){
            image = village.getGraveImage();
        }else{
            image = village.getAvatarFaceImage(avatar);
        }
        this.faceImage = image;
        
        this.caption = new GlyphDraw(this.font, this.frc,
                                     getCaptionString() );
        this.dialog = new GlyphDraw(this.font, this.frc,
                                    this.talk.getDialog() );

        this.caption.setColor(COLOR_ANCHOR);
        this.dialog .setColor(COLOR_ANCHOR);
        
        return;
    }
    
    /**
     * キャプション文字列の取得。
     * @return キャプション文字列
     */
    private CharSequence getCaptionString(){
        StringBuilder result = new StringBuilder();
        Avatar avatar = this.talk.getAvatar();
        result.append(avatar.getFullName())
              .append(' ')
              .append(this.talk.getAnchorNotation());
        return result;
    }
    
    /**
     * 新しい幅を指定し、寸法の再計算、内部の再レイアウトを促す。
     * @param newWidth 新しいピクセル幅
     * @return 新しい寸法
     */
    public Rectangle setWidth(int newWidth){
        int imageWidth  = this.faceImage.getWidth(null);
        int imageHeight = this.faceImage.getHeight(null);
        
        Rectangle captionRect;
        captionRect = this.caption.setWidth(newWidth - imageWidth);
        int captionWidth = captionRect.width;
        int captionHeight = captionRect.height;

        Rectangle dialogRect;
        dialogRect = this.dialog.setWidth(newWidth);
        int dialogWidth = dialogRect.width;
        int dialogHeight = dialogRect.height;

        int headerHeight = Math.max(imageHeight, captionHeight);
        
        int totalWidth = Math.max(imageWidth + captionWidth, dialogWidth);

        int totalHeight = headerHeight;
        totalHeight += dialogHeight;

        int imageYpos;
        int captionYpos;
        if(imageHeight > captionHeight){
            imageYpos = 0;
            captionYpos = (imageHeight - captionHeight) / 2;
        }else{
            imageYpos = (captionHeight - imageHeight) / 2;
            captionYpos = 0;
        }
        
        this.imageOrigin  .setLocation(0,          imageYpos);
        this.captionOrigin.setLocation(imageWidth, captionYpos);
        this.dialogOrigin .setLocation(0,          headerHeight);
        
        this.bounds.width  = totalWidth;
        this.bounds.height = totalHeight + UNDER_MARGIN;
        
        return this.bounds;
    }

    /**
     * 描画領域の寸法を返す。
     * @return 描画領域の寸法
     */
    public Rectangle getBounds(){
        return this.bounds;
    }

    /**
     * 描画開始位置の指定。
     * @param xPos 描画開始位置のx座標
     * @param yPos 描画開始位置のy座標
     */
    public void setPos(int xPos, int yPos){
        this.bounds.x = xPos;
        this.bounds.y = yPos;
        return;
    }

    /**
     * 描画領域の寸法幅を返す。
     * @return 描画領域の寸法幅
     */
    public int getWidth(){
        return this.bounds.width;
    }

    /**
     * 描画領域の寸法高を返す。
     * @return 描画領域の寸法高
     */
    public int getHeight(){
        return this.bounds.height;
    }
    
    /**
     * フォント描画設定を変更する。
     * @param font フォント
     * @param frc フォント描画設定
     */
    public void setFontInfo(Font font, FontRenderContext frc){
        this.font = font;
        this.frc = frc;

        this.caption.setFontInfo(this.font, this.frc);
        this.dialog.setFontInfo(this.font, this.frc);
        
        int width = getWidth();
        setWidth(width);
        
        return;
    }
    
    /**
     * ドラッグ処理を行う。
     * @param from ドラッグ開始位置
     * @param to 現在のドラッグ位置
     */
    public void drag(Point from, Point to){
        this.caption.drag(from, to);
        this.dialog.drag(from, to);
        return;
    }

    /**
     * 受け取った文字列に選択文字列を追加する。
     * @param appendable 追加対象文字列
     * @return 引数と同じインスタンス
     * @throws java.io.IOException
     */
    public Appendable appendSelected(Appendable appendable)
            throws IOException{
        this.caption.appendSelected(appendable);
        this.dialog.appendSelected(appendable);
        return appendable;
    }
    
    /**
     * 選択範囲の解除。
     */
    public void clearSelect(){
        this.caption.clearSelect();
        this.dialog.clearSelect();
        return;
    }

    /**
     * 検索文字列パターンを設定する
     * @param searchRegex パターン
     * @return ヒット数
     */
    public int setRegex(Pattern searchRegex){
        int total = 0;
        
        total += this.dialog.setRegex(searchRegex);

        return total;
    }

    /**
     * 描画を行う。
     * @param g グラフィックコンテキスト
     */
    public void paint(Graphics2D g){
        final int xPos = this.bounds.x;
        final int yPos = this.bounds.y;
        
        g.drawImage(this.faceImage,
                    xPos + this.imageOrigin.x,
                    yPos + this.imageOrigin.y,
                    null );
        
        this.caption.setPos(xPos + this.captionOrigin.x,
                            yPos + this.captionOrigin.y );
        this.caption.paint(g);

        this.dialog.setPos(xPos + this.dialogOrigin.x,
                           yPos + this.dialogOrigin.y );
        this.dialog.paint(g);
        
        return;
    }
}
