/*
 * Jindolf main class
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Jindolf.java 160 2008-09-05 03:11:58Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Dimension;
import java.awt.GraphicsEnvironment;
import java.awt.Toolkit;
import java.awt.Window;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Reader;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JWindow;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

/**
 * Jindolf スタートアップクラス。
 * 
 * コンストラクタは無いよ。
 * アプリ開始はstaticメソッド{@link #main(String[])}呼び出しから。
 */
public class Jindolf{
    
    public static final String version = "2.7.2";

    public static final String license = "The MIT License";
    public static final String contact = "http://jindolf.sourceforge.jp/";
    
    public static final Date initDate;
    public static final long initNano;
    
    public static final ClassLoader loader;
    public static final Class selfClass;
    public static final Package selfPackage;
    public static final Package jrePackage;
    
    public static final Runtime runtime;
    public static SecurityManager secManager;
    public static final Logger logger;
    
    public static final String title;
    public static final String author;
    public static final String copyright;
    public static final String id;

    private static final int STOCK_SIZE = 64 * 1024;
    private static byte[] stock;

    private static final Charset CHARSET_UTF8 = Charset.forName("UTF-8");
    
    static{
        initDate = new Date();
        initNano = System.nanoTime();

        selfClass = Jindolf.class;
        selfPackage = selfClass.getPackage();
        jrePackage = java.lang.Object.class.getPackage();

        runtime = Runtime.getRuntime();
        secManager = System.getSecurityManager();
        logger = Logger.getLogger(selfPackage.getName());
        logger.setUseParentHandlers(false);
        logger.addHandler(new PileHandler());

        ClassLoader thisLoader;
        try{
            thisLoader = selfClass.getClassLoader();
        }catch(SecurityException e){
            logger.log(Level.WARNING, "クラスローダの取得に失敗しました", e);
            thisLoader = null;
        }
        loader = thisLoader;

        String implTitle = selfPackage.getImplementationTitle();
        if(implTitle != null){
            title = implTitle;
        }else{
            title = "Jindolf";
        }

        String implVersion = selfPackage.getImplementationVersion();
        if(implVersion != null){
            if( ! version.equals(implVersion) ){
                String message =
                        "パッケージ定義とバージョン番号が一致しません。"
                        +"["+ implVersion +"]≠["+ version +"]";
                System.err.println(message);
                runtime.exit(1);
            }
        }

        String implVendor = selfPackage.getImplementationVendor();
        if(implVendor != null){
            author = implVendor;
        }else{
            author = "olyutorskii";
        }

        copyright = "Copyright(c) 2008 " + author;

        id = title + " Ver." + version +" "+ copyright +" ("+ license +")";

        keepStock();
    }

    /**
     * クラスローダを介してリソースからの入力を生成する。
     * @param name リソース名
     * @return リソースからの入力
     */
    public static InputStream getResourceAsStream(String name){
        return selfClass.getResourceAsStream(name);
    }
    
    /**
     * クラスローダを介してリソース読み込み用URLを生成する。
     * @param name リソース名
     * @return URL
     */
    public static URL getResource(String name){
        return selfClass.getResource(name);
    }
    
    /**
     * 緊急用ヒープメモリを確保する。
     */
    public static void keepStock(){
        if(stock == null) stock = new byte[STOCK_SIZE];
        return;
    }
    
    /**
     * 緊急用ヒープメモリを放出する。
     */
    public static void freeStock(){
        stock = null;
        runtime.runFinalization();
        Thread.yield();
        runtime.gc();
        Thread.yield();
        return;
    }
    
    /**
     * 任意のクラス群に対して一括ロード／初期化を単一スレッドで順に行う。
     * どーしてもクラス初期化の順序に依存する障害が発生する場合や
     * クラス初期化のオーバーヘッドでGUIの操作性が損なわれるときなどにどうぞ。
     * 
     * @throws java.lang.LinkageError
     * @throws java.lang.ExceptionInInitializerError
     */
    private static void preLoadClass()
            throws LinkageError, ExceptionInInitializerError{
        String[] classes = {
            "java.lang.Object",
            "jp.sourceforge.jindolf.TabBrowser",
            "jp.sourceforge.jindolf.TopFrameView",
            "java.net.HttpURLConnection",
            "java.text.SimpleDateFormat",
        };

        for(String className : classes){
            try{
                if(loader != null){
                    Class.forName(className, true, loader);
                }else{
                    Class.forName(className);
                }
            }catch(ClassNotFoundException e){
                logger.log(Level.WARNING,
                           "クラスの明示的ロードに失敗しました",
                           e);
            }
        }

        return;
    }

    /**
     * JRE1.5相当のランタイムライブラリ環境が提供されているか判定する。
     * 
     * @return JRE1.5相当ならtrueを返す。
     */
    public static boolean hasRuntime5(){
        boolean result;
        try{
            result = jrePackage.isCompatibleWith("1.5");
        }catch(NumberFormatException e){
            assert false;
            return false;
        }
        return result;
    }

    /**
     * JRE1.6相当のランタイムライブラリ環境が提供されているか判定する。
     * 
     * @return JRE1.6相当ならtrueを返す。
     */
    public static boolean hasRuntime6(){
        boolean result;
        try{
            result = jrePackage.isCompatibleWith("1.6");
        }catch(NumberFormatException e){
            assert false;
            return false;
        }
        return result;
    }

    /**
     * VMごとプログラムを終了する。
     * ※おそらく随所でシャットダウンフックが起動されるはず。
     * 
     * @param exitCode 終了コード
     * @throws java.lang.SecurityException
     */
    public static void exit(int exitCode) throws SecurityException{
        try{
            logger.info(
                      "終了コード["
                    + exitCode
                    + "]でVMごとアプリケーションを終了します。");
            runtime.exit(exitCode);
        }catch(SecurityException e){
            logger.log(Level.WARNING, "VMを終了できません。", e);
            throw e;
        }
        return;
    }

    /**
     * Jindolf の実行が可能な環境でなければ、即時VM実行を終了する。
     */
    private static void testEnvironment(){
        if(!hasRuntime5()){
            logger.severe(
                    Jindolf.title
                    + " needs JRE1.5 platform or later.");
            exit(1);
        }

        if(GraphicsEnvironment.isHeadless()){
            logger.severe(
                    Jindolf.title
                    + " needs bit-map display, keyboard, & pointing-device.");
            exit(1);
        }

        if(   '狼' != 0x72fc
           || '　' != 0x3000
           || '~'  != 0x007e
           || '\\' != 0x005c
           || '～' != 0xff5e ){
            logger.severe(
                    "ソースコードの文字コードが正しくコンパイルされていないかも。"
                   +"あなたは今、オリジナル開発元の意図しない文字コード環境で"
                   +"コンパイルされたプログラムを起動しようとしているよ。"
                   +"ソースコードの入手に際して"
                   +"どのような文字コード変換が行われたか認識している？"
                   +"コンパイルオプションで正しい文字コードを指定したかな？"
                    );
            exit(1);
        }
        
        return;
    }

    /**
     * Jindolf のスタートアップエントリ
     *  
     * @param args コマンドライン引数
     */
    public static void main(final String[] args){
        AppSetting.parseOptions(args);
        
        if(AppSetting.needHelp){
            showHelpMessage();
            runtime.exit(0);
        }
        
        if(AppSetting.needVersion){
            System.out.println(id);
            runtime.exit(0);
        }

        if(AppSetting.needVMInfo){
            System.out.println(AppSetting.getVMInfo());
        }

        final Handler consoleHandler;
        if(AppSetting.consolelog){
            consoleHandler = new ConsoleHandler();
            logger.addHandler(consoleHandler);
        }else{
            consoleHandler = null;
        }
        
        if( hasRuntime6() && AppSetting.noSplash){
            System.out.println(
                    "JRE1.6以降では、"
                    +"Java実行系の方でスプラッシュ画面の非表示を指示してください"
                    );
        }
        
        testEnvironment();
        
        runtime.addShutdownHook(
            new Thread(){
                @Override
                public void run(){
                    logger.info("シャットダウン処理に入ります…");
                    
                    Handler[] handlers = logger.getHandlers();
                    for(Handler handler : handlers){
                        handler.flush();
                        if( ! (handler instanceof PileHandler) ) continue;
                        PileHandler pile = (PileHandler) handler;
                        if(consoleHandler != null){
                            pile.delegate(consoleHandler);
                        }else{
                            pile.delegate(new ConsoleHandler());
                        }
                        pile.close();
                        break;
                    }
            
                    System.out.flush();
                    System.err.flush();
                    runtime.runFinalization();
                    Thread.yield();
                    return;
                }
            }
        );
                
        logger.info(id + " が " + initDate + " に実行を開始しました。 " );
        logger.info("Max-heap:" + runtime.maxMemory()/1024/1024 + "MByte"
                + " Total-heap:"+runtime.totalMemory()/1024/1024 + "MByte");
        logger.info("\n" + AppSetting.getVMInfo());
        
        if(AppSetting.boldMetal){
            // もの凄く日本語が汚くなるかもよ！注意
            UIManager.put("swing.boldMetal", Boolean.TRUE);
        }else{
            UIManager.put("swing.boldMetal", Boolean.FALSE);
        }

        Window splashWindow = null;
        if( ! hasRuntime6() && ! AppSetting.noSplash){
            splashWindow = createSplashWindow();
            splashWindow.setVisible(true);
            GUIUtils.dispatchAWTEvent();
        }

        boolean hasError = false;
        try{
            SwingUtilities.invokeAndWait(
                new Runnable(){
                    public void run(){
                        startGUI(args);
                        return;
                    }
                }
            );
        }catch(Throwable e){
            logger.log(Level.SEVERE,
                       "アプリケーション初期化に失敗しました",
                       e);
            hasError = true;
        }finally{
            if(splashWindow != null){
                splashWindow.setVisible(false);
                splashWindow.dispose();
                splashWindow = null;
                GUIUtils.dispatchAWTEvent();
            }
        }

        if(hasError) exit(1);
        

        return;
    }

    /**
     * AWTイベントディスパッチスレッド版スタートアップエントリ
     * 
     * @param args コマンドライン引数
     */
    // TODO コマンドライン解析
    private static void startGUI(String[] args){
        preLoadClass();

        LandsModel model = new LandsModel();
        try{
            model.loadLandList();
        }catch(IOException e){
            logger.log(Level.SEVERE,
                       "国定義一覧の読み込みに失敗しました",
                       e);
            exit(1);
        }
        
        TopFrameView topFrame = buildMVC(model);

        topFrame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        topFrame.setResizable(true);
        topFrame.setIconImage(GUIUtils.getWindowIconImage());
        topFrame.pack();
        
        Dimension initGeometry = new Dimension(AppSetting.frameWidth,
                                               AppSetting.frameHeight);
        topFrame.setSize(initGeometry);
        
        if(   AppSetting.frameXpos <= Integer.MIN_VALUE
           || AppSetting.frameYpos <= Integer.MIN_VALUE ){
            topFrame.setLocationByPlatform(true);
        }else{
            topFrame.setLocation(AppSetting.frameXpos, AppSetting.frameYpos);
        }
        
        Toolkit kit = topFrame.getToolkit();
        kit.setDynamicLayout(false);

        topFrame.setVisible(true);

        return;
    }

    /**
     * スプラッシュウィンドウを作成する。
     * 
     * @return 未表示のスプラッシュウィンドウ。
     */
    private static Window createSplashWindow(){
        JWindow splashWindow = new JWindow();
        JLabel splashLabel = new JLabel(GUIUtils.getLogoIcon());
        splashWindow.add(splashLabel);

        splashWindow.pack();
        splashWindow.setLocationRelativeTo(null); // locate center

        return splashWindow;
    }

    /**
     * モデル・ビュー・コントローラの結合?
     * 
     * @param model 最上位のデータモデル
     * @return 最上位ビュー画面
     */
    private static TopFrameView buildMVC(LandsModel model){
        MenuManager menuManager = new MenuManager();
        TopFrameView topFrame = new TopFrameView();
        topFrame.setJMenuBar(menuManager.createMenuBar());
        new Controller(menuManager, topFrame, model);
        return topFrame;
    }

    /**
     * 標準出力端末にヘルプメッセージ（オプションの説明）を表示する。
     */
    private static void showHelpMessage(){
        System.out.flush();
        System.err.flush();
        
        InputStream is;
        is = getResourceAsStream("resources/help.txt");
        is = new BufferedInputStream(is);
        Reader reader = new InputStreamReader(is, CHARSET_UTF8);
        LineNumberReader lineReader = new LineNumberReader(reader);

        try{
            for(;;){
                String line;
                try{
                    line = lineReader.readLine();
                }catch(IOException e){
                    return;
                }
                if(line == null) break;
                if(line.startsWith("#")) continue;
                System.out.println(line);
            }
        }finally{
            try{
                lineReader.close();
            }catch(IOException e){
                // IGNORE
            }
        }
        
        System.out.flush();
        System.err.flush();
        
        return;
    }
    
    /**
     * hidden default constructor
     */
    private Jindolf(){
        super();
        assert false;
        throw new Error();
    }
}
