/*
 * Lands tree container
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: LandsTree.java 154 2008-09-04 16:15:41Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.border.EmptyBorder;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * 国一覧Tree周辺コンポーネント群
 */
@SuppressWarnings("serial")
public class LandsTree
        extends JPanel
        implements ActionListener, TreeSelectionListener{
    
    private static final String ascendingSymbol = "↑";
    private static final String descendingSymbol = "↓";
    private static final String ascendingTip = "押すと降順に";
    private static final String descendingTip = "押すと昇順に";
    private static final String orderTip = "選択中の国の村一覧を読み込み直す";

    private final JButton orderButton = new JButton(descendingSymbol);
    private final JButton reloadButton = new JButton("村一覧読込");
    private final JTree treeView = new JTree();

    private boolean ascending = false;
    
    public LandsTree(){
        super();

        design();
        
        this.orderButton.setToolTipText(descendingTip);
        this.orderButton.setActionCommand(MenuManager.COMMAND_SWITCHORDER);
        this.orderButton.addActionListener(this);

        this.reloadButton.setToolTipText(orderTip);
        this.reloadButton.setActionCommand(MenuManager.COMMAND_VILLAGELIST);

        TreeSelectionModel selModel = this.treeView.getSelectionModel();
        selModel.setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        this.treeView.addTreeSelectionListener(this);
        
        return;
    }

    /**
     * GUIパーツのデザインを行う。
     */
    private void design(){
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        setLayout(layout);
        
        constraints.weightx = 0.0;
        constraints.weighty = 0.0;
        constraints.anchor = GridBagConstraints.WEST;
        constraints.fill = GridBagConstraints.NONE;
        add(this.orderButton, constraints);

        constraints.gridwidth = GridBagConstraints.REMAINDER;
        add(this.reloadButton, constraints);

        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.BOTH;
        constraints.weightx = 1.0;
        constraints.weighty = 1.0;
        add(createLandSelector(), constraints);
        
        return;
    }
    
    /**
     * 国村選択ツリーコンポーネントを生成する。
     * @return 国村選択ツリーコンポーネント
     */
    private JComponent createLandSelector(){
        this.treeView.setRootVisible(false);
        this.treeView.setCellRenderer(new VillageIconRenderer());
        
        EmptyBorder border = new EmptyBorder(5, 5, 5, 5);
        this.treeView.setBorder(border);

        JScrollPane landSelector = new JScrollPane(this.treeView);

        return landSelector;
    }

    /**
     * リロードボタンを返す。
     * @return リロードボタン
     */
    public JButton getReloadVillageListButton(){
        return this.reloadButton;
    }
    
    /**
     * 国村選択ツリービューを返す。
     * @return 国村選択ツリービュー
     */
    public JTree getTreeView(){
        return this.treeView;
    }

    /**
     * 指定した国を展開する。
     * @param land 国
     */
    public void expandLand(Land land){
        TreeModel model = this.treeView.getModel();
        Object root = model.getRoot();
        Object[] path = {root, land};
        TreePath treePath = new TreePath(path);
        this.treeView.expandPath(treePath);
        updateReloadButton(land);
        return;
    }
    
    /**
     * 国の状態に応じてリロードボタンを変更する。
     * @param land 国
     */
    private void updateReloadButton(Land land){
        if(land.getVillageCount() <= 0){
            this.reloadButton.setText("村一覧読込");
        }else{
            this.reloadButton.setText("再読込");
        }
        return;
    }
    
    /**
     * 管理下のLandsModelを返す。
     * @return LandsModel
     */
    private LandsModel getLandsModel(){
        TreeModel model = this.treeView.getModel();
        if(model instanceof LandsModel){
            return (LandsModel) model;
        }
        return null;
    }
    
    /**
     * Tree表示順を反転させる。
     * @return 反転後が昇順ならtrue
     */
    private boolean toggleTreeOrder(){
        this.ascending = ! this.ascending;

        if(this.ascending){
            this.orderButton.setToolTipText(ascendingTip);
            this.orderButton.setText(ascendingSymbol);
        }else{
            this.orderButton.setToolTipText(descendingTip);
            this.orderButton.setText(descendingSymbol);
        }

        final TreePath lastPath = this.treeView.getSelectionPath();
        
        LandsModel model = getLandsModel();
        if(model != null){
            model.setAscending(this.ascending);
        }
        
        SwingUtilities.invokeLater(new Runnable(){
            public void run(){
                if(lastPath != null){
                    treeView.setSelectionPath(lastPath);
                    treeView.scrollPathToVisible(lastPath);
                }
                return;
            }
        });
        
        return this.ascending;
    }
    
    /**
     * ボタン押下処理
     * @param event ボタン押下イベント
     */
    public void actionPerformed(ActionEvent event){
        String cmd = event.getActionCommand();
        if(MenuManager.COMMAND_SWITCHORDER.equals(cmd)){
            toggleTreeOrder();
        }
        return;
    }
    
    /**
     * ツリーリストで何らかの要素（国、村）がクリックされたときの処理。
     * @param event イベント
     */
    public void valueChanged(TreeSelectionEvent event){
        TreePath path = event.getNewLeadSelectionPath();
        if(path == null){
            this.reloadButton.setEnabled(false);
            return;
        }

        Object selObj = path.getLastPathComponent();
        
        if( selObj instanceof Land ){
            Land land = (Land)selObj;
            this.reloadButton.setEnabled(true);
            updateReloadButton(land);
        }else{
            this.reloadButton.setEnabled(false);
        }

        return;
    }
    
    /**
     * L&F を変更する。
     */
    @Override
    public void updateUI(){
        super.updateUI();
        if(this.treeView != null){
            this.treeView.setCellRenderer(new VillageIconRenderer());
        }
        return;
    }
}
