/*
 * Log frame
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: LogFrame.java 113 2008-07-17 09:25:26Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Container;
import java.awt.Toolkit;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;
import javax.swing.JFrame;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.border.EmptyBorder;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.PlainDocument;

/**
 * ログ表示パネル
 */
@SuppressWarnings("serial")
public class LogFrame extends JFrame implements WindowListener{

    private static final String frameTitle = "ログ表示 - " + Jindolf.title;
    private static final int DOCLIMIT = 100 * 1000; // 単位は文字
    private static final float CHOPRATIO = 0.9f;
    private static final int CHOPPEDLEN = (int)(DOCLIMIT * CHOPRATIO);
    private static final Document dummyDoc = new PlainDocument();
    
    static{
        assert DOCLIMIT > CHOPPEDLEN;
    }
    
    /**
     * ログハンドラ
     */
    private class SwingLogger extends Handler{
        
        /**
         * ログハンドラの生成
         */
        private SwingLogger(){
            Formatter formatter = new SimpleFormatter();
            setFormatter(formatter);
            return;
        }
        
        /**
         * ログを表示する
         * @param record ログレコード
         */
        public void publish(LogRecord record){
            if( ! isLoggable(record) ){
                return;
            }
            Formatter formatter = getFormatter();
            String message = formatter.format(record);
            try{
                document.insertString(document.getLength(), message, null);
            }catch(BadLocationException e){
                // IGNORE
            }

            int docLength = document.getLength();
            if(docLength > DOCLIMIT){
                int offset = docLength - CHOPPEDLEN;
                try{
                    document.remove(0, offset);
                }catch(BadLocationException e){
                    // IGNORE
                }
            }
            
            showLastPos();
                    
            return;
        }

        /**
         * 出力フラッシュ。（何もしない）
         */
        public void flush(){
            return;
        }

        /**
         * ロギングハンドラを閉じる。
         */
        public void close(){
            setLevel(Level.OFF);
            flush();
            return;
        }
    };
    
    private final JTextArea textarea;
    private final Document document = new PlainDocument();
    private final JScrollPane scrollPane;
    private final JScrollBar vertical;
    private final Handler handler = new SwingLogger();
    
    /**
     * ログ表示パネルの生成
     */
    public LogFrame(){
        super(frameTitle);

        setResizable(true);
        Toolkit kit = getToolkit();
        kit.setDynamicLayout(false);
        setIconImage(GUIUtils.getWindowIconImage());

        this.textarea = new JTextArea();
        this.textarea.setEditable(false);
        this.textarea.setLineWrap(true);
        this.textarea.setDocument(dummyDoc);
        this.textarea.setBorder(new EmptyBorder(3, 3, 3, 3));

        this.scrollPane = new JScrollPane(this.textarea);
        this.vertical = this.scrollPane.getVerticalScrollBar();
        
        Container content = getContentPane();
        content.add(this.scrollPane);

        addWindowListener(this);
        
        return;
    }
    
    /**
     * ロギングハンドラを取得する。
     * @return ロギングハンドラ
     */
    public Handler getHandler(){
        return this.handler;
    }

    /**
     * 垂直スクロールバーを末端に設定する。
     */
    private void showLastPos(){
        if( ! isVisible() ) return;
        if(this.textarea.getDocument() != this.document) return;
        
        SwingUtilities.invokeLater(new Runnable(){
            public void run(){
                vertical.setValue(Integer.MAX_VALUE);
                return;
            }
        });
        
        return;
    }
    
    /**
     * ウィンドウの表示・非表示を設定する。
     * @param visible trueなら表示
     */
    @Override
    public void setVisible(boolean visible){
        super.setVisible(visible);

        if(visible){
            this.textarea.setDocument(this.document);
            showLastPos();
        }else{
            this.textarea.setDocument(dummyDoc);
        }
        
        return;
    }
    
    /**
     * ウィンドウイベントリスナ
     * @param event ウィンドウイベント
     */
    public void windowActivated(WindowEvent event){
        return;
    }
    
    /**
     * ウィンドウイベントリスナ
     * @param event ウィンドウイベント
     */
    public void windowDeactivated(WindowEvent event){
        return;
    }
    
    /**
     * ウィンドウイベントリスナ
     * @param event ウィンドウイベント
     */
    public void windowIconified(WindowEvent event){
        this.textarea.setDocument(dummyDoc);
        return;
    }
    
    /**
     * ウィンドウイベントリスナ
     * @param event ウィンドウイベント
     */
    public void windowDeiconified(WindowEvent event){
        this.textarea.setDocument(this.document);
        showLastPos();
        return;
    }
    
    /**
     * ウィンドウイベントリスナ
     * @param event ウィンドウイベント
     */
    public void windowOpened(WindowEvent event){
        this.textarea.setDocument(this.document);
        showLastPos();
        return;
    }

    /**
     * ウィンドウイベントリスナ
     * @param event ウィンドウイベント
     */
    public void windowClosed(WindowEvent event){
        this.textarea.setDocument(dummyDoc);
        return;
    }
    
    /**
     * ウィンドウイベントリスナ
     * @param event ウィンドウイベント
     */
    public void windowClosing(WindowEvent event){
        return;
    }
}
