/*
 * HTTP utilities
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: HttpUtils.java 187 2008-09-14 16:51:27Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.net.URLConnection;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * HTTP関連のユーティリティ群
 */
public class HttpUtils{

    private static String tokenRegex
            = "([^\\(\\)<>@,;:\\\"/\\[\\]\\?=\\{\\}\\p{Blank}\\p{Cntrl}]+)";
    private static String mtypeRegex =
            "[\\p{Blank}]*"
            + tokenRegex + "/" + tokenRegex
            + "[\\p{Blank}]*";
    private static String paramRegex =
            "[\\p{Blank}]*;[\\p{Blank}]*"
            + tokenRegex
            + "[\\p{Blank}]*=[\\p{Blank}]*"
            + "(" + tokenRegex + "|" + "(\"[^\\p{Cntrl}\\\"]*\")" + ")";
    private static Pattern mtypePattern = Pattern.compile(mtypeRegex);
    private static Pattern attrPattern = Pattern.compile(paramRegex);

    /**
     * ユーザエージェント名を返す。
     * @return ユーザエージェント名
     */
    public static String getUserAgentName(){
        StringBuilder result = new StringBuilder();
        result.append(Jindolf.TITLE).append("/").append(Jindolf.VERSION);

        StringBuilder rawComment = new StringBuilder();
        if(AppSetting.osName != null){
            if(rawComment.length() > 0) rawComment.append("; ");
            rawComment.append(AppSetting.osName);
        }
        if(AppSetting.osVersion != null){
            if(rawComment.length() > 0) rawComment.append("; ");
            rawComment.append(AppSetting.osVersion);
        }
        if(AppSetting.osArch != null){
            if(rawComment.length() > 0) rawComment.append("; ");
            rawComment.append(AppSetting.osArch);
        }
        if(AppSetting.javaVendor != null){
            if(rawComment.length() > 0) rawComment.append("; ");
            rawComment.append(AppSetting.javaVendor);
        }
        if(AppSetting.javaVersion != null){
            if(rawComment.length() > 0) rawComment.append("; ");
            rawComment.append(AppSetting.javaVersion);
        }
        
        CharSequence comment = escapeHttpComment(rawComment);
        if(comment != null) result.append(" ").append(comment);

        return result.toString();
    }

    /**
     * 与えられた文字列からHTTPコメントを生成する。
     * @param comment コメント
     * @return HTTPコメント
     */
    public static String escapeHttpComment(CharSequence comment){
        if(comment == null) return null;
        if(comment.length() <= 0) return null;
        
        String result = comment.toString();
        result = result.replaceAll("\\(", "\\\\(");
        result = result.replaceAll("\\)", "\\\\)");
        result = result.replaceAll("[\\u0000-\\u001f]", "?");
        result = result.replaceAll("[\\u007f-\\uffff]", "?");
        result = "(" + result + ")";
        
        return result;
    }
    
    /**
     * HTTP応答からCharsetを取得する。
     * @param connection HTTP接続
     * @return Charset文字列
     */
    public static String getHTMLCharset(URLConnection connection){
        String contentType = connection.getContentType();
        if(contentType == null) return null;
        return getHTMLCharset(contentType);
    }

    /**
     * ContentTypeからCharsetを取得する。
     * @param contentType ContentType
     * @return Charset文字列
     */
    public static String getHTMLCharset(String contentType){
        Matcher matcher;
        boolean matchResult;
        int lastPos;

        matcher = mtypePattern.matcher(contentType);
        matchResult = matcher.lookingAt();
        if(!matchResult) return null;
        lastPos = matcher.end();
        String type = matcher.group(1);
        String subtype = matcher.group(2);

        if(!type.equalsIgnoreCase("text")) return null;
        if(!subtype.equalsIgnoreCase("html")) return null;

        matcher.usePattern(attrPattern);

        String charset = null;
        for(;;){
            matchResult = matcher.find(lastPos);
            if(!matchResult) break;
            lastPos = matcher.end();
            String attribute = matcher.group(1);
            String value = matcher.group(2);
            if(attribute.equalsIgnoreCase("charset")) charset = value;
        }
        return charset;
    }
    
    /**
     * 隠れコンストラクタ
     */
    private HttpUtils(){
        super();
        assert false;
        throw new Error();
    }
}
