/*
 * font chooser
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: FontChooser.java 255 2008-10-11 20:10:29Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsEnvironment;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.awt.font.FontRenderContext;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

/**
 * 発言表示フォント選択パネル
 */
@SuppressWarnings("serial")
public class FontChooser extends JDialog
        implements ListSelectionListener,
                   ActionListener,
                   ItemListener,
                   WindowListener {

    public static final Font defaultFont;
    public static final FontRenderContext defaultRenderContext;

    private static final String FRAMETITLE =
            "発言表示フォントの選択 - " + Jindolf.TITLE;
    private static final String[] initFamilyNames = {
        "Hiragino Kaku Gothic Pro",  // for MacOS X
        "Hiragino Kaku Gothic Std",
        "Osaka",
        "ＭＳ Ｐゴシック",           // for WinXP
        "ＭＳ ゴシック",
        // TODO X用のおすすめは？
    };
    private static final Integer[] pointSizes = {
        8, 10, 12, 16, 18, 24, 32, 36, 48, 72,  // TODO これで十分？
    };
    private static final int defaultFontStyle = 0 | Font.PLAIN;
    private static final int defaultFontSize = 16;
    private static final SortedSet<String> familySet;
    private static final CharSequence previewContent;

    /** JIS0208:1990 チェック用 */
    private static final String CHECK_CODE = "9Aあゑアｱヴヰ┼ЖΩ峠凜熙";

    static{
        familySet            = createFontSet();
        defaultFont          = createDefaultFont();
        defaultRenderContext = createDefaultFontRenderContext();

        CharSequence resourceText;
        try{
            resourceText = Jindolf.loadResourceText("resources/preview.txt");
        }catch(IOException e){
            resourceText = "ABC";
        }
        previewContent = resourceText;
    }

    /**
     * ソートされたフォントファミリ一覧表を生成する。
     * JISX0208:1990相当が表示できないファミリは弾かれる。
     * @return フォント一覧
     */
    private static SortedSet<String> createFontSet(){
        GraphicsEnvironment ge;
        ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        Font[] allFonts = ge.getAllFonts();
        SortedSet<String> result = new TreeSet<String>();
        for(Font font : allFonts){
            if(font.canDisplayUpTo(CHECK_CODE) >= 0) continue;
            String familyName = font.getFamily();
            result.add(familyName.intern());
        }
        return result;
    }

    /**
     * デフォルトフォントを生成する。
     * 適当なファミリが見つからなかったら"Dialog"が選択される。
     * @return デフォルトフォント
     */
    private static Font createDefaultFont(){
        String defaultFamilyName = "Dialog";
        for(String familyName : initFamilyNames){
            Font dummyFont = new Font(familyName, 0 | Font.PLAIN, 1);
            String dummyFontName = dummyFont.getFamily();
            if(dummyFontName.equals(familyName)){
                defaultFamilyName = dummyFontName;
                break;
            }
        }
        Font result = new Font(defaultFamilyName,
                               defaultFontStyle,
                               defaultFontSize);
        return result;
    }

    /**
     * デフォルトのフォント描画情報を生成する。
     * @return フォント描画情報。
     */
    private static FontRenderContext createDefaultFontRenderContext(){
        boolean isAntiAliased         = true;
        boolean usesFractionalMetrics = true;

        if(guessBitmapFont(defaultFont)){
            isAntiAliased         = false;
            usesFractionalMetrics = false;
        }

        FontRenderContext result;
        result = new FontRenderContext(GUIUtils.AFFINETX_IDENTITY,
                                       isAntiAliased,
                                       usesFractionalMetrics );

        return result;
    }

    /**
     * ビットマップフォントか否か見当をつける。
     * ビットマップフォントにはアンチエイリアスやサブピクセルを使わないほうが
     * 見栄えがいいような気がする。
     * @param font 判定対象フォント
     * @return ビットマップフォントらしかったらtrue
     */
    public static boolean guessBitmapFont(Font font){
        String familyName = font.getFamily();
        if(   font.getSize() < 24
           && familyName.startsWith("ＭＳ")
           && (   familyName.contains("ゴシック")
               || familyName.contains("明朝")    ) ){
            return true;
        }
        return false;
    }

    private Font              selectedFont  = defaultFont;
    private FontRenderContext renderContext = defaultRenderContext;
    private Font              lastFont    = this.selectedFont;
    private FontRenderContext lastContext = this.renderContext;

    private final JList familySelector;
    private final JComboBox sizeSelector;
    private final JCheckBox isBoldCheck;
    private final JCheckBox isItalicCheck;
    private final JCheckBox useTextAntiAliaseCheck;
    private final JCheckBox useFractionalCheck;
    private final JLabel maxBounds;
    private final JTextField invokeOption;
    private final FontPreview preview;
    private final JButton applyButton;
    private final JButton cancelButton;

    private boolean maskListener = false;
    private boolean isCanceled = false;

    /**
     * コンストラクタ
     */
    public FontChooser(Frame owner){
        super(owner, FRAMETITLE, true);   // モーダルダイアログ

        Jindolf.logger.info(
                  "デフォルトの発言表示フォントに"
                + this.selectedFont
                + "が選択されました" );
        Jindolf.logger.info(
                  "発言表示のアンチエイリアス指定に"
                + this.renderContext.isAntiAliased()
                + "が指定されました" );
        Jindolf.logger.info(
                  "発言表示のFractional指定に"
                + this.renderContext.usesFractionalMetrics()
                + "が指定されました" );

        GUIUtils.modifyWindowAttributes(this, true, false, true);

        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        addWindowListener(this);

        Vector<String> familyList = new Vector<String>(familySet);
        this.familySelector = new JList(familyList);
        this.familySelector.setVisibleRowCount(-1);
        this.familySelector
            .setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        this.sizeSelector = new JComboBox();
        this.sizeSelector.setEditable(true);
        this.sizeSelector.setActionCommand(ActionManager.COMMAND_FONTSIZESEL);
        for(Integer size : pointSizes){
            this.sizeSelector.addItem(size);
        }

        this.isBoldCheck            = new JCheckBox("ボールド");
        this.isItalicCheck          = new JCheckBox("イタリック");
        this.useTextAntiAliaseCheck = new JCheckBox("アンチエイリアス");
        this.useFractionalCheck     = new JCheckBox("サブピクセル精度");

        this.maxBounds = new JLabel();

        this.invokeOption = new JTextField();
        this.invokeOption.setEditable(false);
        Font oldFont = this.invokeOption.getFont();
        Font newFont = new Font("Monospaced",
                                oldFont.getStyle(),
                                oldFont.getSize() );
        this.invokeOption.setFont(newFont);

        this.preview = new FontPreview(previewContent,
                                       this.selectedFont,
                                       this.renderContext );

        this.applyButton  = new JButton("適用する");
        this.cancelButton = new JButton("キャンセル");
        this.applyButton .setActionCommand(ActionManager.COMMAND_FONTAPPLY);
        this.cancelButton.setActionCommand(ActionManager.COMMAND_FONTCANCEL);
        getRootPane().setDefaultButton(this.applyButton);
        
        design();
        updateControlls();
        updatePreview();

        this.familySelector.addListSelectionListener(this);
        this.sizeSelector  .addActionListener(this);

        this.isBoldCheck           .addItemListener(this);
        this.isItalicCheck         .addItemListener(this);
        this.useTextAntiAliaseCheck.addItemListener(this);
        this.useFractionalCheck    .addItemListener(this);

        this.applyButton .addActionListener(this);
        this.cancelButton.addActionListener(this);

        return;
    }

    /**
     * GUIのデザイン、レイアウトを行う。
     */
    private void design(){
        Container content = getContentPane();

        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();

        content.setLayout(layout);

        Border border;
        JPanel panel;

        JComponent fontPref = createFontPrefPanel();

        constraints.insets = new Insets(5, 5, 5, 5);

        constraints.weightx = 1.0;
        constraints.weighty = 0.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.BOTH;
        content.add(fontPref, constraints);

        border = BorderFactory.createTitledBorder("起動オプション");
        panel = new JPanel();
        panel.setLayout(new BorderLayout());
        panel.add(this.invokeOption, BorderLayout.CENTER);
        panel.setBorder(border);
        content.add(panel, constraints);

        constraints.weightx = 1.0;
        constraints.weighty = 1.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.BOTH;
        border = BorderFactory.createTitledBorder("プレビュー");
        panel = new JPanel();
        panel.add(this.preview);
        panel.setBorder(border);
        content.add(createPreviewPanel(), constraints);

        constraints.weightx = 1.0;
        constraints.weighty = 0.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.BOTH;
        content.add(this.maxBounds, constraints);

        constraints.weightx = 1.0;
        constraints.weighty = 0.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.HORIZONTAL;
        content.add(new JSeparator(), constraints);

        constraints.weightx = 1.0;
        constraints.weighty = 0.0;
        constraints.gridwidth = 1;
        constraints.fill = GridBagConstraints.NONE;
        constraints.anchor = GridBagConstraints.EAST;
        content.add(this.applyButton, constraints);
        constraints.weightx = 0.0;
        content.add(this.cancelButton, constraints);

        return;
    }

    /**
     * フォント設定画面を生成する。
     * @return フォント設定画面
     */
    private JComponent createFontPrefPanel(){
        JPanel result = new JPanel();

        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        result.setLayout(layout);

        Border border;

        constraints.insets = new Insets(0, 0, 0, 5);
        constraints.weightx = 1.0;
        constraints.weighty = 0.0;
        constraints.gridheight = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.BOTH;
        border = BorderFactory.createEmptyBorder(1, 1, 1, 1);
        this.familySelector.setBorder(border);
        JScrollPane familyScroller = new JScrollPane(this.familySelector);
        border = BorderFactory.createTitledBorder("フォントファミリ選択");
        JPanel familyPanel = new JPanel();
        familyPanel.setLayout(new BorderLayout());
        familyPanel.add(familyScroller, BorderLayout.CENTER);
        familyPanel.setBorder(border);
        result.add(familyPanel, constraints);

        constraints.insets = new Insets(0, 0, 0, 0);
        constraints.weightx = 0.0;
        constraints.gridheight = 1;
        constraints.fill = GridBagConstraints.HORIZONTAL;
        constraints.anchor = GridBagConstraints.WEST;

        border = BorderFactory.createTitledBorder("ポイントサイズ指定");
        JPanel panel = new JPanel();
        panel.add(this.sizeSelector);
        panel.setBorder(border);
        result.add(panel, constraints);

        constraints.anchor = GridBagConstraints.NORTHWEST;
        result.add(this.isBoldCheck, constraints);
        result.add(this.isItalicCheck, constraints);
        result.add(this.useTextAntiAliaseCheck, constraints);
        result.add(this.useFractionalCheck, constraints);

        return result;
    }

    /**
     * プレビュー画面を生成する。
     * @return プレビュー画面
     */
    private JComponent createPreviewPanel(){
        JPanel result = new JPanel();

        JScrollPane scroller = new JScrollPane(this.preview);
        scroller.getVerticalScrollBar().setUnitIncrement(8);

        Border border;
        border = BorderFactory.createTitledBorder("プレビュー");
        result.setBorder(border);
        result.setLayout(new BorderLayout());
        result.add(scroller, BorderLayout.CENTER);

        return result;
    }

    /**
     * 選択されたフォントを返す。
     * @return フォント
     */
    public Font getSelectedFont(){
        return this.selectedFont;
    }

    /**
     * フォントを更新する。
     * @param font フォント
     */
    public void setSelectedFont(Font font){
        this.selectedFont = font;
        updateControlls();
        updatePreview();
        return;
    }

    /**
     * 設定されたフォント描画設定を返す。
     * @return 描画設定
     */
    public FontRenderContext getFontRenderContext(){
        return this.renderContext;
    }

    /**
     * フォント描画設定を更新する。
     * @param renderContext フォント描画設定
     */
    public void setFontRenderContext(FontRenderContext renderContext){
        this.renderContext = renderContext;
        updateControlls();
        updatePreview();
        return;
    }

    /**
     * フォント設定に合わせてプレビュー画面を更新する。
     */
    private void updatePreview(){
        this.preview.setFontInfo(this.selectedFont, this.renderContext);
        return;
    }

    /**
     * フォント設定に合わせてGUIを更新する。
     */
    private void updateControlls(){
        this.maskListener = true;

        String defaultFamily = this.selectedFont.getFamily().intern();
        this.familySelector.setSelectedValue(defaultFamily, true);

        Integer selectedInteger = new Integer(this.selectedFont.getSize());
        this.sizeSelector.setSelectedItem(selectedInteger);
        int sizeItems = this.sizeSelector.getItemCount();
        for(int index = 0; index <= sizeItems - 1; index++){
            Object sizeItem = this.sizeSelector.getItemAt(index);
            if(sizeItem.equals(selectedInteger)){
                this.sizeSelector.setSelectedIndex(index);
                break;
            }
        }

        this.isBoldCheck  .setSelected(this.selectedFont.isBold());
        this.isItalicCheck.setSelected(this.selectedFont.isItalic());

        this.useTextAntiAliaseCheck
            .setSelected(this.renderContext.isAntiAliased());
        this.useFractionalCheck
            .setSelected(this.renderContext.usesFractionalMetrics());

        StringBuilder arg = new StringBuilder();

        StringBuilder styleArg = new StringBuilder();
        if(this.selectedFont.isBold()) styleArg.append("BOLD");
        if(this.selectedFont.isItalic()) styleArg.append("ITALIC");
        if(styleArg.length() <= 0) styleArg.append("PLAIN");

        StringBuilder fontid = new StringBuilder();
        fontid.append(this.selectedFont.getFamily());
        fontid.append('-').append(styleArg);
        fontid.append('-').append(this.selectedFont.getSize());
        if(fontid.indexOf(" ") >= 0) fontid.insert(0, '"').append('"');

        arg.append("-initfont ").append(fontid);

        arg.append(" -antialias ");
        if(this.renderContext.isAntiAliased()) arg.append("on");
        else                                   arg.append("off");

        arg.append(" -fractional ");
        if(this.renderContext.usesFractionalMetrics()) arg.append("on");
        else                                           arg.append("off");

        this.invokeOption.setText(arg.toString());
        this.invokeOption.setCaretPosition(0);

        Rectangle2D r2d = this.selectedFont
                              .getMaxCharBounds(this.renderContext);
        Rectangle rect = r2d.getBounds();
        String boundInfo =  "最大文字寸法 : "
                          + rect.width
                          + " pixel幅 × "
                          + rect.height
                          + " pixel高";
        this.maxBounds.setText(boundInfo);

        this.maskListener = false;

        return;
    }

    /**
     * ダイアログの表示・非表示。
     * ダイアログが閉じられるまで制御を返さない。
     * @param isVisible trueなら表示
     */
    @Override
    public void setVisible(boolean isVisible){
        if(isVisible){
            updateControlls();
            updatePreview();
        }
        this.lastFont    = this.selectedFont;
        this.lastContext = this.renderContext;

        super.setVisible(isVisible);

        return;
    }

    /**
     * チェックボックス操作のリスナ
     * @param event 操作イベント
     */
    public void itemStateChanged(ItemEvent event){
        if(this.maskListener) return;

        Object source = event.getSource();

        if(   source != this.isBoldCheck
           && source != this.isItalicCheck
           && source != this.useTextAntiAliaseCheck
           && source != this.useFractionalCheck     ){
            return;
        }

        int style = 0 | Font.PLAIN;
        if(this.isBoldCheck.isSelected()){
            style = style | Font.BOLD;
        }
        if(this.isItalicCheck.isSelected()){
            style = style | Font.ITALIC;
        }
        if(this.selectedFont.getStyle() != style){
            this.selectedFont = this.selectedFont.deriveFont(style);
        }

        boolean isAntiAliases = this.useTextAntiAliaseCheck.isSelected();
        boolean useFractional = this.useFractionalCheck.isSelected();
        AffineTransform tx = this.renderContext.getTransform();

        this.renderContext = new FontRenderContext(tx,
                                                   isAntiAliases,
                                                   useFractional );

        updateControlls();
        updatePreview();

        return;
    }

    /**
     * ダイアログが閉じられた原因を判定する。
     * @return キャンセルもしくはクローズボタンでダイアログが閉じられたらtrue
     */
    public boolean isCanceled(){
        return this.isCanceled;
    }

    /**
     * 適用ボタン押下処理
     */
    private void actionApply(){
        this.isCanceled = false;
        setVisible(false);
        return;
    }

    /**
     * キャンセルボタン押下処理
     */
    private void actionCancel(){
        this.isCanceled = true;
        this.selectedFont  = this.lastFont;
        this.renderContext = this.lastContext;
        setVisible(false);
        return;
    }

    /**
     * フォントサイズ変更処理
     */
    private void actionFontSizeSelected(){
        Object selected = this.sizeSelector.getSelectedItem();
        if(selected == null) return;

        Integer selectedInteger;
        if(selected instanceof Integer){
            selectedInteger = (Integer) selected;
        }else{
            try{
                selectedInteger = new Integer(selected.toString());
            }catch(NumberFormatException e){
                selectedInteger = new Integer(this.lastFont.getSize());
            }
        }

        if(selectedInteger.intValue() <= 0){
            selectedInteger = new Integer(this.lastFont.getSize());
        }

        float fontSize = selectedInteger.floatValue();
        this.selectedFont = this.selectedFont.deriveFont(fontSize);

        int sizeItems = this.sizeSelector.getItemCount();
        for(int index = 0; index <= sizeItems - 1; index++){
            Object sizeItem = this.sizeSelector.getItemAt(index);
            if(sizeItem.equals(selectedInteger)){
                this.sizeSelector.setSelectedIndex(index);
                break;
            }
        }

        updateControlls();
        updatePreview();
    }

    /**
     * ボタン操作及びフォントサイズ指定コンボボックス操作のリスナ
     * @param event 操作イベント
     */
    public void actionPerformed(ActionEvent event){
        if(this.maskListener) return;

        String cmd = event.getActionCommand();
        if      (cmd.equals(ActionManager.COMMAND_FONTAPPLY)){
            actionApply();
        }else if(cmd.equals(ActionManager.COMMAND_FONTCANCEL)){
            actionCancel();
        }else if(cmd.equals(ActionManager.COMMAND_FONTSIZESEL)){
            actionFontSizeSelected();
        }

        return;
    }

    /**
     * フォントファミリリスト選択操作のリスナ
     * @param event 操作イベント
     */
    public void valueChanged(ListSelectionEvent event){
        if(this.maskListener) return;

        if(event.getSource() != this.familySelector) return;
        if(event.getValueIsAdjusting()) return;

        Object selected = this.familySelector.getSelectedValue();
        if(selected == null) return;

        String familyName = selected.toString();
        int style = this.selectedFont.getStyle();
        int size = this.selectedFont.getSize();

        this.selectedFont = new Font(familyName, style, size);

        updateControlls();
        updatePreview();

        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowActivated(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowDeactivated(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowIconified(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowDeiconified(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowOpened(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ。
     * ダイアログのクローズボタン押下処理を行う。
     * @param event ウィンドウ変化イベント
     */
    public void windowClosing(WindowEvent event){
        actionCancel();
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowClosed(WindowEvent event){
        return;
    }

    /**
     * フォントプレビュー画面用コンポーネント。
     */
    private static class FontPreview extends JComponent{

        private static int MARGIN = 5;

        private final GlyphDraw draw;

        private Font font;
        private FontRenderContext renderContext;

        /**
         * コンストラクタ
         * @param source 文字列
         * @param font フォント
         * @param renderContext フォント描画設定
         */
        public FontPreview(CharSequence source,
                             Font font,
                             FontRenderContext renderContext){
            super();

            this.font = font;
            this.renderContext = renderContext;
            this.draw = new GlyphDraw(this.font, this.renderContext, source);

            this.draw.setPos(MARGIN, MARGIN);

            this.draw.setColor(Color.BLACK);
            setBackground(Color.WHITE);

            updateBounds();

            return;
        }

        /**
         * サイズ更新
         */
        private void updateBounds(){
            Rectangle bounds = this.draw.setWidth(Integer.MAX_VALUE);
            Dimension dimension = new Dimension(bounds.width  + MARGIN * 2,
                                                bounds.height + MARGIN * 2 );
            setPreferredSize(dimension);
            revalidate();
            repaint();
            return;
        }

        /**
         * フォント設定の変更
         * @param font フォント
         * @param renderContext フォント描画設定
         */
        public void setFontInfo(Font font, FontRenderContext renderContext){
            this.font = font;
            this.renderContext = renderContext;
            this.draw.setFontInfo(this.font, this.renderContext);

            updateBounds();

            return;
        }

        /**
         * 文字列の描画
         * @param g グラフィックコンテキスト
         */
        @Override
        public void paintComponent(Graphics g){
            super.paintComponent(g);
            Graphics2D g2d = (Graphics2D) g;
            this.draw.paint(g2d);
            return;
        }
    }
}
