/*
 * land
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Land.java 247 2008-10-10 11:10:06Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Image;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.Vector;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.imageio.ImageIO;

/**
 * いわゆる「国」
 */
public class Land {

    /**
     * 国種別
     */
    public static enum Type {
        /** アクセス不可 */
        CLOSED,
        /** 過去ログの閲覧のみが可能 */
        HISTORICAL,
        /** ゲームへの参加が可能 */
        ACTIVE,
    };

    private static final String Spchar = "\u0020\\t\\n\\r";
    private static final String Sp = "[" +Spchar+ "]";
    private static final String Sp_n = Sp + "*";
    private static final String Sp_m = Sp + "+";

    private static final Pattern iso8601Regex;
    private static final Pattern anchorRegex;

    public static final Date DATE_UNKNOWN = new Date(-1L);

    static{
        String year = "([0-9][0-9][0-9][0-9])";
        String month = "([0-1][0-9])";
        String day = "([0-3][0-9])";
        String hour = "([0-2][0-9])";
        String minute = "([0-5][0-9])";
        String second = "([0-6][0-9])";
        String timezone =
                "("+
                    "[\\+\\-][0-2][0-9]"+
                    "(?:"+ ":?[0-5][0-9]" +")?"+
                "|"+
                    "Z"+
                ")";
        String iso8601 =
                year +"\\-"+ month +"\\-"+ day
                +"T"+
                hour +":"+ minute +":"+ second
                +timezone;

        String anchorStart = "<a" +Sp_m+ "href=\"([^\"]*)\"" +Sp_n+ ">";
        String anchorClose = "</a>";
        String deadline = Sp_n+ "<strong>[^<]*</strong>" +Sp_n+ "</td>";
        String status = Sp_n+ "<td>([^<]*)</td>";
        String option = "(?:"+deadline+status+")?";
        String anchor = anchorStart +"([^<]*)"+ anchorClose + option;

        iso8601Regex = Pattern.compile(iso8601);
        anchorRegex = Pattern.compile(anchor, Pattern.DOTALL);
    }

    private final String landName;
    private final String landIdentifier;
    private final String landPrefix;
    private final URL baseURL;
    private final Date startDate;
    private final Date endDate;
    private final Type type;
    private final Locale locale;
    private final TimeZone timeZone;
    private final String description;
    private final String contactInfo;
    private final ServerAccess serverAccess;

    private final List<Village> villageList = new LinkedList<Village>();

    /**
     * ISO8601形式の日付をDateに変換する。
     * @param date ISO8601形式の日付文字列
     * @return Dateインスタンス
     * @throws java.lang.IllegalArgumentException 形式が変な場合。
     */
    private static Date parseISO8601(CharSequence date)
            throws IllegalArgumentException{
        Matcher matcher = iso8601Regex.matcher(date);
        if( ! matcher.lookingAt() ){
            throw new IllegalArgumentException(date.toString());
        }

        String yearStr   = matcher.group(1);
        String monthStr  = matcher.group(2);
        String dayStr    = matcher.group(3);
        String hourStr   = matcher.group(4);
        String minuteStr = matcher.group(5);
        String secondStr = matcher.group(6);
        String tzString  = matcher.group(7);

        int year;
        int month;
        int day;
        int hour;
        int minute;
        int second;
        try{
            year   = Integer.parseInt(yearStr);
            month  = Integer.parseInt(monthStr);
            day    = Integer.parseInt(dayStr);
            hour   = Integer.parseInt(hourStr);
            minute = Integer.parseInt(minuteStr);
            second = Integer.parseInt(secondStr);
        }catch(NumberFormatException e){
            throw new IllegalArgumentException(date.toString(), e);
        }

        String tzID = "GMT";
        if( tzString.compareToIgnoreCase("Z") == 0 ) tzID += "+00:00";
        else                                         tzID += tzString;
        TimeZone timezone = TimeZone.getTimeZone(tzID);

        GregorianCalendar calendar = new GregorianCalendar();
        calendar.clear();
        calendar.setTimeZone(timezone);
        calendar.set(year, month-1, day, hour, minute, second);

        Date result = calendar.getTime();

        return result;
    }

    /**
     * 国を生成する。
     * @param landName          GUIに出す国の名前
     * @param landIdentifier    ファイル名やURLの一部に用いる前提の識別子
     * @param landPrefix        村IDの前につける接頭子 '1321' -> 'F1321'
     * @param baseURL           URLのベース
     * @param startDate         最初の村生成開始日
     * @param endDate           最後の村終了日
     * @param type             国の状態
     * @param locale            CGIがXHTML上に出力する内容のロケール
     * @param timeZone          CGIがXHTML上に出力する時刻のタイムゾーン
     * @param description       少し長めの説明
     * @param contactInfo       サーバ連絡先
     */
    public Land(String landName,
                  String landIdentifier,
                  String landPrefix,
                  URL baseURL,
                  Date startDate,
                  Date endDate,
                  Type type,
                  Locale locale,
                  TimeZone timeZone,
                  String description,
                  String contactInfo) {
        this.landName = landName;
        this.landIdentifier = landIdentifier;
        this.landPrefix = landPrefix;
        this.baseURL = baseURL;
        this.startDate = startDate;
        this.endDate = endDate;
        this.type = type;
        this.locale = locale;
        this.timeZone = timeZone;
        this.description = description;
        this.contactInfo = contactInfo;
        this.serverAccess = new ServerAccess(baseURL);

        return;
    }

    /**
     * プロパティ定義に従い国を生成する。
     * @param landId 国の識別子
     * @param properties プロパティ
     * @throws java.lang.NullPointerException 引数がnullの場合
     * @throws java.lang.IllegalArgumentException 形式が変な場合
     */
    public Land(String landId, Properties properties)
            throws NullPointerException, IllegalArgumentException{
        super();

        if(landId == null || properties == null){
            throw new NullPointerException();
        }
        if( landId.length() <= 0 ){
            throw new IllegalArgumentException();
        }
        this.landIdentifier = landId;

        String value;

        value = properties.getProperty(landId +".landname");
        if( value == null || value.length() <= 0 ){
            throw new IllegalArgumentException();
        }
        this.landName = value;

        value = properties.getProperty(landId +".prefix", "");
        this.landPrefix = value;

        value = properties.getProperty(landId +".baseurl");
        if( value == null || value.length() <= 0 ){
            throw new IllegalArgumentException();
        }
        URL url;
        try{
            url = new URL(value);
        }catch(MalformedURLException e){
            throw new IllegalArgumentException(value, e);
        }
        this.baseURL = url;

        Date date;

        value = properties.getProperty(landId +".startdate");
        if( value == null || value.length() <= 0 ){
            date = DATE_UNKNOWN;
        }else{
            date = parseISO8601(value);
        }
        this.startDate = date;

        value = properties.getProperty(landId +".enddate");
        if( value == null || value.length() <= 0 ){
            date = DATE_UNKNOWN;
        }else{
            date = parseISO8601(value);
        }
        this.endDate = date;

        value = properties.getProperty(landId +".contact", "");
        this.contactInfo = value;

        value = properties.getProperty(landId +".description", "");
        this.description = value;

        String lang    = "";
        String country = "";
        String variant = "";
        value = properties.getProperty(landId +".locale");
        if( value == null || value.length() <= 0 ){
            throw new IllegalArgumentException();
        }
        String[] lcstr = value.split("_", 3);
        if(lcstr.length >= 1) lang    = lcstr[0];
        if(lcstr.length >= 2) country = lcstr[1];
        if(lcstr.length >= 3) variant = lcstr[2];
        this.locale = new Locale(lang, country, variant);

        value = properties.getProperty(landId +".timezone");
        if( value == null || value.length() <= 0 ){
            throw new IllegalArgumentException();
        }
        this.timeZone = TimeZone.getTimeZone(value);

        value = properties.getProperty(landId +".state");
        if( value == null || value.length() <= 0 ){
            throw new IllegalArgumentException();
        }
        if( value.compareToIgnoreCase("CLOSED") == 0){
            this.type = Type.CLOSED;
        }else if( value.compareToIgnoreCase("HISTORICAL") == 0){
            this.type = Type.HISTORICAL;
        }else if( value.compareToIgnoreCase("ACTIVE") == 0){
            this.type = Type.ACTIVE;
        }else{
            throw new IllegalArgumentException(value);
        }

        this.serverAccess = new ServerAccess(this.baseURL);

        return;
    }

    /**
     * 国名を返す
     * @return 国名
     */
    public String getLandName() {
        return this.landName;
    }

    /**
     * 国の識別子を返す
     * @return 識別子
     */
    public String getLandIdentifier() {
        return this.landIdentifier;
    }

    /**
     * 国のPrefixを返す
     * @return Prefix
     */
    public String getLandPrefix() {
        return this.landPrefix;
    }

    /**
     * 国の開始日を返す
     * @return 開始日
     */
    public Date getStartDate() {
        return this.startDate;
    }

    /**
     * 国の終了日を返す
     * @return 終了日
     */
    public Date getEndDate() {
        return this.endDate;
    }

    /**
     * 国種別を返す。
     * @return 種別
     */
    public Type getType() {
        return this.type;
    }

    /**
     * 国が属するロケールを返す。
     * @return ロケール
     */
    public Locale getLocale() {
        return this.locale;
    }

    /**
     * 国が用いるタイムゾーンを返す。
     * @return タイムゾーン
     */
    public TimeZone getTimeZone() {
        return this.timeZone;
    }

    /**
     * 国の説明を返す。
     * @return 説明
     */
    public String getDescription() {
        return this.description;
    }

    /**
     * 国の連絡先を返す。
     * @return 連絡先
     */
    public String getContactInfo() {
        return this.contactInfo;
    }

    /**
     * サーバ接続を返す。
     * @return ServerAccessインスタンス
     */
    public ServerAccess getServerAccess(){
        return this.serverAccess;
    }

    /**
     * 指定されたインデックス位置の村を返す。
     * @param index 0から始まるインデックス値
     * @return 村
     */
    public Village getVillage(int index){
        if(index < 0)                  return null;
        if(index >= getVillageCount()) return null;

        Village result = this.villageList.get(index);
        return result;
    }

    /**
     * 村の総数を返す。
     * @return 村の総数
     */
    public int getVillageCount(){
        int result = this.villageList.size();
        return result;
    }

    /**
     * 村のリストを返す。
     * @return 村のリスト
     */
    // TODO インスタンス変数でいいはず。
    public List<Village> getVillageList(){
        return Collections.unmodifiableList(this.villageList);
    }

    /**
     * 相対URLからイメージ画像をロードする。
     * @param relativeSrc 相対URLの文字列表記
     * @return イメージ
     */
    // TODO ServerAccessに統合したい
    // A,B,D 国は絶対パスらしい
    public Image getImageFromRelativeSrc(String relativeSrc){
        URL url;
        try{
            url = new URL(this.baseURL, relativeSrc);
        }catch(MalformedURLException e){
            assert false;
            return null;
        }

        Image image;
        try{
            image = ImageIO.read(url);
            Jindolf.logger.info("イメージ "
                                + url.toString()
                                + " をダウンロードしました" );
        }catch(IOException e){
            Jindolf.logger.log(Level.WARNING,
                               "イメージのダウンロードに失敗しました",
                               e);
            image = null;  // TODO 代替イメージの提供が必要か
        }

        return image;
    }

    /**
     * クエリー文字列から特定キーの値を得る。
     * クエリーの書式例：「a=b&c=d&e=f」この場合キーcの値はd
     * @param key キー
     * @param allQuery クエリー
     * @return 値
     */
    public static String getValueFromCGIQueries(String key,
                                                   String allQuery){
        String result = null;

        String[] queries = allQuery.split("\\Q&\\E");
        if(queries == null) return null;

        for(String pair : queries){
            if(pair == null) continue;
            String[] namevalue = pair.split("\\Q=\\E");
            if(namevalue == null) continue;
            if(namevalue.length != 2) continue;
            String name  = namevalue[0];
            String value = namevalue[1];
            if(name == null) continue;
            if( name.equals(key) ){
                result = value;
                if(result == null) continue;
                if(result.length() <= 0) continue;
                break;
            }
        }

        return result;
    }

    /**
     * AタグのHREF属性値から村IDを得る。
     * @param hrefValue HREF値
     * @return village 村ID
     */
    public static String getVillageIDFromHREF(CharSequence hrefValue){
        String allQuery = NetUtil.getQueryFromHREF(hrefValue);
        if(allQuery == null) return null;

        String villageID = getValueFromCGIQueries("vid", allQuery);
        if(villageID == null) return null;
        if(villageID.length() <= 0) return null;

        return villageID;
    }

    /**
     * HTMLデータから村情報を抽出し村一覧を生成する。
     * @param html HTMLデータ
     * @return 村一覧
     */
    private Collection<Village> parseVillageList(CharSequence html){
        Collection<Village> result = new Vector<Village>();

        Matcher matcher = anchorRegex.matcher(html);
        for(;;){
            if( ! matcher.find() ) break;
            String hrefValue  = matcher.group(1);
            String anchorText = matcher.group(2);
            String option     = matcher.group(3);

            if(hrefValue == null) continue;
            if(hrefValue.length() <= 0) continue;
            if(anchorText == null) anchorText = "";

            String villageID = getVillageIDFromHREF(hrefValue);
            if(villageID == null) continue;
            if(villageID.length() <= 0) continue;

            String villageName = anchorText.replaceAll(Sp_m, "\u0020").trim();

            String[] parts = villageName.split("\u0020");
            if(   parts != null
               && parts[0] != null
               && parts[0].equals(villageID) ){
                villageName = this.landPrefix + villageName;
            }

            Village village = new Village(this, villageID, villageName);

            Village.State state;
            if(option == null || getType() == Type.HISTORICAL){
                state = Village.State.GAMEOVER;
            }else if(option.equals("参加者募集中です。")){
                state = Village.State.PROLOGUE;
            }else if(option.equals("開始待ちです。")){
                state = Village.State.PROLOGUE;
            }else if(option.equals("進行中です。")){
                state = Village.State.PROGRESS;
            }else if(option.equals("勝敗が決定しました。")){
                state = Village.State.EPILOGUE;
            }else if(option.equals("終了・ログ公開中。")){
                state = Village.State.GAMEOVER;
            }else{
                state = Village.State.UNKNOWN;
            }
            village.setState(state);

            result.add(village);
        }

        return result;
    }

    /**
     * 村リストを更新する。
     * 元情報は国のトップページと村一覧ページ。
     * 村リストはVillageの実装に従いソートされる。重複する村は排除。
     */
    public void updateVillageList() throws IOException{
        Collection<Village> vcol;
        SortedSet<Village> vset = new TreeSet<Village>();

        ServerAccess server = getServerAccess();

        CharSequence html;

        html = server.getHTMLTopPage();
        vcol = parseVillageList(html);
        vset.addAll(vcol);

        html = server.getHTMLLandList();
        vcol = parseVillageList(html);
        vset.addAll(vcol);

        this.villageList.clear();
        this.villageList.addAll(vset);

        return;
    }

    /**
     * 国の文字列表現を返す。
     * @return 文字列表現
     */
    @Override
    public String toString(){
        return this.landName;
    }
}
