/*
 * manage HTTP access
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: ServerAccess.java 245 2008-10-10 10:21:00Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.Date;

/**
 * 国ごとの人狼BBSサーバとの通信を一手に引き受ける。
 */
public class ServerAccess{

    private static final String USER_AGENT = HttpUtils.getUserAgentName();
    private static final String ENCODING_HTML = "Shift_JIS";
    // TODO MS932とかWindows31Jとかのほうがいいか？
    private static final Charset CHARSET_SJIS = Charset.forName(ENCODING_HTML);
    private static final int BUF_SIZE = 1024;
    private static final String JINRO_CGI = "./index.rb";

    /**
     * 与えられた文字列に対し「application/x-www-form-urlencoded」符号化を行う。
     * この符号化はHTTPのPOSTメソッドで必要になる。
     * この処理は、一般的なPC用Webブラウザにおける、
     * Shift_JISで書かれたHTML文書のFORMタグに伴う
     * submit処理を模倣する。
     * @param formData 元の文字列
     * @return 符号化された文字列
     */
    public static String formEncode(String formData){
        if(formData == null){
            return null;
        }
        String result;
        try{
            result = URLEncoder.encode(formData, ENCODING_HTML);
        }catch(UnsupportedEncodingException e){
            assert false;
            result = null;
        }
        return result;
    }

    /**
     * 配列版formEncode
     * @param formData 元の文字列
     * @return 符号化された文字列
     */
    public static String formEncode(char[] formData){
        return formEncode(new String(formData));
    }

    /**
     * 「Shift_JIS」でエンコーディングされた入力ストリームから文字列を生成する。
     * @param is 入力ストリーム
     * @return 文字列
     * @throws java.io.IOException 入出力エラー（おそらくネットワーク関連）
     */
    public static CharSequence downloadHTMLStream(InputStream is)
            throws IOException{
        StringBuilder result = new StringBuilder();

        Reader reader;
        // TODO 「～」が化ける。他にもいろいろあるはず。
        reader = new InputStreamReader(is, CHARSET_SJIS);
        reader = new BufferedReader(reader, BUF_SIZE);

        char[] buffer = new char[BUF_SIZE];
        for(;;){
            int chs = reader.read(buffer);
            if(chs <= 0){
                break;
            }
            result.append(buffer, 0, chs);
        }
        reader.close();

        return result;
    }

    private final URL baseURL;
    private long lastServerMs;
    private long lastLocalMs;
    private long lastSystemMs;
    private AccountCookie cookieAuth = null;
    private String encodedUserID = null;

    /**
     * 人狼BBSサーバとの接続管理を生成する。
     * この時点ではまだ通信は行われない。
     * @param baseURL 国別のベースURL
     */
    public ServerAccess(URL baseURL){
        this.baseURL = baseURL;
        return;
    }

    /**
     * 国のベースURLを返す。
     * @return ベースURL
     */
    protected URL getBaseURL(){
        return this.baseURL;
    }

    /**
     * CGIのURLを返す。
     * @return CGIのURL
     */
    protected URL getCGIURL(){
        URL result;
        try{
            result = new URL(this.baseURL, JINRO_CGI);
        }catch(MalformedURLException e){
            assert false;
            return null;
        }
        return result;
    }

    /**
     * 与えられたクエリーとCGIのURLから新たにURLを合成する。
     * @param query クエリー
     * @return 新たなURL
     */
    protected URL getQueryURL(CharSequence query){
        if(query.length() >= 1 && query.charAt(0) != '?'){
            return null;
        }

        URL result;
        try{
            result = new URL(getCGIURL(), query.toString());
        }catch(MalformedURLException e){
            assert false;
            return null;
        }
        return result;
    }

    /**
     * クエリーから、GETとPOSTいずれにも共通する初期化済みのHTTP接続を返す。
     * 認証データはここで組み込まれる。
     * @param query HTTPクエリー
     * @param method HTTPメソッド。"GET"か"POST"のいずれか。
     * @return 初期化済みHTTP接続
     * @throws java.io.IOException ネットワークエラー
     */
    private HttpURLConnection setupConnection(CharSequence query, String method)
            throws IOException{
        URL url = getQueryURL(query);

        HttpURLConnection httpConnection;
        httpConnection = (HttpURLConnection) (url.openConnection());

        httpConnection.setRequestProperty("Accept", "*" + "/" + "*");
        httpConnection.setRequestProperty("User-Agent", USER_AGENT);
        httpConnection.setUseCaches(false);
        httpConnection.setInstanceFollowRedirects(false);
        httpConnection.setDoInput(true);

        if(method.equalsIgnoreCase("GET")){
            httpConnection.setRequestMethod("GET");
            AccountCookie cookie = this.cookieAuth;
            if(cookie != null){
                if(shouldAccept(url, cookie)){
                    httpConnection.setRequestProperty(
                            "Cookie",
                            "login=" + cookie.getLoginData());
                }else{
                    clearAuthentication();
                }
            }
        }else if(method.equalsIgnoreCase("POST")){
            httpConnection.setRequestMethod("POST");
            httpConnection.setDoOutput(true);
        }else{
            assert false;
            return null;
        }

        return httpConnection;
    }

    /**
     * 与えられたクエリーを用いてHTMLデータを取得する入力ストリームを得る。
     * @param query HTTP-GET クエリー
     * @return 入力ストリーム
     * @throws java.io.IOException ネットワークエラー
     */
    protected InputStream getHTMLStream(CharSequence query)
            throws IOException{
        HttpURLConnection connection = setupConnection(query, "GET");

        String method = connection.getRequestMethod();
        URL url = connection.getURL();
        Jindolf.logger.info(method + " " + url);

        connection.connect();

        updateLastAccess(connection);

        int responseCode = connection.getResponseCode();
        String responseMessage = connection.getResponseMessage();
        Date connDate = new Date(connection.getDate());

        Jindolf.logger.info(responseCode
                            + " " + responseMessage
                            + " " + connDate);

        if(responseCode != HttpURLConnection.HTTP_OK){
            return null;
        }

        String charset = HttpUtils.getHTMLCharset(connection);
        if(!charset.equalsIgnoreCase(ENCODING_HTML)){
            return null;
        }

        InputStream stream = connection.getInputStream();

        return stream;
    }

    /**
     * 与えられたクエリーを用いてHTMLデータを取得する。
     * @param query HTTP-GET クエリー
     * @return HTMLデータ
     * @throws java.io.IOException ネットワークエラー
     */
    protected CharSequence downloadHTML(CharSequence query)
            throws IOException{
        InputStream stream = getHTMLStream(query);
        if(stream == null){
            return null;
        }

        return downloadHTMLStream(stream);
    }

    /**
     * トップページのHTMLデータを取得する。
     * @return HTMLデータ
     * @throws java.io.IOException ネットワークエラー
     */
    public CharSequence getHTMLTopPage() throws IOException{
        return downloadHTML("");
    }

    /**
     * 国に含まれる村一覧HTMLデータを取得する。
     * @return HTMLデータ
     * @throws java.io.IOException ネットワークエラー
     */
    public CharSequence getHTMLLandList() throws IOException{
        return downloadHTML("?cmd=log");
    }

    /**
     * 指定された村のPeriod一覧のHTMLデータを取得する。
     * 現在ゲーム進行中の村にも可能。
     * ※ 古国では使えないよ！
     * @param village 村
     * @return HTMLデータ
     * @throws java.io.IOException ネットワークエラー
     */
    public CharSequence getHTMLBoneHead(Village village) throws IOException{
        String villageID = village.getVillageID();
        return downloadHTML("?vid=" + villageID + "&meslog=");
    }

    /**
     * 指定された村の最新PeriodのHTMLデータをロードする。
     * 既にGAMEOVERの村ではPeriod一覧のHTMLデータとなる。
     * @param village 村
     * @return HTMLデータ
     * @throws java.io.IOException ネットワークエラー
     */
    public CharSequence getHTMLVillage(Village village) throws IOException{
        String villageID = village.getVillageID();
        return downloadHTML("?vid=" + villageID);
    }

    /**
     * 指定されたPeriodのHTMLデータをロードする
     * @param period Period
     * @return HTMLデータ
     * @throws java.io.IOException ネットワークエラー
     */
    public CharSequence getHTMLPeriod(Period period) throws IOException{
        return downloadHTML(period.getCGIQuery());
    }

    /**
     * 最終アクセス時刻を更新する。
     * @param connection HTTP接続
     */
    public void updateLastAccess(HttpURLConnection connection){
        this.lastServerMs = connection.getDate();
        this.lastLocalMs = System.currentTimeMillis();
        this.lastSystemMs = System.nanoTime() / (1000 * 1000);
    }

    /**
     * 指定された認証情報をPOSTする。
     * @param authData 認証情報
     * @return 認証情報が受け入れられたらtrue
     * @throws java.io.IOException ネットワークエラー
     */
    protected boolean postAuthData(String authData) throws IOException{
        byte[] authBytes = authData.getBytes();

        HttpURLConnection connection = setupConnection("", "POST");

        String method = connection.getRequestMethod();
        URL url = connection.getURL();
        Jindolf.logger.info(method + " " + url);

        OutputStream os = connection.getOutputStream();
        os.write(authBytes);
        os.flush();
        os.close();

        int responseCode = connection.getResponseCode();
        String responseMessage = connection.getResponseMessage();
        Date connDate = new Date(connection.getDate());

        Jindolf.logger.info(responseCode
                            + " " + responseMessage
                            + " " + connDate);

        if(responseCode != HttpURLConnection.HTTP_MOVED_TEMP){    // 302
            return false;
        }

        updateLastAccess(connection);

        AccountCookie loginCookie = AccountCookie.createCookie(connection);
        if(loginCookie == null){
            return false;
        }

        setAuthentication(loginCookie);

        return true;
    }

    /**
     * 指定したURLに対しCookieを送っても良いか否か判定する。
     * 判別材料は Cookie の寿命とパス指定のみ。
     * @param url URL
     * @param cookie Cookie
     * @return 送ってもよければtrue
     */
    private static boolean shouldAccept(URL url, AccountCookie cookie){
        if(cookie.hasExpired()){
            return false;
        }

        String urlPath = url.getPath();
        String cookiePath = cookie.getPathURI().getPath();

        if( ! urlPath.startsWith(cookiePath) ){
            return false;
        }

        return true;
    }

    /**
     * 現在ログイン中か否か判別する。
     * @return ログイン中ならtrue
     */
    // TODO interval call
    public boolean hasLoggedIn(){
        AccountCookie cookie = this.cookieAuth;
        if(cookie == null){
            return false;
        }
        if(cookie.hasExpired()){
            clearAuthentication();
            return false;
        }
        return true;
    }

    /**
     * 与えられたユーザIDとパスワードでログイン処理を行う。
     * @param userID ユーザID
     * @param password パスワード
     * @return ログインに成功すればtrue
     * @throws java.io.IOException ネットワークエラー
     */
    public final boolean login(String userID, char[] password)
            throws IOException{
        if(hasLoggedIn()){
            return true;
        }

        String id = formEncode(userID);
        if(id == null || id.length() <= 0){
            return false;
        }

        String pw = formEncode(password);
        if(pw == null || pw.length() <= 0){
            return false;
        }

        this.encodedUserID = id;

        String redirect = formEncode("&#bottom");   // TODO ほんとに必要？

        StringBuilder postData = new StringBuilder();
        postData.append("cmd=login");
        postData.append('&').append("cgi_param=").append(redirect);
        postData.append('&').append("user_id=").append(id);
        postData.append('&').append("password=").append(pw);

        boolean result;
        try{
            result = postAuthData(postData.toString());
        }catch(IOException e){
            clearAuthentication();
            throw e;
        }

        return result;
    }

    /**
     * ログアウト処理を行う。
     * @throws java.io.IOException
     */
    // TODO シャットダウンフックでログアウトさせようかな…
    public void logout() throws IOException{
        if(!hasLoggedIn()){
            return;
        }
        if(this.encodedUserID == null){
            clearAuthentication();
            return;
        }

        String redirect = formEncode("&#bottom"); // TODO 必要？

        StringBuilder postData = new StringBuilder();
        postData.append("cmd=logout");
        postData.append('&').append("cgi_param=").append(redirect);
        postData.append('&').append("user_id=").append(this.encodedUserID);

        try{
            postAuthData(postData.toString());
        }finally{
            clearAuthentication();
        }

        return;
    }

    /**
     * 認証情報クリア
     */
    // TODO タイマーでExpire date の時刻にクリアしたい。
    protected void clearAuthentication(){
        this.cookieAuth = null;
        this.encodedUserID = null;
        return;
    }

    /**
     * 認証情報のセット
     * @param cookie 認証Cookie
     */
    private void setAuthentication(AccountCookie cookie){
        this.cookieAuth = cookie;
        return;
    }

    // TODO JRE1.6対応するときに HttpCookie, CookieManager 利用へ移行したい。

}
