/*
 * Regex pattern
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: RegexPattern.java 308 2008-11-29 15:54:21Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * 正規表現
 */
public class RegexPattern{

    private final String editSource;
    private final boolean isRegex;
    private final Pattern pattern;
    private final String comment;

    /**
     * コンストラクタ。
     *
     * @param editSource リテラル文字列または正規表現
     * @param isRegex 指定文字列が正規表現ならtrue。リテラルならfalse
     * @param flag 正規表現フラグ
     * @param comment コメント
     * @throws java.util.regex.PatternSyntaxException 正規表現がおかしい
     */
    public RegexPattern(String editSource,
                          boolean isRegex,
                          int flag,
                          String comment)
            throws PatternSyntaxException{
        super();

        this.editSource = editSource;
        this.isRegex    = isRegex;
        this.comment    = comment;

        String regexExpr;
        if(this.isRegex){
            regexExpr = this.editSource;
        }else{
            regexExpr = Pattern.quote(this.editSource);
        }

        this.pattern = Pattern.compile(regexExpr, flag);

        return;
    }

    /**
     * コンストラクタ。
     *
     * @param editSource リテラル文字列または正規表現
     * @param isRegex 指定文字列が正規表現ならtrue。リテラルならfalse
     * @param flag 正規表現フラグ
     * @throws java.util.regex.PatternSyntaxException 正規表現がおかしい
     */
    public RegexPattern(String editSource,
                          boolean isRegex,
                          int flag )
            throws PatternSyntaxException{
        this(editSource, isRegex, flag, " ");
        return;
    }

    /**
     * 元の入力文字列を返す。
     * @return 入力文字列
     */
    public String getEditSource(){
        return this.editSource;
    }

    /**
     * コメントを返す。
     * @return コメント
     */
    public String getComment(){
        return this.comment;
    }

    /**
     * 元の入力文字列が正規表現か否か返す。
     * @return 正規表現ならtrue
     */
    public boolean isRegex(){
        return this.isRegex;
    }

    /**
     * 正規表現フラグを返す。
     * @return 正規表現フラグ。
     * @see java.util.regex.Pattern#flags()
     */
    public int getFlag(){
        return this.pattern.flags();
    }

    /**
     * コンパイルされた正規表現形式を返す。
     * @return コンパイルされた正規表現形式
     */
    public Pattern getPattern(){
        return this.pattern;
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public String toString(){
        return this.editSource;
    }

    /**
     * {@inheritDoc}
     * @param obj {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj){
        if(obj == null){
            return false;
        }
        if( ! (obj instanceof RegexPattern) ){
            return false;
        }
        RegexPattern other = (RegexPattern) obj;

        String thisPattern = this.pattern.pattern();
        String otherPattern = other.pattern.pattern();

        if( ! thisPattern.equals(otherPattern) ) return false;

        if(this.pattern.flags() != other.pattern.flags()) return false;

        return true;
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public int hashCode(){
        int hash = this.pattern.pattern().hashCode();
        hash ^= this.pattern.flags();
        return hash;
    }
}
