/*
 * system event in game
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: SysEvent.java 309 2008-11-29 16:18:53Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 人狼BBSシステムが生成する各種メッセージ。
 * Topicの具体化。
 */
// TODO 狼の襲撃先表示は Talk か SysEvent どちらにしよう...
public class SysEvent implements Topic{

    /** イベントのおおまかな種別 */
    public static enum EventClass{
        /** &lt;div class="announce"&gt;に対応 */
        ANNOUNCE,
        /** &lt;div class="order"&gt;に対応 */
        ORDER,
        /** &lt;div class="extra"&gt;に対応 */
        EXTRA,
    }

    /** イベント種別。 無意味なのも含む。そのうち増やす。 */
    public static enum Type{
        STARTPROLOGUE,
        ONSTAGE,
        WINVILLAGE,
        WINWOLF,
        WINHAMSTER,
        MURDER,
        NOMURDER,
        GAMEOVER,
        UNKNOWN,
        // 開村
        // アバター初登場
        // 鏡
        // 役職確定
        // ゲルト死亡
        // 最初の犠牲者
        // 現在の生存者
        // 突然死
        // 誰が誰を占ったか
        // 誰が誰を守ったか
        // 誰が誰に投票したか&誰が処刑されたか
        // 誰が誰を襲おうと試みたか
        // 誰が襲撃されたか
        // 襲撃は行われなかったか
        // 村側勝利
        // 狼勝利
        // 役職&ログイン名公開
        // 終了しました。もう誰も書き込めません。
        // 日付更新
        // ハムスター勝利?
        // ハムスターが溶けた？
        // ORDER
        // まだ発言していないもの
        // 更新時刻の通知と投票促し
        // 閉村時刻の通知と投稿促し
    }

    private static Pattern murderRegex =
            Pattern.compile("次の日の朝、(.+) が無残な姿で発見された。\\n",
                            Pattern.DOTALL );
    private static Pattern nomurderRegex =
            Pattern.compile("今日は犠牲者がいないようだ。"
                            +"人狼は襲撃に失敗したのだろうか。\\n",
                            Pattern.DOTALL );

    /**
     * イベントクラス文字列をデコードする。
     * @param eventClass イベントクラス文字列
     * @return デコードしたEventClass
     */
    public static EventClass decodeEventClass(CharSequence eventClass){
        EventClass result;
        if("announce".contentEquals(eventClass)){
            result = EventClass.ANNOUNCE;
        }else if("order".contentEquals(eventClass)){
            result = EventClass.ORDER;
        }else if("extra".contentEquals(eventClass)){
            result = EventClass.EXTRA;
        }else{
            assert false;
            return null;
        }

        return result;
    }

    /**
     * イベントクラスを文字列にエンコードする。
     * @param eventClass イベントクラス
     * @return エンコードされた文字列
     */
    public static CharSequence encodeEventClass(EventClass eventClass){
        CharSequence result;
        switch(eventClass){
        case ANNOUNCE:
            result = "announce";
            break;
        case ORDER:
            result = "order";
            break;
        case EXTRA:
            result = "extra";
            break;
        default:
            assert false;
            return null;
        }

        return result;
    }

    private final EventClass eventClass;
    private final String message;
    private Type type = Type.UNKNOWN;

    /**
     * システムイベントを生成する。
     * @param eventClass イベント種別。
     * @param message イベントメッセージ（HTMLタグ含む）
     * @throws java.lang.NullPointerException null引数の検出
     */
    public SysEvent(EventClass eventClass, String message)
            throws NullPointerException{
        super();
        if(eventClass == null || message == null){
            throw new NullPointerException();
        }

        this.eventClass = eventClass;
        this.message = message.replace("<br />","\n")
                              .replace("&gt;",">")
                              .replace("&lt;","<")
                              .replace("&nbsp;"," ")
                              .replace("&quot;","\"")
                              .replace("&amp;","&")
                              .replaceAll("<a [^>]*>", "")
                              .replace("</a>", "");
        // TODO パーサでやるべきか？

        Matcher matcher;

        matcher = murderRegex.matcher(this.message);
        if(matcher.matches()){
            this.type = Type.MURDER;
            return;
        }
        matcher = nomurderRegex.matcher(this.message);
        if(matcher.matches()){
            this.type = Type.NOMURDER;
            return;
        }

        return;
    }

    /**
     * イベントメッセージを取得する。
     * @return イベントメッセージ
     */
    public String getMessage(){
        return this.message;
    }

    /**
     * イベントクラスを取得する。
     * @return イベントクラス
     */
    public EventClass getEventClass(){
        return this.eventClass;
    }

    /**
     * イベント種別を取得する。
     * @return イベント種別
     */
    public Type getType(){
        return this.type;
    }
}
