/*
 * Roles in game
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: GameRole.java 461 2009-03-30 15:22:46Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.util.Comparator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 役職
 */
public enum GameRole{

    /** 村人 */
    HABITANT("村人", '村'),
    /** 人狼 */
    WOLF("人狼", '狼'),
    /** 占い師 */
    SEER("占い師", '占'),
    /** 霊能者 */
    SHAMAN("霊能者", '霊'),
    /** 狂人 */
    MADMAN("狂人", '狂'),
    /** 狩人 */
    HUNTER("狩人", '狩'),
    /** 共有者 */
    FRATER("共有者", '共'),
    /** ハムスター人間 */
    HAMSTER("ハムスター人間", '公'),
    ;

    private static final Comparator<GameRole> powerBalanceComparator
            = new PowerBalanceComparator();

    private static final Pattern roleRegex;

    static{
        String roleGroup = "";
        for(GameRole role : values()){
            if(roleGroup.length() > 0){
                roleGroup += "|";
            }
            roleGroup += "(" + Pattern.quote(role.getRoleName()) + ")";
        }
        roleRegex = Pattern.compile(roleGroup);
    }

    /**
     * 定義済みRole名に一致しないか調べる。
     * @param matcher マッチャ
     * @return 一致したRole。一致しなければnull。
     */
    public static GameRole lookingAtRole(Matcher matcher){
        matcher.usePattern(roleRegex);

        if( ! matcher.lookingAt() ) return null;
        int groupCt = matcher.groupCount();
        for(int group = 1; group <= groupCt; group++){
            if(matcher.start(group) >= 0){
                GameRole role = values()[group-1];
                return role;
            }
        }

        return null;
    }

    /**
     * パワーバランス表記用Comparatorを得る。
     * @return Comparator
     */
    public static Comparator<GameRole> getPowerBalanceComparator(){
        return powerBalanceComparator;
    }

    private final String roleName;
    private final char shortName;

    /**
     * コンストラクタ
     * @param roleName 役職名
     */
    private GameRole(String roleName, char shortName){
        this.roleName = roleName;
        this.shortName = shortName;
        return;
    }

    /**
     * 役職名を返す。
     * @return 役職名
     */
    public String getRoleName(){
        return this.roleName;
    }

    /**
     * 一文字に短縮された役職名を得る
     * @return 短縮された役職名
     */
    public char getShortName(){
        return this.shortName;
    }

    /**
     * 陣営種別
     */
    public static enum Team{
        VILLAGE ("村陣営"),
        WOLF    ("狼陣営"),
        HAMSTER ("ハムスター人間"),
        ;

        private final String teamName;

        /**
         * コンストラクタ
         * @param teamName チーム名
         */
        private Team(String teamName){
            this.teamName = teamName;
            return;
        }

        /**
         * チーム名を取得する
         * @return チーム名
         */
        public String getTeamName(){
            return this.teamName;
        }
    }

    /**
     * パワーバランス表記用Comparator。
     * 「村狂占霊狩公狂狼」の順で役職を一意に順序づける。
     */
    private static class PowerBalanceComparator
            implements Comparator<GameRole>{

        /**
         * {@inheritDoc}
         * @param role1 {@inheritDoc}
         * @param role2 {@inheritDoc}
         * @return {@inheritDoc}
         */
        public int compare(GameRole role1, GameRole role2){
            if(role1 == role2) return 0;
            if(role1 == null && role2 == null) return 0;
            if(role1 != null && role2 == null) return +1;
            if(role1 == null && role2 != null) return -1;

            int power1 = getPowerValue(role1);
            int power2 = getPowerValue(role2);

            return power1 - power2;
        }

        /**
         * 役職に強さを割り当てる。
         * 村人のほうが狼より大きい値を返す。
         * @param role 役職
         * @return 強さ
         */
        private static int getPowerValue(GameRole role){
            int power;
            switch(role){
            case HABITANT: power = -50; break;
            case FRATER:   power = -40; break;
            case SEER:     power = -30; break;
            case SHAMAN:   power = -20; break;
            case HUNTER:   power = -10; break;
            case HAMSTER:  power =   0; break;
            case MADMAN:   power = +10; break;
            case WOLF:     power = +20; break;
            default:
                assert false;
                power = Integer.MAX_VALUE;
                break;
            }
            return power;
        }
    }
}
