/*
 * daily period in village
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Period.java 452 2009-03-26 14:31:30Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Image;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * いわゆる「日」。
 * 村の進行の一区切り。プロローグやエピローグも含まれる。
 *
 * 将来、24時間更新でなくなる可能性の考慮が必要。
 * 人気のないプロローグなどで、
 * 24時間以上の期間を持つPeriodが生成される可能性の考慮が必要。
 */
public class Period{   // TODO Comparable も implement する？

    /**
     * Periodの種別
     */
    public static enum Type{
        /** プロローグ */
        PROLOGUE,
        /** 通常のゲーム進行 */
        PROGRESS,
        /** エピローグ */
        EPILOGUE
    }

    private static final Pattern villageInfoRegex;
    private static final Pattern startMessageRegex;
    private static final Pattern switchTagRegex;
    private static final Pattern talkRegex;
    private static final Pattern eventBodyRegex;
    private static final Pattern anchorRegex;
    private static final Pattern loginRegex;

    private static final String[][] enRefTable = {
        {"<br />", "\n"},
        {"&lt;",   "<" },
        {"&gt;",   ">" },
        {"&quot;", "\""},
        {"&amp;",  "&" },
    };
    private static final Pattern enRefRegex;

    static{
        String Spchar = "\u0020\\t\\n\\r";
        String Sp = "[" +Spchar+ "]";
        String Sp_n = Sp + "*";
        String Sp_m = Sp + "+";

        String villageInfo =
                ">" +Sp_n+ "([^<]+?)" +Sp_n+
                "<strong>" +Sp_n+
                    "[\\(（]" +Sp_n+
                    "([1-9][0-9]?)/([1-9][0-9]?)" +Sp_n+
                    "(午前|午後)" +Sp_n+
                    "([0-9][0-9]?)時" +Sp_n+
                    "([0-9][0-9]?)分" +Sp_n+
                    "に更新" +Sp_n+
                    "[\\)）]" +Sp_n+
                "</strong>";

        String divMessage  = Sp_n+ "<div class=\"message\">" +Sp_n;
        String divEvent = "<div class=\"(announce|order|extra)\">";
        String anchorTalk = "<a name=\"([^\"]+)\">([^<]+)</a>";
        String tagSwitch =
                Sp_n+
                "(?:" +
                    divEvent +
                "|" +
                    anchorTalk +
                ")"
                +Sp_n;

        String brEmpty = "<br" +Sp_n+ "/>";
        String anchorStart = "<a" +Sp_m+ "href=\"([^\"]*)\"" +Sp_n+ ">";
        String anchorClose = "</a>";
        String anchor = anchorStart +"([^<]*)"+ anchorClose;
        String divEnd = "</div>";
        String talkPattern =
                Sp_n+
                "<span class=\"time\">" +Sp_n+
                    "(午前|午後)" +Sp_n+
                    "([0-9][0-9]?)時" +Sp_n+
                    "([0-9][0-9]?)分" +Sp_n+
                "</span>" +Sp_n+
                "<table [^>]*? class=\"message_box\">" +Sp_n+
                    "<tr>" +Sp_n+
                        "<td width=\"[^\"]*\">" +Sp_n+
                            "<img src=\"([^\"]+)\">" +Sp_n+
                        "</td>" +Sp_n+
                        "<td width=\"[^\"]*\">" +Sp_n+
                            "<img src=\"([^\"]+)\">" +Sp_n+
                        "</td>" +Sp_n+
                        "<td>" +Sp_n+
                            "<div class=\"mes_([^\"]+)_body[01]\">" +Sp_n+
                                "<div class=\"mes_([^\"]+)_body[01]\">" +Sp_n+
                                    "(.*?)" +Sp_n+
                                "<?/div>" +Sp_n+   // F1603 2d21:12 対策
                            divEnd +Sp_n+
                        "</td>" +Sp_n+
                    "</tr>" +Sp_n+
                "</table>" +Sp_n+
            divEnd +Sp_n;
        String eventBody =
                "(?:" +
                    "[^<]*?" +
                "|" +
                    brEmpty +
                "|" +
                    anchor +
                ")*?";
        eventBody =
                "(" +eventBody+ ")" +Sp_n+ divEnd +Sp_n+ divEnd +Sp_n;
        String prologue = "プロローグ";
        String epilogue = "エピローグ";
        String gameover = "終了";
        String progress = "([1-9][0-9]*)日目";
        String content = "("+
                            prologue +
                         "|"+
                            epilogue +
                         "|"+
                            progress +
                         "|"+
                            gameover +
                         ")";
        // TODO Villageに似たようなパターンがあるので統一したい。
        String periodAnchor = anchorStart + content + anchorClose;

        String loginForm =
                  "<input type=(?:\"hidden\"|\"text\")(?: size=\"10\")?"
                + " name=\"user_id\""
                + " value=\"([^\"]*)\" />"
                + "|"
                + "\\Q<a href=\"index.rb?cmd=to_sign_in\">sign in</a>\\E";

        villageInfoRegex = Pattern.compile(villageInfo, Pattern.DOTALL);
        startMessageRegex = Pattern.compile(divMessage, Pattern.DOTALL);
        switchTagRegex = Pattern.compile(tagSwitch, Pattern.DOTALL);
        talkRegex = Pattern.compile(talkPattern, Pattern.DOTALL);
        eventBodyRegex = Pattern.compile(eventBody, Pattern.DOTALL);
        anchorRegex = Pattern.compile(periodAnchor, Pattern.DOTALL);
        loginRegex = Pattern.compile(loginForm, Pattern.DOTALL);

        String refGroups = "";
        for(String[] pair : enRefTable){
            if(refGroups.length() > 0) refGroups += "|";
            String group = "(" + pair[0] + ")";
            refGroups += group;
        }
        enRefRegex = Pattern.compile(refGroups, Pattern.DOTALL);
    }

    /**
     * 実体参照の解決を行う
     * @param from 解決前の文字列
     * @return 解決後の文字列
     */
    public static CharSequence resolveRef(CharSequence from){
        StringBuilder result = new StringBuilder(from.length());

        int fromPos = 0;

        Matcher matcher = enRefRegex.matcher(from);
        while(matcher.find()){
            for(int idx=0; idx < enRefTable.length; idx++){
                int start = matcher.start(idx + 1);
                int end   = matcher.end  (idx + 1);
                if(start < 0) continue;

                result.append(from, fromPos, start);
                result.append(enRefTable[idx][1]);

                fromPos = end;
            }
        }
        result.append(from, fromPos, from.length());

        return result;
    }

    private final Village homeVillage;
    private final Type    periodType;
    private final int     day;
    private int limitHour;
    private int limitMinute;
    private String loginName;
    private boolean isFullOpen = false;

    private final List<Topic> topicList = new LinkedList<Topic>();
    private final List<Topic> unmodList
            = Collections.unmodifiableList(this.topicList);
    private final Map<Avatar, int[]> countMap = new HashMap<Avatar, int[]>();

    /**
     * この Period が進行中の村の最新日で、今まさに次々と発言が蓄積されているときは
     * true になる。
     * ※重要: Hot な Period は meslog クエリーを使ってダウンロードできない。
     */
    private boolean isHot;

    /**
     * Periodを生成する。
     * この段階では発言データのロードは行われない。
     * デフォルトで非Hot状態。
     * @param homeVillage 所属するVillage
     * @param periodType Period種別
     * @param day Period通番
     * @throws java.lang.NullPointerException 引数にnullが渡された場合。
     */
    public Period(Village homeVillage,
                   Type periodType,
                   int day)
                   throws NullPointerException{
        this(homeVillage, periodType, day, false);
        return;
    }

    /**
     * Periodを生成する。
     * この段階では発言データのロードは行われない。
     * @param homeVillage 所属するVillage
     * @param periodType Period種別
     * @param day Period通番
     * @param isHot Hotか否か
     * @throws java.lang.NullPointerException 引数にnullが渡された場合。
     */
    private Period(Village homeVillage,
                    Type periodType,
                    int day,
                    boolean isHot)
                    throws NullPointerException{
        if(   homeVillage == null
           || periodType  == null ) throw new NullPointerException();
        if(day < 0){
            throw new IllegalArgumentException("Period day is too small !");
        }
        switch(periodType){
        case PROLOGUE:
            assert day == 0;
            break;
        case PROGRESS:
        case EPILOGUE:
            assert day > 0;
            break;
        }

        this.homeVillage = homeVillage;
        this.periodType  = periodType;
        this.day         = day;

        unload();

        this.isHot = isHot;

        return;
    }

    /**
     * 所属する村を返す。
     * @return 村
     */
    public Village getVillage(){
        return this.homeVillage;
    }

    /**
     * Period種別を返す。
     * @return 種別
     */
    public Type getType(){
        return this.periodType;
    }

    /**
     * Period通番を返す。
     * プロローグは常に0番。
     * n日目のゲーム進行日はn番
     * エピローグは最後のゲーム進行日+1番
     * @return Period通番
     */
    public int getDay(){
        return this.day;
    }

    /**
     * 更新時刻の文字表記を返す。
     * @return 更新時刻の文字表記
     */
    public String getLimit(){
        StringBuilder result = new StringBuilder();

        if(this.limitHour < 10) result.append('0');
        result.append(this.limitHour).append(':');

        if(this.limitMinute < 10) result.append('0');
        result.append(this.limitMinute);

        return result.toString();
    }

    /**
     * Hotか否か返す。
     * @return Hotか否か
     */
    public boolean isHot(){
        return this.isHot;
    }

    /**
     * Hotか否か設定する。
     * @param isHot Hot指定
     */
    public void setHot(boolean isHot){
        this.isHot = isHot;
    }

    /**
     * プロローグか否か判定する。
     * @return プロローグならtrue
     */
    public boolean isPrologue(){
        if(getType() == Type.PROLOGUE) return true;
        return false;
    }

    /**
     * エピローグか否か判定する。
     * @return エピローグならtrue
     */
    public boolean isEpilogue(){
        if(getType() == Type.EPILOGUE) return true;
        return false;
    }

    /**
     * 進行日か否か判定する。
     * @return 進行日ならtrue
     */
    public boolean isProgress(){
        if(getType() == Type.PROGRESS) return true;
        return false;
    }

    /**
     * このPeriodにアクセスするためのクエリーを生成する。
     * @return CGIに渡すクエリー
     */
    public String getCGIQuery(){
        StringBuilder result = new StringBuilder();

        Village village = getVillage();
        result.append(village.getCGIQuery());

        if(isHot()){
            result.append("&mes=all");   // 全表示指定
            return result.toString();
        }

        result.append("&meslog=").append(village.getVillageID());
        switch(getType()){
        case PROLOGUE:
            result.append("_ready_0");
            break;
        case PROGRESS:
            result.append("_progress_").append(getDay() -1);
            break;
        case EPILOGUE:
            result.append("_party_").append(getDay() -1);
            break;
        default:
            assert false;
            return null;
        }
        result.append("&mes=all");

        return result.toString();
    }

    /**
     * Periodに含まれるTopicのリストを返す。
     * このリストは上書き操作不能。
     * @return Topicのリスト
     */
    public List<Topic> getTopicList(){
        return this.unmodList;
    }

    /**
     * Periodに含まれるTopicの総数を返す。
     * @return Topic総数
     */
    public int getTopics(){
        return this.topicList.size();
    }

    /**
     * Topicを追加する。
     * @param topic Topic
     * @throws java.lang.NullPointerException nullが渡された場合。
     */
    protected void addTopic(Topic topic) throws NullPointerException{
        if(topic == null) throw new NullPointerException();
        this.topicList.add(topic);
        return;
    }

    /**
     * Periodのキャプション文字列を返す。
     * 主な用途はタブ画面の耳のラベルなど。
     * @return キャプション文字列
     */
    public String getCaption(){
        String result;

        switch(getType()){
        case PROLOGUE:
            result = "プロローグ";
            break;
        case PROGRESS:
            result = "" + getDay() + "日目";
            break;
        case EPILOGUE:
            result = "エピローグ";
            break;
        default:
            assert false;
            result = null;
            break;
        }

        return result;
    }

    /**
     * このPeriodをダウンロードしたときのログイン名を返す。
     * @return ログイン名。ログアウト中はnull。
     */
    public String getLoginName(){
        return this.loginName;
    }

    /**
     * このPeriodの内容にゲーム進行上隠された部分がある可能性を判定する。
     * @return 隠れた要素がありうるならfalse
     */
    public boolean isFullOpen(){
        return this.isFullOpen;
    }

    /**
     * ロード済みか否かチェックする。
     * @return ロード済みならtrue
     */
    public boolean hasLoaded(){
        return getTopics() > 0;
    }

    /**
     * Topicリストを強制再読み込みする。
     * @throws java.io.IOException ネットワーク入出力の異常
     */
    public void reloadPeriod() throws IOException{
        loadPeriod(true);
        return;
    }

    /**
     * Topicリストがまだ無ければ読み込む。
     * @throws java.io.IOException ネットワーク入出力の異常
     */
    public void updatePeriod() throws IOException{
        loadPeriod(false);
        return;
    }

    /**
     * Topicリストを読み込む。
     * @param force trueなら強制再読み込み。
     *               falseならまだ読み込んで無い時のみ読み込み。
     * @throws java.io.IOException ネットワーク入出力の異常
     */
    public void loadPeriod(boolean force) throws IOException{
        if( ! force && hasLoaded() ) return;

        Village village = getVillage();
        Land land = village.getParentLand();
        ServerAccess server = land.getServerAccess();

        if(village.getState() != Village.State.PROGRESS){
            this.isFullOpen = true;
        }else if(getType() != Period.Type.PROGRESS){
            this.isFullOpen = true;
        }else{
            this.isFullOpen = false;
        }


        HtmlSequence html = server.getHTMLPeriod(this);

        this.topicList.clear();
        this.countMap.clear();

        int lastPos = 0;
        int htmlLength = html.length();

        lastPos = parseLoginForm(html, lastPos, htmlLength);
        lastPos = parseDeadLine(html, lastPos, htmlLength);

        boolean wasHot = isHot();
        lastPos = parseDayLink(html, lastPos, htmlLength);
        if(wasHot && ! isHot() ){
            reloadPeriod();
            return;
        }

        lastPos = parseTopics(html, lastPos, htmlLength);

        return;
    }

    /**
     * ログインフォームからログイン名を抽出する。
     * @param html HTML文書
     * @param startPos パース開始位置指定
     * @param endPos パース終了位置指定
     * @return パースが完了した位置
     */
    private int parseLoginForm(CharSequence html,
                                 final int startPos,
                                 final int endPos   ){
        int lastPos = startPos;

        this.loginName = null;

        Matcher matcher = loginRegex.matcher(html);
        matcher.region(startPos, endPos);
        if( ! matcher.find() ) return lastPos;
        lastPos = matcher.end();

        String account = matcher.group(1);
        if(account != null && account.length() > 0){
            this.loginName = account;
        }

        return lastPos;
    }

    /**
     * HTMLページ上部の締め切り時刻を解析する。
     * @param html HTML文書
     * @param startPos パース開始位置指定
     * @param endPos パース終了位置指定
     * @return パースが完了した位置
     */
    private int parseDeadLine(CharSequence html,
                                final int startPos,
                                final int endPos   ){
        int lastPos = startPos;

        int upMonth;
        int upDay;
        int upHour;
        int upMinute;

        Matcher matcher = villageInfoRegex.matcher(html);
        matcher.region(startPos, endPos);
        if( ! matcher.find() ) return lastPos;
        lastPos = matcher.end();

        String vName = matcher.group(1);
        String month = matcher.group(2);
        String xday   = matcher.group(3);
        String ampm  = matcher.group(4);
        String hh    = matcher.group(5);
        String mm    = matcher.group(6);

        // TODO どこかへ格納が必要
        upMonth  = Integer.parseInt(month);
        upDay    = Integer.parseInt(xday);
        upHour   = Integer.parseInt(hh);
        upMinute = Integer.parseInt(mm);
        // TODO 「午後12時」は存在するか？たぶんない。
        if( ampm.equals("午後") ) upHour += 12;
        upHour = upHour % 24;     // 24時 = 0時

        this.limitHour   = upHour;
        this.limitMinute = upMinute;

        return lastPos;
    }

    /**
     * 自分へのリンクが無いかチェックする。
     * 自分へのリンクが見つかればこのPeriodを非Hotにする。
     * 自分へのリンクがあるということは、
     * 今読んでるHTMLは別のPeriodのために書かれたものということ。
     * 考えられる原因は、HotだったPeriodがゲーム進行に従いHotでなくなったこと。
     * @param html HTML文書
     * @param startPos パース範囲指定始め
     * @param endPos パース範囲指定終わり
     * @return パースが完了した位置
     */
    private int parseDayLink(CharSequence html, int startPos, int endPos){
        int lastPos = startPos;

        Matcher matcher = anchorRegex.matcher(html);
        matcher.region(startPos, endPos);
        for(;;){
            if( ! matcher.find() ) break;
            lastPos = matcher.end();
            String hrefValue  = matcher.group(1);
            String anchorText = matcher.group(2);
            String dayString  = matcher.group(3);

            if( anchorText.equals("プロローグ") ){
                if(getType() == Type.PROLOGUE){
                    setHot(false);
                }
            }else if( anchorText.endsWith("日目") ){
                int dayNum = Integer.parseInt(dayString);
                if(getType() == Type.PROGRESS && getDay() == dayNum){
                    setHot(false);
                }
            }else if( anchorText.equals("エピローグ") ){
                if(getType() == Type.EPILOGUE){
                    setHot(false);
                }
            }else if( anchorText.equals("終了") ){
                break;
            }
        }

        return lastPos;
    }

    /**
     * 0個以上のTopicを、見つからなくなるまで次々とパースする。
     * @param html HTML文書
     * @param startPos パース開始位置指定
     * @param endPos パース終了位置指定
     * @return パースが完了した位置
     */
    private int parseTopics(HtmlSequence html, int startPos, int endPos){
        int lastPos = startPos;
        for(;;){
            int newPos = findTopic(html, lastPos, endPos);
            if(newPos <= lastPos) break;
            lastPos = newPos;
        }
        return lastPos;
    }

    /**
     * Topicを一つパースする。
     * 必要に応じてシステムイベントと会話データのパース処理を振り分ける。
     * @param html HTML文書
     * @param startPos パース開始位置指定
     * @param endPos パース終了位置指定
     * @return パースが完了した位置
     */
    private int findTopic(HtmlSequence html,
                           final int startPos,
                           final int endPos    ){
        int lastPos;
        Matcher matcher;

        matcher = startMessageRegex.matcher(html);
        matcher.region(startPos, endPos);
        if( ! matcher.find() ) return startPos;
        lastPos = matcher.end();

        matcher = switchTagRegex.matcher(html);
        matcher.region(lastPos, endPos);
        if( ! matcher.find() ) return startPos;
        lastPos = matcher.end();
        String eventClassStr = matcher.group(1);
        String mesID = matcher.group(2);
        String fullName = matcher.group(3);

        int nextPos;
        if(eventClassStr != null){
            SysEvent.EventClass eventClass =
                    SysEvent.decodeEventClass(eventClassStr);
            nextPos = findSystemEvent(eventClass, html, lastPos, endPos);
        }else{
            nextPos = findTalk(mesID, fullName, html, lastPos, endPos);
        }
        if(lastPos >= nextPos) return startPos;

        return nextPos;
    }

    /**
     * システムイベントを一件パースしTopicリストに追加する。
     * @param eventClass システムイベント種別
     * @param html HTML文書
     * @param startPos パース開始位置指定
     * @param endPos パース終了位置指定
     * @return パースが完了した位置
     */
    private int findSystemEvent(SysEvent.EventClass eventClass,
                                  CharSequence html,
                                  final int startPos,
                                  final int endPos ){
        Matcher matcher = eventBodyRegex.matcher(html);
        matcher.region(startPos, endPos);
        if( ! matcher.find() ) return startPos;
        int nextPos = matcher.end();
        String body = matcher.group(1);

        SysEvent event = new SysEvent(eventClass, body);
        addTopic(event);

        SysEvent.Type eventType = event.getType();
        if(   eventType == SysEvent.Type.MURDER
           || eventType == SysEvent.Type.NOMURDER ){
            for(Topic topic : this.topicList){
                if( ! (topic instanceof Talk) ) continue;
                Talk talk = (Talk) topic;
                if(talk.getTalkType() != Talk.Type.WOLFONLY) continue;
                if( ! StringUtils
                     .isTerminated(talk.getDialog(),
                                   "！\u0020今日がお前の命日だ！") ){
                    continue;
                }
                talk.setCount(-1);
                this.countMap.clear();
            }
        }

        return nextPos;
    }

    /**
     * 会話データを一件パースしTopicリストに追加する。
     * @param anchorID 発言を識別する識別子
     * @param fullName 発言者フルネーム
     * @param html HTML文書
     * @param startPos パース開始位置指定
     * @param endPos パース終了位置指定
     * @return パースが完了した位置
     */
    private int findTalk(String anchorID,
                          String fullName,
                          HtmlSequence html,
                          final int startPos,
                          final int endPos){
        Matcher matcher;
        matcher = talkRegex.matcher(html);
        matcher.region(startPos, endPos);
        if( ! matcher.find() ) return startPos;
        int nextPos = matcher.end();

        int hour   = StringUtils.parseInt(html, matcher, 2);
        int minute = StringUtils.parseInt(html, matcher, 3);
        // TODO 「午後12時」は存在するか？たぶんない。
        if(StringUtils.compareSubSequence(
                "午後", html, matcher.start(1), matcher.end(1)
           ) == 0){
            hour += 12;
        }
        hour = hour % 24;     // 24時 = 0時

        CharSequence mclass1 = html.subSequence(matcher.start(6),
                                                matcher.end(6)   );
        Talk.Type type = Talk.decodeType(mclass1);
        if(type == null) return startPos;

        Village village = getVillage();
        Land land = village.getParentLand();

        Avatar avatar = village.getAvatar(fullName);
        if(avatar == null){
            avatar = new Avatar(fullName);
            village.addAvatar(avatar);
        }

        String faceimg = matcher.group(4);

        if(type == Talk.Type.GRAVE){
            if(village.getGraveImage() == null ){
                Image graveImage = land.downloadImage(faceimg);
                village.setGraveImage(graveImage);
            }
        }else if(village.getAvatarFaceImage(avatar) == null && faceimg != null){
            Image image = land.downloadImage(faceimg);
            village.putAvatarFaceImage(avatar, image);
        }

        int dialogStart = matcher.start(8);
        int dialogEnd   = matcher.end(8);
        CharSequence dialog = html.subSequence(dialogStart, dialogEnd);
        SortedMap<Long, byte[]> subError
                = html.subEncodeError((long)dialogStart,
                                      (long)dialogEnd   );

        dialog = resolveRef(dialog);  // TODO エンコードエラーとズレまくり

        Talk talk = new Talk(this,
                             type,
                             avatar,
                             anchorID,
                             hour,
                             minute,
                             dialog,
                             subError);

        int[] counts = this.countMap.get(avatar);
        if(counts == null){
            counts = new int[4];
            this.countMap.put(avatar, counts);
        }
        int count = ++counts[type.ordinal()];
        talk.setCount(count);

        addTopic(talk);

        return nextPos;
    }

    /**
     * 発言データをアンロードする。
     */
    public void unload(){
        this.limitHour = 0;
        this.limitMinute = 0;
        this.loginName = null;
        this.isFullOpen = false;

        this.isHot = false;

        this.topicList.clear();
        this.countMap.clear();

        return;
    }
}
