/*
 * system event in game
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: SysEvent.java 455 2009-03-26 17:45:15Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 人狼BBSシステムが生成する各種メッセージ。
 * Topicの具体化。
 */
// TODO 狼の襲撃先表示は Talk か SysEvent どちらにしよう...
public class SysEvent implements Topic{

    /** イベントのおおまかな種別 */
    public static enum EventClass{
        /** &lt;div class="announce"&gt;に対応 */
        ANNOUNCE,
        /** &lt;div class="order"&gt;に対応 */
        ORDER,
        /** &lt;div class="extra"&gt;に対応 */
        EXTRA,
    }

    /** イベント種別。  */
    public static enum Type{
        /** エントリースタート */
        STARTENTRY,
        /** キャラ初登場 */
        ONSTAGE,
        /** 役職確定 */
        STARTMIRROR,
        /** 役職人数内訳発表 */
        OPENROLE,
        /** 襲撃開始 */
        STARTMURDER,
        /** 生存者確認 */
        SURVIVOR,
        /** 投票結果 */
        COUNTING,
        /** 突然死 */
        SUDDENDEATH,
        /** 襲撃成功 */
        MURDER,
        /** 襲撃失敗 */
        NOMURDER,
        /** 村側勝利 */
        WINVILLAGE,
        /** 狼側勝利 */
        WINWOLF,
        /** ハムスター勝利 */
        WINHAMSTER,
        /** 役職&ログイン名公開 */
        PLLIST,
        /** 何らかのシステムトラブル。A国末期で頻発 */
        NOTHING,

        /** 参加者募集 */
        RECRUIT,
        /** 行動確定要求 */
        PROMPT,
        /** 未発言者一覧 */
        WORDLESS,
        /** エピローグ案内 */
        ONEPILOGUE,
        /** 村終了 */
        GAMEOVER,

        /** 占い先表示 */
        SCRY,
        /** 護衛先表示 */
        GUARD,

        /** 未定義 */
        UNKNOWN,

    }

    private static final Pattern defaultRegex
            = Pattern.compile(".+", Pattern.DOTALL);
    private static final Pattern newlineRegex
            = Pattern.compile("\\n*");
    private static final Pattern hhmmRegex
            = Pattern.compile("(?:(午前)|(午後)) ([0-9]+)時 ([0-9]+)分");

    /**
     * イベントクラス文字列をデコードする。
     * @param eventClass イベントクラス文字列
     * @return デコードしたEventClass
     */
    public static EventClass decodeEventClass(CharSequence eventClass){
        EventClass result;
        if("announce".contentEquals(eventClass)){
            result = EventClass.ANNOUNCE;
        }else if("order".contentEquals(eventClass)){
            result = EventClass.ORDER;
        }else if("extra".contentEquals(eventClass)){
            result = EventClass.EXTRA;
        }else{
            assert false;
            return null;
        }

        return result;
    }

    /**
     * イベントクラスを文字列にエンコードする。
     * @param eventClass イベントクラス
     * @return エンコードされた文字列
     */
    public static CharSequence encodeEventClass(EventClass eventClass){
        CharSequence result;
        switch(eventClass){
        case ANNOUNCE:
            result = "announce";
            break;
        case ORDER:
            result = "order";
            break;
        case EXTRA:
            result = "extra";
            break;
        default:
            assert false;
            return null;
        }

        return result;
    }

    /**
     * 前回マッチ部分をリージョンから外す。
     * @param matcher マッチャ
     * @return 引数と同一のマッチャインスタンス
     */
    private static Matcher slideRegion(Matcher matcher){
        int regionStart = matcher.end();
        int regionEnd = matcher.regionEnd();
        matcher.region(regionStart, regionEnd);
        return matcher;
    }

    /**
     * Matcherのリージョン先頭がPatternに一致するか判定する。
     * マッチした場合、リージョンは自動的に前進する。
     * @param pattern パターン
     * @param matcher マッチャ
     * @return 一致すればtrue
     */
    private static boolean lookStep(Pattern pattern, Matcher matcher){
        matcher.usePattern(pattern);
        if( ! matcher.lookingAt() ) return false;
        slideRegion(matcher);
        return true;
    }

    /**
     * 時刻表記の解析を行う。
     * マッチした領域はリージョンから外れる。
     * @param matcher マッチャ
     * @return 深夜零時からの経過分 (hour * 60 + minute)
     */
    private static int lookStepHHMM(Matcher matcher){
        matcher.usePattern(hhmmRegex);
        if( ! matcher.lookingAt() ) return -1;

        boolean isPM;
        if(matcher.start(1) >= 0){
            isPM = false;
        }else if(matcher.start(2) >= 0){
            isPM = true;
        }else{
            assert false;
            return -1;
        }

        int hour   = StringUtils.parseInt(matcher.group(3));
        int minute = StringUtils.parseInt(matcher.group(4));
        if(isPM) hour += 12;
        hour %= 24;

        int result = hour * 60 + minute;

        slideRegion(matcher);

        return result;
    }

    /**
     * 整数表記の解析を行う。
     * 正の数のみ。
     * マッチした領域はリージョンから外れる。
     * @param pattern 整数表記のグループをただ一つ持つパターン
     * @param matcher マッチャ
     * @return 解析された整数
     */
    private static Integer lookStepInt(Pattern pattern, Matcher matcher){
        matcher.usePattern(pattern);
        if( ! matcher.lookingAt() ) return null;

        if(matcher.start(1) < 0) return null;
        int num = StringUtils.parseInt(matcher.group(1));

        slideRegion(matcher);

        return num;
    }

    /**
     * 文字列表記の解析を行う。
     * マッチした領域はリージョンから外れる。
     * @param pattern 文字列表記のグループを一つ以上持つパターン
     * @param matcher マッチャ
     * @return 解析された文字列のリスト
     */
    private static List<String> lookStepString(Pattern pattern,
                                                 Matcher matcher){
        matcher.usePattern(pattern);
        if( ! matcher.lookingAt() ) return null;

        List<String> result = new LinkedList<String>();
        for(int group = 1; group <= matcher.groupCount(); group++){
            if(matcher.start(group) < 0) continue;
            String str = matcher.group(group);
            result.add(str);
        }

        slideRegion(matcher);

        return result;
    }

    /**
     * 最初にマッチしたグループの番号を得る。
     * マッチした領域はリージョンから外れる。
     * @param pattern グループを1つ以上含むパターン
     * @param matcher マッチャ
     * @return マッチしたグループ番号。
     */
    private static int lookStepExcGroup(Pattern pattern, Matcher matcher){
        matcher.usePattern(pattern);
        if( ! matcher.lookingAt() ) return -1;

        int groupCount = matcher.groupCount();
        for(int group = 1; group <= groupCount; group++){
            if(matcher.start(group) >= 0){
                slideRegion(matcher);
                return group;
            }
        }

        slideRegion(matcher);

        return -1;
    }

    /**
     * Matcherのリージョン先頭がAvatar名のいずれかに一致するか判定する。
     * マッチした領域はリージョンから外れる。
     * @param matcher マッチャ
     * @return 名前が一致したAvatar
     */
    private static Avatar lookStepAvatar(Matcher matcher){
        Avatar avatar = Avatar.lookingAtAvatar(matcher);
        if(avatar == null) return null;
        slideRegion(matcher);
        return avatar;
    }

    /**
     * Matcherのリージョン先頭がRole名のいずれかに一致するか判定する。
     * マッチした領域はリージョンから外れる。
     * @param matcher マッチャ
     * @return 名前が一致したRole
     */
    private static GameRole lookStepRole(Matcher matcher){
        GameRole role = GameRole.lookingAtRole(matcher);
        if(role == null) return null;
        slideRegion(matcher);
        return role;
    }

    /**
     * マッチャの残りリージョン全部が0個以上の改行のみか判定する。
     * マッチした領域はリージョンから外れる。
     * @param matcher マッチャ
     * @return 0個以上の改行のみならtrue
     */
    private static boolean hasCleanTail(Matcher matcher){
        matcher.usePattern(newlineRegex);
        if( ! matcher.matches() ) return false;
        slideRegion(matcher);
        return true;
    }

    /**
     * デフォルトマッチャを生成する。
     * @param seq 解析対象文字列
     * @param start 解析開始位置
     * @param end 解析終了位置
     * @return マッチャ
     */
    private static Matcher getDefaultMatcher(CharSequence seq,
                                               int start,
                                               int end){
        Matcher matcher = defaultRegex.matcher(seq);
        matcher.region(start, end);
        return matcher;
    }

    /**
     * 実体参照の解決を行う。
     * 「&gt;」「&lt;」「&quot;」「&amp;」の4種のみ。
     * 今のところこれ以外の実体参照は人狼BBSで現れないはず。
     * @param text 実体参照を含むテキスト
     * @return 解決済みのテキスト
     */
    // TODO CharSequenceベースでの書き直し。省インスタンス化。
    private String replaceRef(String text){
        text = text.replace("&gt;",">")
                   .replace("&lt;","<")
                   .replace("&quot;","\"")
                   .replace("&amp;","&");
        return text;
    }

    private final EventClass eventClass;
    private CharSequence message;
    private Type type = Type.UNKNOWN;

    private final List<Avatar>   avatarList  = new LinkedList<Avatar>();
    private final List<GameRole> roleList    = new LinkedList<GameRole>();
    private final List<Integer>  integerList = new LinkedList<Integer>();
    private final List<CharSequence>  charseqList
            = new LinkedList<CharSequence>();

    /**
     * システムイベントを生成する。
     * @param eventClass イベント種別。
     * @param message イベントメッセージ（HTMLタグ含む）
     * @throws java.lang.NullPointerException null引数の検出
     */
    public SysEvent(EventClass eventClass,
                     CharSequence message )
            throws NullPointerException{
        super();
        if(eventClass == null || message == null){
            throw new NullPointerException();
        }

        this.eventClass = eventClass;
        this.message = message.toString().replace("<br />","\n");

        boolean matched;
        int start = 0;
        int end = this.message.length();

        switch(this.eventClass){
        case ANNOUNCE:
            matched = parseAnnounce(this.message, start, end);
            break;
        case ORDER:
            matched = parseOrder(this.message, start, end);
            break;
        case EXTRA:
            matched = parseExtra(this.message, start, end);
            break;
        default:
            assert false;
            matched = false;
            break;
        }

        if( ! matched ){
            this.type = Type.UNKNOWN;
            clearList();
            Jindolf.logger
                   .severe("解析不能なシステムメッセージが出現しました");
            return;
        }

        if(this.type == Type.PLLIST){
            this.message = this.message.toString()
                                       .replaceAll("<a [^>]*>", "")
                                       .replace("</a>", "");
        }

        return;
    }

    /**
     * 各種リストデータをクリアする。
     */
    private void clearList(){
        this.avatarList .clear();
        this.roleList   .clear();
        this.integerList.clear();
        this.charseqList.clear();
        return;
    }

    /**
     * announceメッセージの解析を行う。
     * @param seq 解析対象文字列
     * @param start 解析開始インデックス
     * @param end 解析終了インデックス
     * @return 解析に成功すればtrue
     */
    private boolean parseAnnounce(CharSequence seq, int start, int end){
        if(parseSimpleAnnounce(seq, start, end)) return true;

        Matcher matcher = getDefaultMatcher(seq, start, end);

        matcher.region(start, end);
        if(parseOnStage(matcher)) return true;

        matcher.region(start, end);
        if(parseOpenRole(matcher)) return true;

        matcher.region(start, end);
        if(parseSurvivor(matcher)) return true;

        matcher.region(start, end);
        if(parseCounting(matcher)) return true;

        matcher.region(start, end);
        if(parseSuddenDeath(matcher)) return true;

        matcher.region(start, end);
        if(parseMurder(matcher)) return true;

        matcher.region(start, end);
        if(parsePlList(matcher)) return true;

        return false;
    }

    private static final Pattern startentryRegex
            = Pattern.compile(
             "昼間は人間のふりをして、夜に正体を現すという人狼。\\n"
            +"その人狼が、この村に紛れ込んでいるという噂が広がった。\\n\\n"
            +"村人達は半信半疑ながらも、"
            +"村はずれの宿に集められることになった。");
    private static final Pattern startmirrorRegex
            = Pattern.compile(
             "さあ、自らの姿を鏡に映してみよう。\\n"
            +"そこに映るのはただの村人か、それとも血に飢えた人狼か。\\n\\n"
            +"例え人狼でも、多人数で立ち向かえば怖くはない。\\n"
            +"問題は、だれが人狼なのかという事だ。\\n"
            +"占い師の能力を持つ人間ならば、それを見破れるだろう。");
    private static final Pattern startmurderRegex
            = Pattern.compile(
             "ついに犠牲者が出た。人狼はこの村人達のなかにいる。\\n"
            +"しかし、それを見分ける手段はない。\\n\\n"
            +"村人達は、疑わしい者を排除するため、"
            +"投票を行う事にした。\\n"
            +"無実の犠牲者が出るのもやむをえない。"
            +"村が全滅するよりは……。\\n\\n"
            +"最後まで残るのは村人か、それとも人狼か。");
    private static final Pattern nomurderRegex
            = Pattern.compile(
             "今日は犠牲者がいないようだ。人狼は襲撃に失敗したのだろうか。");
    private static final Pattern winvillageRegex
            = Pattern.compile(
             "全ての人狼を退治した……。人狼に怯える日々は去ったのだ！");
    private static final Pattern winwolfRegex
            = Pattern.compile(
             "もう人狼に抵抗できるほど村人は残っていない……。\\n"
            +"人狼は残った村人を全て食らい、"
            +"別の獲物を求めてこの村を去っていった。");
    private static final Pattern winhamsterRegex
            = Pattern.compile(
              "全ては終わったかのように見えた。\\n"
             +"だが、奴が生き残っていた……。");
    private static final Pattern nothingRegex
            = Pattern.compile("……。");

    private static Object[][] simpleRegexToType = {
        { startentryRegex,  Type.STARTENTRY  },
        { startmirrorRegex, Type.STARTMIRROR },
        { startmurderRegex, Type.STARTMURDER },
        { nomurderRegex,    Type.NOMURDER    },
        { winvillageRegex,  Type.WINVILLAGE  },
        { winwolfRegex,     Type.WINWOLF     },
        { winhamsterRegex,  Type.WINHAMSTER  },
        { nothingRegex,     Type.NOTHING     },
    };

    /**
     * 固定したシンプルなannounceメッセージを解析する。
     * @param seq 解析対象文字列
     * @param start 解析開始インデックス
     * @param end 解析終了インデックス
     * @return 解析に成功すればtrue
     */
    private boolean parseSimpleAnnounce(CharSequence seq,
                                           int start,
                                           int end ){
        clearList();

        Matcher matcher = getDefaultMatcher(seq, start, end);

        Type matchedType = null;

        for(Object[] pair : simpleRegexToType){
            Pattern pattern = (Pattern)( pair[0] );
            Type msgtype    = (Type)   ( pair[1] );

            matcher.region(start, end);
            if( lookStep(pattern, matcher) ){
                if( hasCleanTail(matcher) ){
                    matchedType = msgtype;
                    break;
                }
            }
        }

        if(matchedType == null) return false;

        this.type = matchedType;

        return true;
    }

    private static final Pattern onstageRegex_1
            = Pattern.compile("([0-9]+)人目、");
    private static final Pattern onstageRegex_2
            = Pattern.compile("。");

    /**
     * ONSTAGEメッセージのパースを行う。
     * @param matcher マッチャ
     * @return ONSTAGEメッセージだったらtrue
     */
    private boolean parseOnStage(Matcher matcher){
        clearList();

        Integer orderNum = lookStepInt(onstageRegex_1, matcher);
        if(orderNum == null) return false;

        Avatar avatar = lookStepAvatar(matcher);
        if(avatar == null) return false;

        if( ! lookStep(onstageRegex_2, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.ONSTAGE;
        this.integerList.add(orderNum);
        this.avatarList.add(avatar);

        return true;
    }

    private static final Pattern openroleRegex_1
            = Pattern.compile("どうやらこの中には、");
    private static final Pattern openroleRegex_2
            = Pattern.compile("が([0-9]+)名、?");
    private static final Pattern openroleRegex_3
            = Pattern.compile("いるようだ。");

    /**
     * OPENROLEメッセージのパースを行う。
     * @param matcher マッチャ
     * @return OPENROLEメッセージだったらtrue
     */
    private boolean parseOpenRole(Matcher matcher){
        clearList();

        if( ! lookStep(openroleRegex_1, matcher) ) return false;

        for(;;){
            GameRole role = lookStepRole(matcher);
            if(role == null) break;

            Integer num = lookStepInt(openroleRegex_2, matcher);
            if(num == null) return false;

            this.roleList.add(role);
            this.integerList.add(num);
        }

        if( ! lookStep(openroleRegex_3, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.OPENROLE;

        return true;
    }

    private static final Pattern survivorRegex_1
            = Pattern.compile("現在の生存者は");
    private static final Pattern survivorRegex_2
            = Pattern.compile("、");
    private static final Pattern survivorRegex_3
            = Pattern.compile(" の ([0-9]*) 名。");

    /**
     * SURVIVORメッセージのパースを行う。
     * @param matcher マッチャ
     * @return SURVIVORメッセージだったらtrue
     */
    private boolean parseSurvivor(Matcher matcher){
        clearList();

        if( ! lookStep(survivorRegex_1, matcher) ) return false;

        for(;;){
            if( ! lookStep(survivorRegex_2, matcher) ) break;

            Avatar avatar = lookStepAvatar(matcher);
            if(avatar == null) return false;

            this.avatarList.add(avatar);
        }

        Integer num = lookStepInt(survivorRegex_3, matcher);
        if(num == null) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.SURVIVOR;
        this.integerList.add(num);

        return true;
    }

    private static final Pattern countingRegex_1
            = Pattern.compile(" は ");
    private static final Pattern countingRegex_2
            = Pattern.compile(" に投票した。\\n*");
    private static final Pattern countingRegex_3
            = Pattern.compile(" は村人達の手により処刑された。");

    /**
     * COUNTINGメッセージのパースを行う。
     * @param matcher マッチャ
     * @return COUNTINGメッセージだったらtrue
     */
    private boolean parseCounting(Matcher matcher){
        clearList();

        Avatar victim = null;
        for(;;){
            Avatar avatarBy = lookStepAvatar(matcher);
            if(avatarBy == null) break;
            if( ! lookStep(countingRegex_1, matcher) ){
                if( ! lookStep(countingRegex_3, matcher) ) return false;
                victim = avatarBy;
                break;
            }
            Avatar avatarWho = lookStepAvatar(matcher);
            if(avatarWho == null) return false;
            if( ! lookStep(countingRegex_2, matcher) ) return false;
            this.avatarList.add(avatarBy);
            this.avatarList.add(avatarWho);
        }

        if(victim != null){
            this.avatarList.add(victim);
        }

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.COUNTING;

        return true;
    }

    private static final Pattern suddendeathRegex_1
            = Pattern.compile(" は、突然死した。");

    /**
     * SUDDENDEATHメッセージのパースを行う。
     * @param matcher マッチャ
     * @return SUDDENDEATHメッセージだったらtrue
     */
    private boolean parseSuddenDeath(Matcher matcher){
        clearList();

        Avatar avatar = lookStepAvatar(matcher);
        if(avatar == null) return false;

        if( ! lookStep(suddendeathRegex_1, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.SUDDENDEATH;
        this.avatarList.add(avatar);

        return true;
    }

    private static final Pattern murderRegex_1
            = Pattern.compile("次の日の朝、");
    private static final Pattern murderRegex_2
            = Pattern.compile(" と ");
    private static final Pattern murderRegex_3
            = Pattern.compile(" が無残な姿で発見された。");

    /**
     * MURDERメッセージのパースを行う。
     * @param matcher マッチャ
     * @return MURDERメッセージだったらtrue
     */
    private boolean parseMurder(Matcher matcher){
        clearList();

        if( ! lookStep(murderRegex_1, matcher) ) return false;

        Avatar avatar1 = lookStepAvatar(matcher);
        if(avatar1 == null) return false;

        Avatar avatar2 = null;
        if( lookStep(murderRegex_2, matcher) ){
            avatar2 = lookStepAvatar(matcher);
            if(avatar2 == null) return false;
        }

        if( ! lookStep(murderRegex_3, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.MURDER;
        this.avatarList.add(avatar1);
        if(avatar2 != null){
            this.avatarList.add(avatar2);
        }

        return true;
    }

    private static final Pattern pllistRegex_1
            = Pattern.compile(" （([^<]+?)）、");
    private static final Pattern pllistRegex_2
            = Pattern.compile(" （(?:<a href=\"([^\"]*)\">([^<]+)</a>)）、");
    private static final Pattern pllistRegex_3
            = Pattern.compile("(?:(生存)|(死亡))。");
    private static final Pattern pllistRegex_4
            = Pattern.compile("だった。\\n*");

    /**
     * PLLISTメッセージのパースを行う。
     * @param matcher マッチャ
     * @return PLLISTメッセージだったらtrue
     */
    private boolean parsePlList(Matcher matcher){
        clearList();

        for(;;){
            Avatar avatar = lookStepAvatar(matcher);
            if(avatar == null) break;

            String urlText;
            String name;
            List<String> idList;
            idList = lookStepString(pllistRegex_1, matcher);
            if(idList != null && idList.size() == 1){
                urlText = "";
                name    = idList.get(0);
            }else{
                idList = lookStepString(pllistRegex_2, matcher);
                if(idList == null || idList.size() != 2) return false;
                urlText = idList.get(0);
                name    = idList.get(1);
            }
            urlText = replaceRef(urlText);
            name    = replaceRef(name);

            int liveOrDead;
            int group = lookStepExcGroup(pllistRegex_3, matcher);
            if     (group == 1) liveOrDead = 1; // live
            else if(group == 2) liveOrDead = 0; // dead
            else                return false;

            GameRole role = lookStepRole(matcher);
            if(role == null) return false;

            if( ! lookStep(pllistRegex_4, matcher) ) return false;

            this.avatarList .add(avatar);
            this.charseqList.add(urlText);
            this.charseqList.add(name);
            this.integerList.add(liveOrDead);
            this.roleList   .add(role);
        }

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.PLLIST;

        return true;
    }

    /**
     * orderメッセージの解析を行う
     * @param seq 解析対象文字列
     * @param start 解析開始インデックス
     * @param end 解析終了インデックス
     * @return 解析に成功すればtrue
     */
    private boolean parseOrder(CharSequence seq, int start, int end){
        Matcher matcher = getDefaultMatcher(seq, start, end);

        matcher.region(start, end);
        if(parseOnEpilogue(matcher)) return true;

        matcher.region(start, end);
        if(parsePrompt(matcher)) return true;

        matcher.region(start, end);
        if(parseWordless(matcher)) return true;

        matcher.region(start, end);
        if(parseRecruit(matcher)) return true;

        matcher.region(start, end);
        if(parseGameover(matcher)) return true;

        return false;
    }

    private static final Pattern gameoverRegex
            = Pattern.compile("終了しました。");

    /**
     * GAMEOVERメッセージのパースを行う。
     * Jindolfではたぶん出てこないはず。
     * @param matcher マッチャ
     * @return GAMEOVERメッセージだったらtrue
     */
    private boolean parseGameover(Matcher matcher){
        clearList();

        if( ! lookStep(gameoverRegex, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.GAMEOVER;

        return true;
    }

    private static final Pattern recruitRegex_1
            = Pattern.compile(
            "演じたいキャラクターを選び、発言してください。\\n");
    private static final Pattern recruitRegex_2
            = Pattern.compile(
            " に([0-9]+)名以上がエントリーしていれば進行します。\\n");
    private static final Pattern recruitRegex_3
            = Pattern.compile("最大([0-9]+)名まで参加可能です。\\n\\n");
    private static final Pattern recruitRegex_4
            = Pattern.compile(
             "※エントリーは取り消せません。"
            +"ルールをよく理解した上でご参加下さい。\\n"
            +"※始めての方は、村人希望での参加となります。\\n"
            +"※希望能力についての発言は控えてください。");

    /**
     * RECRUITメッセージのパースを行う。
     * @param matcher マッチャ
     * @return RECRUITメッセージだったらtrue
     */
    private boolean parseRecruit(Matcher matcher){
        clearList();

        if( ! lookStep(recruitRegex_1, matcher) ) return false;

        int hhmm = lookStepHHMM(matcher);
        if(hhmm < 0) return false;

        Integer minMember = lookStepInt(recruitRegex_2, matcher);
        if(minMember == null) return false;

        Integer maxMember = lookStepInt(recruitRegex_3, matcher);
        if(maxMember == null) return false;

        if( ! lookStep(recruitRegex_4, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.RECRUIT;
        this.integerList.add(hhmm);
        this.integerList.add(minMember);
        this.integerList.add(maxMember);

        return true;
    }

    private static final Pattern wordlessRegex_1
            = Pattern.compile("本日まだ発言していない者は、");
    private static final Pattern wordlessRegex_2
            = Pattern.compile("、");
    private static final Pattern wordlessRegex_3
            = Pattern.compile("以上 ([0-9]+) 名。");

    /**
     * WORDLESSメッセージのパースを行う。
     * @param matcher マッチャ
     * @return WORDLESSメッセージだったらtrue
     */
    private boolean parseWordless(Matcher matcher){
        clearList();

        if( ! lookStep(wordlessRegex_1, matcher) ) return false;

        for(;;){
            Avatar avatar = lookStepAvatar(matcher);
            if(avatar == null) break;
            this.avatarList.add(avatar);
            if( ! lookStep(wordlessRegex_2, matcher) ) return false;
        }

        Integer num = lookStepInt(wordlessRegex_3, matcher);
        if(num == null) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.WORDLESS;
        this.integerList.add(num);

        return true;
    }

    private static final Pattern promptRegex_1
            = Pattern.compile(
             " までに、誰を処刑するべきかの投票先を決定して下さい。\\n"
            +"一番票を集めた人物が処刑されます。"
            +"同数だった場合はランダムで決定されます。\\n\\n");
    private static final Pattern promptRegex_2
            = Pattern.compile("特殊な能力を持つ人は、");
    private static final Pattern promptRegex_3
            = Pattern.compile(" までに行動を確定して下さい。");

    /**
     * PROMPTメッセージのパースを行う。
     * @param matcher マッチャ
     * @return PROMPTメッセージだったらtrue
     */
    private boolean parsePrompt(Matcher matcher){
        clearList();

        int hhmm1 = lookStepHHMM(matcher);
        if(hhmm1 >= 0){
            if( ! lookStep(promptRegex_1, matcher) ) return false;
        }

        if( ! lookStep(promptRegex_2, matcher) ) return false;

        int hhmm2 = lookStepHHMM(matcher);
        if(hhmm2 < 0) return false;

        if( ! lookStep(promptRegex_3, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.PROMPT;
        if(hhmm1 >= 0) this.integerList.add(hhmm1);
        this.integerList.add(hhmm2);

        return true;
    }

    private static final Pattern onepilogueRegex_1
            = Pattern.compile(
             "側の勝利です！\\n全てのログとユーザー名を公開します。");
    private static final Pattern onepilogueRegex_2
            = Pattern.compile(
             " まで自由に書き込めますので、今回の感想などをどうぞ。");

    /**
     * ONEPILOGUEメッセージのパースを行う。
     * @param matcher マッチャ
     * @return ONEPILOGUEメッセージだったらtrue
     */
    private boolean parseOnEpilogue(Matcher matcher){
        clearList();

        GameRole role = lookStepRole(matcher);
        if(role == null) return false;
        if(   role != GameRole.HABITANT
           && role != GameRole.WOLF
           && role != GameRole.HAMSTER ){
            return false;
        }

        if( ! lookStep(onepilogueRegex_1, matcher) ) return false;

        int hhmm = lookStepHHMM(matcher);
        if(hhmm < 0) return false;

        if( ! lookStep(onepilogueRegex_2, matcher) ) return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.type = Type.ONEPILOGUE;
        this.roleList.add(role);
        this.integerList.add(hhmm);

        return true;
    }

    private static final Pattern waRegex
            = Pattern.compile(" は、");
    private static final Pattern actionRegex
            = Pattern.compile("( を占った。)|( を守っている。)");

    /**
     * extraメッセージの解析を行う
     * @param seq 解析対象文字列
     * @param regionStart 解析開始インデックス
     * @param regionEnd 解析終了インデックス
     * @return 解析に成功すればtrue
     */
    private boolean parseExtra(CharSequence seq,
                                int regionStart,
                                int regionEnd ){
        Matcher matcher = getDefaultMatcher(seq, regionStart, regionEnd);
        boolean result = parseExtra(matcher);
        return result;
    }

    /**
     * extraメッセージの解析を行う
     * @param matcher マッチャ
     * @return 解析に成功すればtrue
     */
    private boolean parseExtra(Matcher matcher){
        clearList();

        Avatar avatarBy = lookStepAvatar(matcher);
        if(avatarBy == null) return false;

        if( ! lookStep(waRegex, matcher) ) return false;

        Avatar avatarWho = lookStepAvatar(matcher);
        if(avatarWho == null) return false;

        int group = lookStepExcGroup(actionRegex, matcher);
        if     (group == 1) this.type = Type.SCRY;
        else if(group == 2) this.type = Type.GUARD;
        else                return false;

        if( ! hasCleanTail(matcher) ) return false;

        this.avatarList.add(avatarBy);
        this.avatarList.add(avatarWho);

        return true;
    }

    /**
     * Avatarリストを取得する。
     * @return Avatarリスト
     */
    public List<Avatar> getAvatarList(){
        List<Avatar> result = Collections.unmodifiableList(this.avatarList);
        return result;
    }

    /**
     * Roleリストを取得する。
     * @return Roleリスト
     */
    public List<GameRole> getRoleList(){
        List<GameRole> result = Collections.unmodifiableList(this.roleList);
        return result;
    }

    /**
     * Integerリストを取得する。
     * @return Integerリスト
     */
    public List<Integer> getIntegerList(){
        List<Integer> result = Collections.unmodifiableList(this.integerList);
        return result;
    }

    /**
     * CharSequenceリストを取得する。
     * @return CharSequenceリスト
     */
    public List<CharSequence> getCharSequenceList(){
        List<CharSequence> result
                = Collections.unmodifiableList(this.charseqList);
        return result;
    }

    /**
     * イベントメッセージを取得する。
     * @return イベントメッセージ
     */
    public CharSequence getMessage(){
        return this.message;
    }

    /**
     * イベントクラスを取得する。
     * @return イベントクラス
     */
    public EventClass getEventClass(){
        return this.eventClass;
    }

    /**
     * イベント種別を取得する。
     * @return イベント種別
     */
    public Type getType(){
        return this.type;
    }
}
