/*
 * land
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Land.java 508 2009-04-29 10:29:00Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import jp.sourceforge.jindolf.core.LandDef;
import jp.sourceforge.jindolf.core.LandState;
import jp.sourceforge.jindolf.core.VillageState;
import java.awt.Image;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * いわゆる「国」
 */
public class Land {

    private static final String Spchar = "\u0020\\t\\n\\r";
    private static final String Sp = "[" +Spchar+ "]";
    private static final String Sp_n = Sp + "*";
    private static final String Sp_m = Sp + "+";

    private static final Pattern anchorRegex;

    static{
        String anchorStart = "<a" +Sp_m+ "href=\"([^\"]*)\"" +Sp_n+ ">";
        String anchorClose = "</a>";
        String deadline = Sp_n+ "<strong>[^<]*</strong>" +Sp_n+ "</td>";
        String status = Sp_n+ "<td>([^<]*)</td>";
        String option = "(?:"+deadline+status+")?";
        String anchor = anchorStart +"([^<]*)"+ anchorClose + option;

        anchorRegex = Pattern.compile(anchor, Pattern.DOTALL);
    }

    /**
     * クエリー文字列から特定キーの値を得る。
     * クエリーの書式例：「a=b&c=d&e=f」この場合キーcの値はd
     * @param key キー
     * @param allQuery クエリー
     * @return 値
     */
    public static String getValueFromCGIQueries(String key,
                                                   String allQuery){
        String result = null;

        String[] queries = allQuery.split("\\Q&\\E");
        if(queries == null) return null;

        for(String pair : queries){
            if(pair == null) continue;
            String[] namevalue = pair.split("\\Q=\\E");
            if(namevalue == null) continue;
            if(namevalue.length != 2) continue;
            String name  = namevalue[0];
            String value = namevalue[1];
            if(name == null) continue;
            if( name.equals(key) ){
                result = value;
                if(result == null) continue;
                if(result.length() <= 0) continue;
                break;
            }
        }

        return result;
    }

    /**
     * AタグのHREF属性値から村IDを得る。
     * @param hrefValue HREF値
     * @return village 村ID
     */
    public static String getVillageIDFromHREF(CharSequence hrefValue){
        String allQuery = NetUtil.getQueryFromHREF(hrefValue);
        if(allQuery == null) return null;

        String villageID = getValueFromCGIQueries("vid", allQuery);
        if(villageID == null) return null;
        if(villageID.length() <= 0) return null;

        return villageID;
    }

    private final LandDef landDef;
    private final ServerAccess serverAccess;

    private final List<Village> villageList = new LinkedList<Village>();

    /**
     * コンストラクタ
     * @param landDef 国定義
     * @throws java.lang.IllegalArgumentException 不正な国定義
     */
    public Land(LandDef landDef) throws IllegalArgumentException{
        super();

        this.landDef = landDef;

        URL url;
        try{
            url = this.landDef.getCgiURI().toURL();
        }catch(MalformedURLException e){
            throw new IllegalArgumentException(e);
        }
        this.serverAccess = new ServerAccess(url);

        return;
    }

    /**
     * 国定義を得る。
     * @return 国定義
     */
    public LandDef getLandDef(){
        return this.landDef;
    }

    /**
     * サーバ接続を返す。
     * @return ServerAccessインスタンス
     */
    public ServerAccess getServerAccess(){
        return this.serverAccess;
    }

    /**
     * 指定されたインデックス位置の村を返す。
     * @param index 0から始まるインデックス値
     * @return 村
     */
    public Village getVillage(int index){
        if(index < 0)                  return null;
        if(index >= getVillageCount()) return null;

        Village result = this.villageList.get(index);
        return result;
    }

    /**
     * 村の総数を返す。
     * @return 村の総数
     */
    public int getVillageCount(){
        int result = this.villageList.size();
        return result;
    }

    /**
     * 村のリストを返す。
     * @return 村のリスト
     */
    // TODO インスタンス変数でいいはず。
    public List<Village> getVillageList(){
        return Collections.unmodifiableList(this.villageList);
    }

    /**
     * 絶対または相対URLの指すパーマネントなイメージ画像をダウンロードする。
     * ※ A,B,D 国の顔アイコンは絶対パスらしい…。
     * @param imageURL 画像URL文字列
     * @return 画像イメージ
     */
    public Image downloadImage(String imageURL){
        ServerAccess server = getServerAccess();
        Image image;
        try{
            image = server.downloadImage(imageURL);
        }catch(IOException e){
            Jindolf.logger.log(Level.WARNING,
                                 "イメージ[" + imageURL + "]"
                               + "のダウンロードに失敗しました",
                                 e                               );
            return null;
        }
        return image;
    }

    /**
     * HTMLデータから村情報を抽出し村一覧を生成する。
     * @param html HTMLデータ
     * @return 村一覧
     */
    private Collection<Village> parseVillageList(CharSequence html){
        Collection<Village> result = new Vector<Village>();

        Matcher matcher = anchorRegex.matcher(html);
        for(;;){
            if( ! matcher.find() ) break;
            String hrefValue  = matcher.group(1);
            String anchorText = matcher.group(2);
            String option     = matcher.group(3);

            if(hrefValue == null) continue;
            if(hrefValue.length() <= 0) continue;
            if(anchorText == null) anchorText = "";

            String villageID = getVillageIDFromHREF(hrefValue);
            if(villageID == null) continue;
            if(villageID.length() <= 0) continue;

            String villageName = anchorText.replaceAll(Sp_m, "\u0020").trim();

            String[] parts = villageName.split("\u0020");
            if(   parts != null
               && parts[0] != null
               && parts[0].equals(villageID) ){
                villageName = getLandDef().getLandPrefix() + villageName;
            }

            Village village = new Village(this, villageID, villageName);

            VillageState vstate;
            if(   option == null
               || getLandDef().getLandState() == LandState.HISTORICAL ){
                vstate = VillageState.GAMEOVER;
            }else if(option.equals("参加者募集中です。")){
                vstate = VillageState.PROLOGUE;
            }else if(option.equals("開始待ちです。")){
                vstate = VillageState.PROLOGUE;
            }else if(option.equals("進行中です。")){
                vstate = VillageState.PROGRESS;
            }else if(option.equals("勝敗が決定しました。")){
                vstate = VillageState.EPILOGUE;
            }else if(option.equals("終了・ログ公開中。")){
                vstate = VillageState.GAMEOVER;
            }else{
                vstate = VillageState.UNKNOWN;
            }
            village.setState(vstate);

            result.add(village);
        }

        return result;
    }

    /**
     * 村リストを更新する。
     * 元情報は国のトップページと村一覧ページ。
     * 村リストはVillageの実装に従いソートされる。重複する村は排除。
     * @throws java.io.IOException ネットワーク入出力の異常
     */
    public void updateVillageList() throws IOException{
        Collection<Village> vcol;
        SortedSet<Village> vset = new TreeSet<Village>();

        ServerAccess server = getServerAccess();

        HtmlSequence html;

        html = server.getHTMLTopPage();
        vcol = parseVillageList(html);
        vset.addAll(vcol);

        html = server.getHTMLLandList();
        vcol = parseVillageList(html);
        vset.addAll(vcol);

        this.villageList.clear();
        this.villageList.addAll(vset);

        return;
    }

    /**
     * 国の文字列表現を返す。
     * @return 文字列表現
     */
    @Override
    public String toString(){
        return getLandDef().getLandName();
    }

}
