/*
 * Village
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Village.java 511 2009-05-21 17:04:10Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import jp.sourceforge.jindolf.core.PeriodType;
import jp.sourceforge.jindolf.core.LandState;
import jp.sourceforge.jindolf.core.VillageState;
import java.awt.Image;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * いわゆる「村」
 */
public class Village implements Comparable<Village>{

    /**
     * 村同士を比較するためのComparator
     */
    private static class VillageComparator implements Comparator<Village>{
        /**
         * {@inheritDoc}
         * @param v1 {@inheritDoc}
         * @param v2 {@inheritDoc}
         * @return {@inheritDoc}
         */
        public int compare(Village v1, Village v2){
            if     ( v1 == null && v2 == null ) return 0;
            else if( v1 == null && v2 != null ) return -1;
            else if( v1 != null && v2 == null ) return +1;

            int v1Num = v1.getVillageIDNum();
            int v2Num = v2.getVillageIDNum();

            return v1Num - v2Num;
        }
    }

    private static final Comparator<Village> vComparator
            = new VillageComparator();

    private static final Pattern anchorRegex;
    private static final Pattern titleRegex;
    private static final Pattern villageInfoRegex;

    static{
        String Spchar = "\u0020\\t\\n\\r";
        String Sp = "[" +Spchar+ "]";
        String Sp_n = Sp + "*";
        String Sp_m = Sp + "+";

        String prologue = "プロローグ";
        String epilogue = "エピローグ";
        String gameover = "終了";
        String progress = "([1-9][0-9]*)日目";
        String anchorStart = "<a" +Sp_m+ "href=\"([^\"]*)\"" +Sp_n+ ">" +Sp_n;
        String anchorClose = Sp_n+ "</a>";
        String content = "("+
                            prologue +
                         "|"+
                            epilogue +
                         "|"+
                            progress +
                         "|"+
                            gameover +
                         ")";
        String anchor = anchorStart + content + anchorClose;
        String title = "<title>" +Sp_n+ "([^<]*)" +Sp_n+ "</title>";
        String villageInfo =
                ">" +Sp_n+ "([^<]+?)" +Sp_n+
                "<strong>" +Sp_n+
                    "[\\(（]" +Sp_n+
                    "([1-9][0-9]?)/([1-9][0-9]?)" +Sp_n+
                    "(午前|午後)?" +Sp_n+
                    "([0-9][0-9]?)(?:時|\\:)" +Sp_n+
                    "([0-9][0-9]?)分?" +Sp_n+
                    "に更新" +Sp_n+
                    "[\\)）]" +Sp_n+
                "</strong>";

        anchorRegex      = Pattern.compile(anchor,      Pattern.DOTALL);
        titleRegex       = Pattern.compile(title,       Pattern.DOTALL);
        villageInfoRegex = Pattern.compile(villageInfo, Pattern.DOTALL);
    }

    /**
     * 村同士を比較するためのComparatorを返す。
     * @return Comparatorインスタンス
     */
    public static Comparator<Village> comparator(){
        return vComparator;
    }

    private final Land parentLand;
    private final String villageID;
    private final int villageIDNum;
    private final String villageName;

    private final boolean isValid;

    private int limitMonth;
    private int limitDay;
    private int limitHour;
    private int limitMinute;

    private VillageState state = VillageState.UNKNOWN;

    private final LinkedList<Period> periodList = new LinkedList<Period>();
    private final List<Period> unmodList =
            Collections.unmodifiableList(this.periodList);

    private Image tombImage = null;

    private final Map<String, Avatar> avatarMap =
            new HashMap<String, Avatar>();
    private final Map<Avatar, Image> faceImageMap =
            new HashMap<Avatar, Image>();

    /**
     * Villageを生成する。
     * @param parentLand Villageの所属する国
     * @param villageID 村のID
     * @param villageName 村の名前
     */
    public Village(Land parentLand, String villageID, String villageName) {
        this.parentLand    = parentLand;
        this.villageID   = villageID.intern();
        this.villageIDNum = Integer.parseInt(this.villageID);
        this.villageName = villageName.intern();

        this.isValid = this.parentLand.getLandDef()
                       .isValidVillageId(this.villageIDNum);

        return;
    }

    /**
     * 所属する国を返す。
     * @return 村の所属する国（Land）
     */
    public Land getParentLand(){
        return this.parentLand;
    }

    /**
     * 村のID文字列を返す。
     * @return 村ID
     */
    public String getVillageID(){
        return this.villageID;
    }

    /**
     * 村のID数値を返す。
     * @return 村ID
     */
    public int getVillageIDNum(){
        return this.villageIDNum;
    }

    /**
     * 村の名前を返す。
     * @return 村の名前
     */
    public String getVillageName(){
        return this.parentLand.getLandDef().getLandPrefix() + getVillageID();
    }

    /**
     * 村の長い名前を返す。
     * @return 村の長い名前
     */
    public String getVillageFullName(){
        return this.villageName;
    }

    /**
     * 村の状態を返す。
     * @return 村の状態
     */
    public VillageState getState(){
        return this.state;
    }

    /**
     * 村の状態を設定する。
     * @param state 村の状態
     */
    public void setState(VillageState state){
        this.state = state;
        return;
    }

    /**
     * プロローグを返す。
     * @return プロローグ
     */
    public Period getPrologue(){
        for(Period period : this.periodList){
            if(period.isPrologue()) return period;
        }
        return null;
    }

    /**
     * エピローグを返す。
     * @return エピローグ
     */
    public Period getEpilogue(){
        for(Period period : this.periodList){
            if(period.isEpilogue()) return period;
        }
        return null;
    }

    /**
     * 指定された日付の進行日を返す。
     * @param day 日付
     * @return Period
     */
    public Period getProgress(int day){
        for(Period period : this.periodList){
            if(   period.isProgress()
               && period.getDay() == day ) return period;
        }
        return null;
    }

    /**
     * PROGRESS状態のPeriodの総数を返す。
     * @return PROGRESS状態のPeriod総数
     */
    public int getProgressDays(){
        int result = 0;
        for(Period period : this.periodList){
            if(period.isProgress()) result++;
        }
        return result;
    }

    /**
     * 指定されたPeriodインデックスのPeriodを返す。
     * プロローグやエピローグへのアクセスも可能。
     * @param day Periodインデックス
     * @return Period
     */
    public Period getPeriod(int day){
        return this.periodList.get(day);
    }

    /**
     * 指定されたアンカーの対象のPeriodを返す。
     * @param anchor アンカー
     * @return Period
     */
    public Period getPeriod(Anchor anchor){
        Period anchorPeriod;

        if(anchor.isEpilogueDay()){
            anchorPeriod = getEpilogue();
            return anchorPeriod;
        }

        int anchorDay = anchor.getDay();
        anchorPeriod = getPeriod(anchorDay);

        return anchorPeriod;
    }

    /**
     * Period総数を返す。
     * @return Period総数
     */
    public int getPeriodSize(){
        return this.periodList.size();
    }

    /**
     * Periodへのリストを返す。
     * @return Periodのリスト。
     */
    public List<Period> getPeriodList(){
        return this.unmodList;
    }

    /**
     * 指定した名前で村に登録されているAvatarを返す。
     * @param fullName Avatarの名前
     * @return Avatar
     */
    public Avatar getAvatar(String fullName){
        Avatar avatar;

        avatar = Avatar.getPredefinedAvatar(fullName);
        if( avatar != null ){
            return avatar;
        }

        avatar = this.avatarMap.get(fullName);
        if( avatar != null ){
            return avatar;
        }

        return null;
    }

    /**
     * Avatarを村に登録する。
     * @param avatar Avatar
     */
    // 未知のAvatar出現時の処理が不完全
    public void addAvatar(Avatar avatar){
        if(avatar == null) return;
        String fullName = avatar.getFullName();
        this.avatarMap.put(fullName, avatar);
        return;
    }

    /**
     * Avatarの顔イメージを村に登録する。
     * @param avatar Avatar
     * @param faceImage 顔イメージ
     */
    // デカキャラ対応
    public void putAvatarFaceImage(Avatar avatar, Image faceImage){
        this.faceImageMap.put(avatar, faceImage);
        return;
    }

    /**
     * 村に登録されたAvatarの顔イメージを返す。
     * @param avatar Avatar
     * @return 顔イメージ
     * TODO 失敗したらプロローグを強制読み込みして再トライしたい
     */
    public Image getAvatarFaceImage(Avatar avatar){
        return this.faceImageMap.get(avatar);
    }

    /**
     * 村に登録された墓イメージを返す。
     * @return 墓イメージ
     */
    public Image getGraveImage(){
        return this.tombImage;
    }

    /**
     * 村に墓イメージを登録する。
     * @param image 墓イメージ
     * @throws java.lang.NullPointerException 引数がnullだった場合。
     */
    public void setGraveImage(Image image) throws NullPointerException{
        if(image == null) throw new NullPointerException();
        this.tombImage = image;
    }

    /**
     * 村にアクセスするためのCGIクエリーを返す。
     * @return CGIクエリー
     */
    public CharSequence getCGIQuery(){
        StringBuilder result = new StringBuilder();
        result.append("?vid=").append(getVillageID());
        return result;
    }

    /**
     * 次回更新月を返す。
     * @return 更新月(1-12)
     */
    public int getLimitMonth(){
        return this.limitMonth;
    }

    /**
     * 次回更新日を返す。
     * @return 更新日(1-31)
     */
    public int getLimitDay(){
        return this.limitDay;
    }

    /**
     * 次回更新時を返す。
     * @return 更新時(0-23)
     */
    public int getLimitHour(){
        return this.limitHour;
    }

    /**
     * 次回更新分を返す。
     * @return 更新分(0-59)
     */
    public int getLimitMinute(){
        return this.limitMinute;
    }

    /**
     * 有効な村か否か判定する。
     * @return 無効な村ならfalse
     */
    public boolean isValid(){
        return this.isValid;
    }

    /**
     * 人狼BBSサーバからPeriod一覧情報が含まれたHTMLを取得し、
     * Periodリストを更新する。
     * @throws java.io.IOException ネットワーク入出力の異常
     */
    public void updatePeriodList() throws IOException{
        Land land = getParentLand();
        ServerAccess server = land.getServerAccess();

        HtmlSequence html;
        if(land.getLandDef().getLandState() == LandState.ACTIVE){
            html = server.getHTMLBoneHead(this);
        }else{
            html = server.getHTMLVillage(this);
        }

        parseTitle(html);
        updatePeriodList(html);

        return;
    }

    /**
     * 与えられたHTMLのリンク情報を抽出してPeriodリストを更新する。
     * まだPeriodデータのロードは行われない。
     * ゲーム進行中の村で更新時刻をまたいで更新が行われた場合、
     * 既存のPeriodリストが伸張する場合がある。
     * @param html HTMLデータ
     */
    private void updatePeriodList(CharSequence html){
        boolean hasPrologue = false;
        boolean hasEpilogue = false;
        boolean hasProgress = false;
        boolean hasDone = false;
        int maxProgress = 0;

        Matcher matcher = anchorRegex.matcher(html);
        for(;;){
            if( ! matcher.find() ) break;
//          String hrefValue  = matcher.group(1);
            String anchorText = matcher.group(2);
            String dayString  = matcher.group(3);

            if( anchorText.equals("プロローグ") ){
                hasPrologue = true;
            }else if( anchorText.endsWith("日目") ){
                hasProgress = true;
                int day = Integer.parseInt(dayString);
                maxProgress = day;
            }else if( anchorText.equals("エピローグ") ){
                hasEpilogue = true;
            }else if( anchorText.equals("終了") ){
                hasDone = true;
                break;
            }else{
                continue;
            }
        }

        if(getParentLand().getLandDef().getLandState() != LandState.ACTIVE){
            setState(VillageState.GAMEOVER);
        }else if(hasDone){
            setState(VillageState.GAMEOVER);
        }else if(hasEpilogue){
            setState(VillageState.EPILOGUE);
        }else if(hasProgress){
            setState(VillageState.PROGRESS);
        }else if(hasPrologue){
            setState(VillageState.PROLOGUE);
        }else{
            setState(VillageState.UNKNOWN);
            this.periodList.clear();
            assert false;
            return;
        }

        Period lastPeriod = null;
        if(hasPrologue){
            Period prologue = getPrologue();
            if(prologue == null){
                lastPeriod = new Period(this, PeriodType.PROLOGUE, 0);
                setPeriod(0, lastPeriod);
            }else{
                lastPeriod = prologue;
            }
        }
        if(hasProgress){
            for(int day=1; day<=maxProgress; day++){
                Period progress = getProgress(day);
                if(progress == null){
                    lastPeriod = new Period(this, PeriodType.PROGRESS, day);
                    setPeriod(day, lastPeriod);
                }else{
                    lastPeriod = progress;
                }
            }
        }
        if(hasEpilogue){
            Period epilogue = getEpilogue();
            if(epilogue == null){
                lastPeriod = new Period(this,
                                        PeriodType.EPILOGUE,
                                        maxProgress +1);
                setPeriod(maxProgress +1, lastPeriod);
            }else{
                lastPeriod = epilogue;
            }
        }

        assert getPeriodSize() > 0;
        assert lastPeriod != null;

        // 念のためチョップ。
        // リロードで村が縮むわけないじゃん。みんな大げさだなあ
        while(this.periodList.getLast() != lastPeriod){
            this.periodList.removeLast();
        }

        if(getState() != VillageState.GAMEOVER){
            lastPeriod.setHot(true);
        }

        return;
    }

    /**
     * Periodリストの指定したインデックスにPeriodを上書きする。
     * リストのサイズと同じインデックスを指定する事が許される。
     * その場合の動作はList.addと同じ。
     * @param index Periodリストのインデックス。
     * @param period 上書きするPeriod
     * @throws java.lang.IndexOutOfBoundsException インデックスの指定がおかしい
     */
    private void setPeriod(int index, Period period)
            throws IndexOutOfBoundsException{
        int listSize = this.periodList.size();
        if(index == listSize){
            this.periodList.add(period);
        }else if(index < listSize){
            this.periodList.set(index, period);
        }else{
            throw new IndexOutOfBoundsException();
        }
        return;
    }

    /**
     * 村タイトルと更新時刻のパースを行う。
     * 現時点では何も反映されない。
     * @param html HTMLデータ。
     */
    private void parseTitle(CharSequence html){
        Matcher matcher;

        matcher = titleRegex.matcher(html);
        if( matcher.find() ){
            String title = matcher.group(1);
        }

        matcher = villageInfoRegex.matcher(html);
        if( matcher.find() ){
            String vName = matcher.group(1);
            String month = matcher.group(2);
            String day   = matcher.group(3);
            String ampm  = matcher.group(4);
            String hh    = matcher.group(5);
            String mm    = matcher.group(6);

            this.limitMonth  = Integer.parseInt(month);
            this.limitDay    = Integer.parseInt(day);
            this.limitHour   = Integer.parseInt(hh);
            this.limitMinute = Integer.parseInt(mm);
            // TODO 「午後12時」は存在するか？たぶんない。
            if(ampm != null){
                if( ampm.equals("午後") ) this.limitHour += 12;
                this.limitHour = this.limitHour % 24;     // 24時 = 0時
            }
        }

        return;
    }

    /**
     * アンカーに一致する会話(Talk)のリストを取得する。
     * @param anchor アンカー
     * @return Talkのリスト
     * @throws java.io.IOException おそらくネットワークエラー
     */
    public List<Talk> getTalkListFromAnchor(Anchor anchor)
            throws IOException{
        List<Talk> result = new LinkedList<Talk>();

        Period anchorPeriod = getPeriod(anchor);
        if(anchorPeriod == null) return result;

        anchorPeriod.updatePeriod();

        for(Topic topic : anchorPeriod.getTopicList()){
            if( ! (topic instanceof Talk) ) continue;
            Talk talk = (Talk) topic;
            if(talk.getHour()   != anchor.getHour()  ) continue;
            if(talk.getMinute() != anchor.getMinute()) continue;
            result.add(talk);
        }
        return result;
    }

    /**
     * 全Periodの発言データをアンロードする。
     */
    public void unloadPeriods(){
        for(Period period : this.periodList){
            period.unload();
        }
        return;
    }

    /**
     * {@inheritDoc}
     * 二つの村を順序付ける。
     * @param village {@inheritDoc}
     * @return {@inheritDoc}
     */
    public int compareTo(Village village){
        int cmpResult = vComparator.compare(this, village);
        return cmpResult;
    }

    /**
     * {@inheritDoc}
     * 二つの村が等しいか調べる。
     * @param obj {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj){
        if(obj == null) return false;
        if( ! (obj instanceof Village) ) return false;
        Village village = (Village) obj;

        if( getParentLand() != village.getParentLand() ) return false;

        int cmpResult = compareTo(village);
        if(cmpResult == 0) return true;
        return false;
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public int hashCode(){
        int homeHash = getParentLand().hashCode();
        int vidHash = getVillageID().hashCode();
        int result = homeHash ^ vidHash;
        return result;
    }

    /**
     * {@inheritDoc}
     * 村の文字列表現を返す。
     * 村の名前と等しい。
     * @return 村の名前
     */
    @Override
    public String toString(){
        return getVillageFullName();
    }
}
