/*
 * Inter Process Communication with Web browser
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: WebIPC.java 371 2009-01-24 15:05:08Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.GraphicsEnvironment;
import java.awt.HeadlessException;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URI;

/**
 * Webブラウザとのプロセス間通信。
 * java.awt.Desktopの代用品。
 * JRE1.6でしか使えない。がしかし、JRE1.5でもコンパイル＆ロード可能。
 * @see java.awt.Desktop
 */
public class WebIPC{

    private static final Class  desktopKlass;
    private static final Method isDesktopSupportedMethod;
    private static final Method getDesktopMethod;
    private static final Method isSupportedMethod;
    private static final Method browseMethod;

    private static final Class desktopActionKlass;
    private static final Enum  browserEnum;

    static{
        desktopKlass       = getClass("java.awt.Desktop");
        desktopActionKlass = getInnerClass(desktopKlass,
                                           "java.awt.Desktop.Action");

        isDesktopSupportedMethod = getMethod("isDesktopSupported");
        getDesktopMethod         = getMethod("getDesktop");
        isSupportedMethod        = getMethod("isSupported",
                                             desktopActionKlass);
        browseMethod             = getMethod("browse",
                                             URI.class);

        browserEnum = getEnumMember(desktopActionKlass, "BROWSE");
    }

    /**
     * クラス名からClassインスタンスを探す。
     * @param klassName クラス名
     * @return Classインスタンス。クラスが見つからなければnull。
     */
    private static Class getClass(String klassName){
        Class result;
        try{
            result = Class.forName(klassName);
        }catch(ClassNotFoundException e){
            result = null;
        }
        return result;
    }

    /**
     * 内部クラス名からClassインスタンスを探す。
     * @param parent 囲む親クラス。
     * @param canonical 内部クラスのカノニカル名
     * @return Classインスタンス。クラスが見つからなければnull。
     */
    private static Class getInnerClass(Class parent, String canonical){
        if(parent == null) return null;

        Class result = null;

        Class[] innerKlasses = parent.getClasses();
        for(Class klass : innerKlasses){
            if(klass.getCanonicalName().equals(canonical)){
                result = klass;
                break;
            }
        }

        return result;
    }

    /**
     * Desktopクラスのメソッド名からMethodインスタンスを探す。
     * @param methodName メソッド名
     * @param paramTypes 引数型並び
     * @return Methodインスタンス。見つからなければnull。
     */
    @SuppressWarnings("unchecked")
    private static Method getMethod(String methodName,
                                     Class<?>... paramTypes){
        if(desktopKlass == null) return null;

        Method result;

        try{
            result = desktopKlass.getMethod(methodName, paramTypes);
        }catch(NoSuchMethodException e){
            result = null;
        }

        return result;
    }

    /**
     * Enumのメンバを探す。
     * @param parent Enumの型
     * @param memberName メンバ名
     * @return Enumインスタンス。見つからなければnull。
     */
    private static Enum getEnumMember(Class parent,
                                        String memberName){
        if(parent == null) return null;

        Field field;
        try{
            field = parent.getField(memberName);
        }catch(NoSuchFieldException e){
            return null;
        }

        Object value;
        try{
            value = field.get(null);
        }catch(IllegalAccessException e){
            return null;
        }catch(IllegalArgumentException e){
            return null;
        }

        if( ! (value instanceof Enum) ) return null;

        return (Enum) value;
    }

    /**
     * @see java.awt.Desktop#isDesktopSupported()
     * @return Desktopが利用可能ならtrue。JRE1.5以前だとたぶんfalse。
     */
    public static boolean isDesktopSupported(){
        if(isDesktopSupportedMethod == null) return false;

        Object invokeResult;
        try{
            invokeResult = isDesktopSupportedMethod.invoke(null,
                                                           (Object[]) null);
        }catch(InvocationTargetException e){
            Throwable targetException = e.getTargetException();
            if(targetException instanceof RuntimeException){
                throw (RuntimeException) targetException;
            }else if(targetException instanceof Error){
                throw (Error) targetException;
            }
            return false;
        }catch(IllegalAccessException e){
            assert false;
            return false;
        }catch(IllegalArgumentException e){
            assert false;
            return false;
        }catch(NullPointerException e){
            assert false;
            return false;
        }

        if( ! (invokeResult instanceof Boolean) ){
            assert false;
            return false;
        }

        boolean result = ((Boolean)invokeResult).booleanValue();

        return result;
    }

    /**
     * @see java.awt.Desktop#getDesktop()
     * @return インスタンス
     * @throws java.awt.HeadlessException
     * @throws java.lang.UnsupportedOperationException
     */
    public static WebIPC getWebIPC()
            throws HeadlessException,
                   UnsupportedOperationException {
        if(GraphicsEnvironment.isHeadless()) throw new HeadlessException();
        if( ! isDesktopSupported() ) throw new UnsupportedOperationException();

        WebIPC webIPC;
        try{
            webIPC = new WebIPC();
        }catch(HeadlessException e){
            throw e;
        }catch(UnsupportedOperationException e){
            throw e;
        }catch(Error e){
            throw e;
        }catch(Throwable e){
            throw new UnsupportedOperationException(e);
        }

        return webIPC;
    }

    private final Object desktop;

    /**
     * 見えないコンストラクタ
     * @throws java.lang.Throwable 各種異常系
     */
    private WebIPC() throws Throwable{
        super();

        try{
            this.desktop = getDesktopMethod.invoke(null, (Object[]) null);
        }catch(InvocationTargetException e){
            Throwable targetException = e.getTargetException();
            throw targetException;
        }catch(IllegalAccessException e){
            assert false;
            throw e;
        }catch(IllegalArgumentException e){
            assert false;
            throw e;
        }catch(NullPointerException e){
            assert false;
            throw e;
        }

        return;
    }

    /**
     * @see java.awt.Desktop#browse(java.net.URI)
     * @param uri
     * @throws java.lang.NullPointerException
     * @throws java.lang.UnsupportedOperationException
     * @throws java.io.IOException
     * @throws java.lang.SecurityException
     * @throws java.lang.IllegalArgumentException
     */
    public void browse(URI uri)
            throws NullPointerException,
                   UnsupportedOperationException,
                   IOException,
                   SecurityException,
                   IllegalArgumentException {
        if(uri == null) throw new NullPointerException();
        if( ! isSupported(Action.BROWSE) ){
            throw new UnsupportedOperationException();
        }

        try{
            browseMethod.invoke(this.desktop, uri);
        }catch(InvocationTargetException e){
            Throwable targetException = e.getTargetException();
            if(targetException instanceof IOException){
                throw (IOException) targetException;
            }else if(targetException instanceof SecurityException){
                throw (SecurityException) targetException;
            }else if(targetException instanceof IllegalArgumentException){
                throw (IllegalArgumentException) targetException;
            }else if(targetException instanceof RuntimeException){
                throw (RuntimeException) targetException;
            }else if(targetException instanceof Error){
                throw (Error) targetException;
            }
        }catch(IllegalAccessException e){
            assert false;
            throw new UnsupportedOperationException();
        }catch(IllegalArgumentException e){
            assert false;
            throw new UnsupportedOperationException();
        }catch(NullPointerException e){
            assert false;
            throw new UnsupportedOperationException();
        }

        return;
    }

    /**
     * @see java.awt.Desktop#isSupported(java.awt.Desktop.Action)
     * @param action
     * @return アクションがサポートされていればtrue。
     */
    public boolean isSupported(Action action){
        if(Action.BROWSE.equals(action)){
            Object invokeResult = null;
            try{
                invokeResult = isSupportedMethod.invoke(this.desktop,
                                                        browserEnum);
            }catch(InvocationTargetException e){
                Throwable targetException = e.getTargetException();
                if(targetException instanceof RuntimeException){
                    throw (RuntimeException) targetException;
                }else if(targetException instanceof Error){
                    throw (Error) targetException;
                }
                return false;
            }catch(IllegalAccessException e){
                assert false;
                return false;
            }catch(IllegalArgumentException e){
                assert false;
                return false;
            }catch(NullPointerException e){
                assert false;
                return false;
            }catch(RuntimeException e){
                throw e;
            }catch(Error e){
                throw e;
            }catch(Throwable e){
                return false;
            }

            if( ! (invokeResult instanceof Boolean) ){
                assert false;
                return false;
            }

            boolean result = ((Boolean)invokeResult).booleanValue();

            return result;
        }

        return false;
    }

    /**
     * 各種デスクトップアクション
     * @see java.awt.Desktop.Action
     */
    public static enum Action{
        BROWSE,
    //  EDIT,
    //  MAIL,
    //  OPEN,
    //  PRINT,
    }
}
