/*
 * land information definition
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: LandDef.java 509 2009-04-29 10:51:45Z olyutorskii $
 */

package jp.sourceforge.jindolf.core;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.validation.Schema;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * 人狼BBSの各国設定
 */
public class LandDef{
    private static final String SCHEMA_LANDDEF =
            "http://jindolf.sourceforge.jp/xml/xsd/coreXML.xsd";

    private static final String RES_LANDDEF =
            "resources/xml/landDefList.xml";

    private static final List<LandDef> unmodList;

    private static final Pattern iso8601Regex;

    public static final Date DATE_UNKNOWN = new Date(-1L);

    static{
        String year = "([0-9][0-9][0-9][0-9])";
        String month = "([0-1][0-9])";
        String day = "([0-3][0-9])";
        String hour = "([0-2][0-9])";
        String minute = "([0-5][0-9])";
        String second = "([0-6][0-9])";
        String timezone =
                "("+
                    "[\\+\\-][0-2][0-9]"+
                    "(?:"+ ":?[0-5][0-9]" +")?"+
                "|"+
                    "Z"+
                ")";
        String iso8601 =
                year +"\\-"+ month +"\\-"+ day
                +"T"+
                hour +":"+ minute +":"+ second
                +timezone;

        iso8601Regex = Pattern.compile(iso8601);

        DocumentBuilder builder;
        try{
            Schema xsdSchema = SchemaUtilities.createSchema(SCHEMA_LANDDEF);
            builder = SchemaUtilities.createBuilder(xsdSchema);
        }catch(RuntimeException e){
            throw e;
        }catch(Exception e){
            throw new ExceptionInInitializerError(e);
        }

        List<LandDef> landDefList;
        try{
            Element list = loadLandInfoList(builder);
            landDefList = registLandInfoList(list);
        }catch(RuntimeException e){
            throw e;
        }catch(Exception e){
            throw new ExceptionInInitializerError(e);
        }

        unmodList = Collections.unmodifiableList(landDefList);

    }

    /**
     * 国設定に関する定義をロードする。
     * @param builder DOMビルダ
     * @return ルート要素
     * @throws javax.xml.parsers.ParserConfigurationException
     * @throws java.io.IOException
     * @throws org.xml.sax.SAXException
     */
    public static Element loadLandInfoList(DocumentBuilder builder)
            throws ParserConfigurationException,
                   IOException,
                   SAXException {
        InputStream is = LandDef.class.getResourceAsStream(RES_LANDDEF);

        Document document = builder.parse(is);

        Element root = document.getDocumentElement();
        String tagName = root.getTagName();
        if( ! tagName.equals("landDefList") ){
            throw new SAXException("illegal root " + tagName);
        }

        return root;
    }

    /**
     * 要素内部を探索し、国設定を登録する。
     * @param list ルート要素
     * @return 国設定が登録されたList
     * @throws org.xml.sax.SAXException
     */
    private static List<LandDef> registLandInfoList(Element list)
            throws SAXException {
        NodeList elems = list.getElementsByTagName("landDef");
        int landNum = elems.getLength();
        if(landNum <= 0){
            throw new SAXException("there is no <landDef>");
        }
        List<LandDef> landDefList = new ArrayList<LandDef>(landNum);

        for(int index=0; index < landNum; index++){
            Node node = elems.item(index);
            Element elem = (Element) node;
            LandDef landDef = buildLandDef(elem);
            landDefList.add(landDef);
        }

        return landDefList;
    }

    /**
     * 個々の国設定をオブジェクトに変換する。
     * @param 国設定要素
     * @return 国設定オブジェクト
     * @throws org.xml.sax.SAXException
     */
    private static LandDef buildLandDef(Element landDef)
            throws SAXException{
        String attr;

        String landName = landDef.getAttributeNode("landName").getValue();
        String landId = landDef.getAttributeNode("landId").getValue();
        String formalName = landDef.getAttributeNode("formalName").getValue();
        String landPrefix = landDef.getAttributeNode("landPrefix").getValue();

        LandState landState;
        attr = landDef.getAttributeNode("landState").getValue();
        if     (attr.equals("closed")    ) landState = LandState.CLOSED;
        else if(attr.equals("historical")) landState = LandState.HISTORICAL;
        else if(attr.equals("active")    ) landState = LandState.ACTIVE;
        else throw new SAXException("illegal land status");

        attr = landDef.getAttributeNode("minMembers").getValue();
        int minMembers = Integer.parseInt(attr);
        attr = landDef.getAttributeNode("maxMembers").getValue();
        int maxMembers = Integer.parseInt(attr);

        URI webURI;
        URI cgiURI;
        URI tombFaceIconURI;
        URI tombBodyIconURI;
        try{
            attr = landDef.getAttributeNode("webURI").getValue();
            webURI = new URI(attr);
            attr = landDef.getAttributeNode("cgiURI").getValue();
            cgiURI = new URI(attr);
            attr = landDef.getAttributeNode("tombFaceIconURI").getValue();
            tombFaceIconURI = new URI(attr);
            attr = landDef.getAttributeNode("tombBodyIconURI").getValue();
            tombBodyIconURI = new URI(attr);
        }catch(URISyntaxException e){
            throw new SAXException("illegal URI", e);
        }

        Attr attrNode;
        String faceURITemplate = null;
        String bodyURITemplate = null;
        attrNode = landDef.getAttributeNode("faceIconURITemplate");
        if(attrNode != null){
            faceURITemplate = attrNode.getValue();
        }
        attrNode = landDef.getAttributeNode("bodyIconURITemplate");
        if(attrNode != null){
            bodyURITemplate = attrNode.getValue();
        }

        String lang    = "";
        String country = "";
        String variant = "";
        attr = landDef.getAttributeNode("locale").getValue();
        String[] lcstr = attr.split("-", 3);
        if(lcstr.length >= 1) lang    = lcstr[0];
        if(lcstr.length >= 2) country = lcstr[1];
        if(lcstr.length >= 3) variant = lcstr[2];
        Locale locale = new Locale(lang, country, variant);

        attr = landDef.getAttributeNode("encoding").getValue();
        Charset encoding = Charset.forName(attr);
        attr = landDef.getAttributeNode("timeZone").getValue();
        TimeZone timeZone = TimeZone.getTimeZone(attr);

        String dateTime;
        Calendar cal;
        dateTime = landDef.getAttributeNode("startDate").getValue();
        cal = parseISO8601(dateTime);
        Date startDate = cal.getTime();
        dateTime = landDef.getAttribute("endDate");
        Date endDate = null;
        if(dateTime != null && dateTime.length() > 0){
            cal = parseISO8601(dateTime);
            endDate = cal.getTime();
        }

        String description = landDef.getAttributeNode("description").getValue();
        String contactInfo = landDef.getAttributeNode("contactInfo").getValue();

        String invalidVid = landDef.getAttribute("invalidVid");
        SortedSet<Integer> invalidSet = parseIntList(invalidVid);

        LandDef landDefInfo;
        try{
            landDefInfo = new LandDef(landName,
                                    landId,
                                    formalName,
                                    landPrefix,
                                    landState,
                                    minMembers,
                                    maxMembers,
                                    webURI,
                                    cgiURI,
                                    tombFaceIconURI,
                                    tombBodyIconURI,
                                    faceURITemplate,
                                    bodyURITemplate,
                                    locale,
                                    encoding,
                                    timeZone,
                                    startDate,
                                    endDate,
                                    description,
                                    contactInfo,
                                    invalidSet );
        }catch(RuntimeException e){
            throw e;
        }catch(Exception e){
            throw new SAXException(e);
        }

        return landDefInfo;
    }

    /**
     * コンマとハイフンで区切られた整数の羅列をパースする。
     * 「10,23-25」なら10,23,24,25を結果に返す。
     * @param seq パース対象文字列
     * @return ソートされたIntegerのSet
     * @throws java.lang.NullPointerException 入力がnull
     */
    private static SortedSet<Integer> parseIntList(CharSequence seq)
            throws NullPointerException{
        final String SP = "[\u0020\\t]*";
        final Pattern delimit = Pattern.compile("[,\u0020\\t]+");
        final Pattern range   = Pattern.compile(SP + "\\-" + SP);

        if(seq == null) throw new NullPointerException();

        SortedSet<Integer> result = new TreeSet<Integer>();
        SortedSet<Integer> unmod  = Collections.unmodifiableSortedSet(result);

        if(seq.length() <= 0 ) return unmod;

        String[] tokens = delimit.split(seq);
        for(String token : tokens){
            if(token == null || token.length() <= 0) continue;
            String[] ivalues = range.split(token);
            if(ivalues.length >= 3){
                throw new IllegalArgumentException(token);
            }

            int ivalStart;
            int ivalEnd;
            try{
                ivalStart = Integer.parseInt(ivalues[0]);
                if(ivalues.length >= 2) ivalEnd = Integer.parseInt(ivalues[1]);
                else                    ivalEnd = ivalStart;
            }catch(NumberFormatException e){
                throw new IllegalArgumentException(token, e);
            }

            if(ivalStart > ivalEnd){
                int dummy = ivalStart;
                ivalStart = ivalEnd;
                ivalEnd = dummy;
                assert ivalStart <= ivalEnd;
            }

            for(int ival = ivalStart; ival <= ivalEnd; ival++){
                result.add(ival);
            }
        }

        return unmod;
    }

    /**
     * ISO8601形式の日付をDateに変換する。
     * JRE1.6のjavax.xml.bind.DatatypeConverter代替品
     * @param date ISO8601形式の日付文字列
     * @return Dateインスタンス
     * @throws java.lang.IllegalArgumentException 形式が変な場合。
     */
    private static Calendar parseISO8601(CharSequence date)
            throws IllegalArgumentException{
        Matcher matcher = iso8601Regex.matcher(date);
        if( ! matcher.lookingAt() ){
            throw new IllegalArgumentException(date.toString());
        }

        String yearStr   = matcher.group(1);
        String monthStr  = matcher.group(2);
        String dayStr    = matcher.group(3);
        String hourStr   = matcher.group(4);
        String minuteStr = matcher.group(5);
        String secondStr = matcher.group(6);
        String tzString  = matcher.group(7);

        int year;
        int month;
        int day;
        int hour;
        int minute;
        int second;
        try{
            year   = Integer.parseInt(yearStr);
            month  = Integer.parseInt(monthStr);
            day    = Integer.parseInt(dayStr);
            hour   = Integer.parseInt(hourStr);
            minute = Integer.parseInt(minuteStr);
            second = Integer.parseInt(secondStr);
        }catch(NumberFormatException e){
            throw new IllegalArgumentException(date.toString(), e);
        }

        String tzID = "GMT";
        if( tzString.compareToIgnoreCase("Z") == 0 ) tzID += "+00:00";
        else                                         tzID += tzString;
        TimeZone timezone = TimeZone.getTimeZone(tzID);

        GregorianCalendar calendar = new GregorianCalendar();
        calendar.clear();
        calendar.setTimeZone(timezone);
        calendar.set(year, month-1, day, hour, minute, second);

        return calendar;
    }

    /**
     * 引数がnullなら例外を投げる。
     * @param obj 任意のオブジェクト
     * @throws java.lang.NullPointerException 引数がnullだった。
     */
    private static void checkNull(Object obj) throws NullPointerException{
        if(obj == null) throw new NullPointerException();
        return;
    }

    /**
     * 文字列引数が1文字以上なければ例外を投げる。
     * @param seq 文字列
     * @throws java.lang.IllegalArgumentException 引数がnullか0文字だった。
     */
    private static void checkEmpty(CharSequence seq)
            throws IllegalArgumentException{
        checkNull(seq);
        if(seq.length() <= 0) throw new IllegalArgumentException();
    }

    /**
     * 国設定のListを返す。
     * @return List
     */
    public static List<LandDef> getLandDefList(){
        return unmodList;
    }

    private final String landName;
    private final String landId;
    private final String formalName;
    private final String landPrefix;
    private final LandState landState;
    private final int minMembers;
    private final int maxMembers;
    private final URI webURI;
    private final URI cgiURI;
    private final URI tombFaceIconURI;
    private final URI tombBodyIconURI;
    private final String faceURITemplate;
    private final String bodyURITemplate;
    private final Locale locale;
    private final Charset encoding;
    private final TimeZone timeZone;
    private final Date startDate;
    private final Date endDate;
    private final String description;
    private final String contactInfo;
    private final SortedSet<Integer> invalidSet;

    /**
     * コンストラクタ
     * @param landName 国名
     * @param landId 国識別子
     * @param formalName 正式名称
     * @param landPrefix 各村名の前置文字
     * @param landState 国の状態
     * @param minMembers 最少定員
     * @param maxMembers 最大定員
     * @param webURI Webブラウザ用エントリURI
     * @param cgiURI CGIのURI
     * @param tombFaceIconURI 墓アイコンURI
     * @param tombBodyIconURI 大きな墓アイコンURI
     * @param faceURITemplate Avatarの顔画像URIのテンプレート。MessageFormat
     * @param bodyURITemplate Avatarの全身像URIのテンプレート。MessageFormat
     * @param locale 国のロケール
     * @param encoding この国で使われるエンコーディング
     * @param timeZone この国の時刻表記のタイムゾーン
     * @param startDate この国が始まった日
     * @param endDate この国が参加者募集を打ち切った日
     * @param description 国の説明
     * @param contactInfo 連絡先
     * @param invalidSet 無効村の集合
     */
    private LandDef(String landName,
                      String landId,
                      String formalName,
                      String landPrefix,
                      LandState landState,
                      int minMembers,
                      int maxMembers,
                      URI webURI,
                      URI cgiURI,
                      URI tombFaceIconURI,
                      URI tombBodyIconURI,
                      String faceURITemplate,
                      String bodyURITemplate,
                      Locale locale,
                      Charset encoding,
                      TimeZone timeZone,
                      Date startDate,
                      Date endDate,
                      String description,
                      String contactInfo,
                      SortedSet<Integer> invalidSet
                      ){
        super();

        checkEmpty(landName);
        checkEmpty(landId);
        checkEmpty(formalName);
        checkNull(landPrefix);
        checkNull(landState);
        if(minMembers <= 0) throw new IllegalArgumentException();
        if(maxMembers < minMembers) throw new IllegalArgumentException();
        checkNull(webURI);
        checkNull(cgiURI);
        checkNull(tombFaceIconURI);
        checkNull(tombBodyIconURI);
        checkNull(locale);
        checkNull(encoding);
        checkNull(timeZone);
        checkNull(startDate);
        if(endDate == null) endDate = DATE_UNKNOWN;
        checkEmpty(description);
        checkEmpty(contactInfo);
        checkNull(invalidSet);

        if( ! webURI.isAbsolute() ) throw new IllegalArgumentException();
        if( ! cgiURI.isAbsolute() ) throw new IllegalArgumentException();

        this.landName = landName;
        this.landId = landId;
        this.formalName = formalName;
        this.landPrefix = landPrefix;
        this.landState = landState;
        this.minMembers = minMembers;
        this.maxMembers = maxMembers;
        this.webURI = webURI;
        this.cgiURI = cgiURI;
        this.tombFaceIconURI = tombFaceIconURI;
        this.tombBodyIconURI = tombBodyIconURI;
        this.faceURITemplate = faceURITemplate;
        this.bodyURITemplate = bodyURITemplate;
        this.locale = locale;
        this.encoding = encoding;
        this.timeZone = timeZone;
        this.startDate = startDate;
        this.endDate = endDate;
        this.description = description;
        this.contactInfo = contactInfo;
        this.invalidSet = invalidSet;

        return;
    }

    /**
     * 国名を得る。
     * @return 国名
     */
    public String getLandName(){
        return this.landName;
    }

    /**
     * 国識別子を得る。
     * @return 識別子
     */
    public String getLandId(){
        return this.landId;
    }

    /**
     * 正式名称を得る。
     * @return 正式名称
     */
    public String getFormalName(){
        return this.formalName;
    }

    /**
     * 各村の前置文字。
     * F国なら「F」
     * @return 全治文字
     */
    public String getLandPrefix(){
        return this.landPrefix;
    }

    /**
     * 国の状態を得る。
     * @return 状態
     */
    public LandState getLandState(){
        return this.landState;
    }

    /**
     * 最小定員を得る
     * @return 最小定員
     */
    public int getMinMembers(){
        return this.minMembers;
    }

    /**
     * 最大定員を得る。
     * @return 最大定員
     */
    public int getMaxMembers(){
        return this.maxMembers;
    }

    /**
     * Webアクセス用の入り口URIを得る
     * @return 入り口URI
     */
    public URI getWebURI(){
        return this.webURI;
    }

    /**
     * クエリーを投げるCGIのURIを得る。
     * @return CGIのURI
     */
    public URI getCgiURI(){
        return this.cgiURI;
    }

    /**
     * 墓画像のURIを得る。
     * @return 墓URI
     */
    public URI getTombFaceIconURI(){
        return this.tombFaceIconURI;
    }

    /**
     * 大きな墓画像のURIを得る。
     * @return 墓URI
     */
    public URI getTombBodyIconURI(){
        return this.tombBodyIconURI;
    }

    /**
     * この国のロケールを得る。
     * @return ロケール
     */
    public Locale getLocale(){
        return this.locale;
    }

    /**
     * この国が使うエンコーディングを得る。
     * @return エンコーディング
     */
    public Charset getEncoding(){
        return this.encoding;
    }

    /**
     * この国の時刻表記で使うタイムゾーンを得る
     * @return タイムゾーン
     */
    public TimeZone getTimeZone(){
        TimeZone result = (TimeZone)( this.timeZone.clone() );
        return result;
    }

    /**
     * この国の始まった日を得る。
     * @return 始まった日
     */
    public Date getStartDate(){
        Date result = (Date)( this.startDate.clone() );
        return result;
    }

    /**
     * この国が参加者募集を打ち切った日を得る。
     * @return 終わった日。
     */
    public Date getEndDate(){
        Date result = (Date)( this.endDate.clone() );
        return result;
    }

    /**
     * この国の説明を得る。
     * @return 説明文字列
     */
    public String getDescription(){
        return this.description;
    }

    /**
     * この国の連絡先を得る。
     * @return 連絡先文字列
     */
    public String getContactInfo(){
        return this.contactInfo;
    }

    /**
     * 有効な村IDか否か判定する。
     * @param id 村ID
     * @return 無効な村ならfalse
     */
    public boolean isValidVillageId(int id){
        if(this.invalidSet.contains(id)) return false;
        return true;
    }

}
