/*
 * xml resource resolver
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: XmlResourceResolver.java 510 2009-05-01 12:47:30Z olyutorskii $
 */

package jp.sourceforge.jindolf.core;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * URL変換マップに従い、XML文書からの外部参照をリダイレクトする。
 * 相対URIはこのクラスをベースに解決される。
 * 主な用途は外部スキーマのリソース化など。
 */
public class XmlResourceResolver
        implements LSResourceResolver,
                   EntityResolver{

    protected static final Map<URL, URL> defaultMap;
    protected static final URI emptyURI = URI.create("");

    static{
        String[][] uriTxArray = {
            { "http://www.w3.org/2001/xml.xsd",
              "resources/xsd/ext/xml-2009-01.xsd" },
            { "http://jindolf.sourceforge.jp/xml/xsd/coreType.xsd",
              "resources/xsd/coreType.xsd" },
            { "http://jindolf.sourceforge.jp/xml/xsd/coreXML.xsd",
              "resources/xsd/coreXML.xsd" },
//          { "http://jindolf.sourceforge.jp/xml/xsd/bbsArchive.xsd",
//            "resources/xsd/bbsArchive.xsd" },
        };

        Map<URL, URL> map = new HashMap<URL, URL>();
        for(String[] uriPair : uriTxArray){
            String txFrom   = uriPair[0];
            String txTo     = uriPair[1];

            URL originalURL;
            URL resourceURL;
            try{
                originalURL = new URI(txFrom).normalize().toURL();
            }catch(MalformedURLException e){
                continue;
            }catch(URISyntaxException e){
                continue;
            }
            resourceURL = XmlResourceResolver.class.getResource(txTo);

            map.put(originalURL, resourceURL);
        }
        defaultMap = Collections.unmodifiableMap(map);

        DOMImplementationRegistry registry;
        try{
            registry = DOMImplementationRegistry.newInstance();
        }catch(ClassNotFoundException e){
            throw new ExceptionInInitializerError(e);
        }catch(InstantiationException e){
            throw new ExceptionInInitializerError(e);
        }catch(IllegalAccessException e){
            throw new ExceptionInInitializerError(e);
        }
    }

    /**
     * 絶対URIと相対URIを合成したURLを返す。
     * 正規化も行われる。
     * @param base 絶対URIでなければならない。nullでもよい。
     * @param relative 絶対URIでもよいがその場合baseは無視される。null可。
     * @return 合成結果のURLオブジェクト。必ず絶対URIになる。
     * @throws java.net.URISyntaxException URIとして変。
     * @throws MalformedURLException URLとして変。
     * @throws java.lang.IllegalArgumentException 絶対URIが生成できない。
     */
    protected static URL buildBaseRelativeURL(String base, String relative)
            throws URISyntaxException,
                   MalformedURLException,
                   IllegalArgumentException {
        URI baseURI = null;
        if(base != null){
            baseURI = new URI(base);
            if( ! baseURI.isAbsolute() ) throw new IllegalArgumentException();
        }

        URI relativeURI = emptyURI;
        if(relative != null){
            relativeURI = new URI(relative);
        }

        URI resultURI;
        if(baseURI == null || relativeURI.isAbsolute()){
            resultURI = relativeURI;
        }else{
            resultURI = baseURI.resolve(relativeURI);
        }

        if( ! resultURI.isAbsolute() ) throw new IllegalArgumentException();

        URL resultURL = resultURI.normalize().toURL();

        return resultURL;
    }

    /**
     * デフォルトのURL変換マップを返す。
     * キーが変換前のURL、値は変換後のURL。
     * @return 変換マップ
     */
    public static Map<URL, URL> getDefaultTxMap(){
        return defaultMap;
    }

    /**
     * LSInput実装を生成する。
     * @return LSInput実装
     */
    public static LSInput createLSInput(){
        LSInput input = new LSInputImpl();
        return input;
    }

    private Map<URL, URL> uriMap;

    /**
     * コンストラクタ
     */
    public XmlResourceResolver(){
        super();
        this.uriMap = defaultMap;
        return;
    }

    /**
     * URL変換マップを返す。
     * キーが変換前のURL、値は変換後のURL。
     * @return 変換マップ
     */
    public Map<URL, URL> getTxMap(){
        return this.uriMap;
    }

    /**
     * URL変換マップを設定する。
     * キーが変換前のURL、値は変換後のURL。
     * @param map 新しい変換マップ。
     * @return 置き換わった古い変換マップ。
     */
    public Map<URL, URL> setTxMap(Map<URL, URL> map){
        if(map == null) throw new NullPointerException();

        Map<URL, URL> old = this.uriMap;
        this.uriMap = map;

        return old;
    }

    /**
     * 変換後のリソースの入力ストリームを得る。
     * @param originalURIStr オリジナルURI
     * @return 入力ストリーム
     * @throws java.net.URISyntaxException 不正なURI
     * @throws java.io.IOException 入出力エラー
     */
    public InputStream getXMLResourceAsStream(String originalURIStr)
            throws URISyntaxException, IOException{
        if(originalURIStr == null) throw new NullPointerException();

        URI uri = new URI(originalURIStr);
        InputStream is = getXMLResourceAsStream(uri);

        return is;
    }

    /**
     * 変換後のリソースの入力ストリームを得る。
     * @param originalURI オリジナルURI
     * @return 入力ストリーム
     * @throws java.io.IOException 入出力エラー
     */
    public InputStream getXMLResourceAsStream(URI originalURI)
            throws IOException{
        if(originalURI == null) throw new NullPointerException();

        URL url = originalURI.normalize().toURL();
        InputStream is = getXMLResourceAsStream(url);

        return is;
    }

    /**
     * 変換後のリソースの入力ストリームを得る。
     * @param originalURL オリジナルURL
     * @return 入力ストリーム
     * @throws java.io.IOException 入出力エラー
     */
    protected InputStream getXMLResourceAsStream(URL originalURL)
            throws IOException{
        if(originalURL == null) throw new NullPointerException();

        URL resourceURL = this.uriMap.get(originalURL);
        InputStream is = resourceURL.openStream();

        return is;
    }

    /**
     * {@inheritDoc}
     * URL変換したあとの入力ソースを返す。
     * @param type {@inheritDoc}
     * @param namespaceURI {@inheritDoc}
     * @param publicId {@inheritDoc}
     * @param systemId {@inheritDoc}
     * @param baseURI {@inheritDoc}
     * @return {@inheritDoc}
     */
    public LSInput resolveResource(String type,
                                     String namespaceURI,
                                     String publicId,
                                     String systemId,
                                     String baseURI ){
        if(systemId == null) return null;

        URL originalURL;
        try{
            originalURL = buildBaseRelativeURL(baseURI, systemId);
        }catch(URISyntaxException e){
            return null;
        }catch(MalformedURLException e){
            return null;
        }

        InputStream is;
        try{
            is = getXMLResourceAsStream(originalURL);
        }catch(IOException e){
            return null;
        }

        LSInput input = createLSInput();
        input.setBaseURI(baseURI);
        input.setPublicId(publicId);
        input.setSystemId(systemId);
        input.setByteStream(is);

        return input;
    }

    /**
     * {@inheritDoc}
     * URL変換したあとの入力ソースを返す。
     * @param publicId {@inheritDoc}
     * @param systemId {@inheritDoc}
     * @return {@inheritDoc}
     * @throws org.xml.sax.SAXException {@inheritDoc}
     * @throws java.io.IOException {@inheritDoc}
     */
    public InputSource resolveEntity(String publicId, String systemId)
            throws SAXException, IOException{
        if(systemId == null) return null;

        URI originalUri;
        try{
            originalUri = new URI(systemId);
        }catch(URISyntaxException e){
            return null;
        }

        URL originalURL = originalUri.normalize().toURL();
        InputStream is = getXMLResourceAsStream(originalURL);

        InputSource source = new InputSource(is);
        source.setPublicId(publicId);
        source.setSystemId(systemId);

        return source;
    }

    /**
     * JRE1.5用LSInput実装。
     * JRE1.6なら
     * org.w3c.dom.ls.DOMImplementationLS#createLSInput()
     * で生成可能かも。
     */
    public static class LSInputImpl implements LSInput{

        private String baseURI = null;
        private InputStream byteStream = null;
        private boolean certifiedText = false;
        private Reader characterStream = null;
        private String encoding = null;
        private String publicId = null;
        private String stringData = null;
        private String systemId = null;

        /**
         * コンストラクタ
         */
        public LSInputImpl(){
            super();
            return;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public String getBaseURI(){
            return this.baseURI;
        }

        /**
         * {@inheritDoc}
         * @param baseURI {@inheritDoc}
         */
        public void setBaseURI(String baseURI){
            this.baseURI = baseURI;
            return;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public InputStream getByteStream(){
            return this.byteStream;
        }

        /**
         * {@inheritDoc}
         * @param byteStream {@inheritDoc}
         */
        public void setByteStream(InputStream byteStream){
            this.byteStream = byteStream;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public boolean getCertifiedText(){
            return this.certifiedText;
        }

        /**
         * {@inheritDoc}
         * @param certifiedText {@inheritDoc}
         */
        public void setCertifiedText(boolean certifiedText){
            this.certifiedText = certifiedText;
            return;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public Reader getCharacterStream(){
            return this.characterStream;
        }

        /**
         * {@inheritDoc}
         * @param characterStream {@inheritDoc}
         */
        public void setCharacterStream(Reader characterStream){
            this.characterStream = characterStream;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public String getEncoding(){
            return this.encoding;
        }

        /**
         * {@inheritDoc}
         * @param encoding {@inheritDoc}
         */
        public void setEncoding(String encoding){
            this.encoding = encoding;
            return;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public String getPublicId(){
            return this.publicId;
        }

        /**
         * {@inheritDoc}
         * @param publicId {@inheritDoc}
         */
        public void setPublicId(String publicId){
            this.publicId = publicId;
            return;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public String getStringData(){
            return this.stringData;
        }

        /**
         * {@inheritDoc}
         * @param stringData {@inheritDoc}
         */
        public void setStringData(String stringData){
            this.stringData = stringData;
            return;
        }

        /**
         * {@inheritDoc}
         * @return {@inheritDoc}
         */
        public String getSystemId(){
            return this.systemId;
        }

        /**
         * {@inheritDoc}
         * @param systemId {@inheritDoc}
         */
        public void setSystemId(String systemId){
            this.systemId = systemId;
            return;
        }

    }

    // TODO OASIS XML Catalog などと調和したい。
}
