/*
 * pre-defined avatar info
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: PreDefAvatar.java 509 2009-04-29 10:51:45Z olyutorskii $
 */

package jp.sourceforge.jindolf.core;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.validation.Schema;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * プリセット済みAvatarに関する情報。
 * F国運用の時点で20キャラクタ。
 */
public class PreDefAvatar{

    private static final String SCHEMA_AVATARDEF =
            "http://jindolf.sourceforge.jp/xml/xsd/coreXML.xsd";
    private static final String RES_AVATARDEF =
            "resources/xml/preDefAvatarList.xml";

    private static final String faceURITemplate;
    private static final String bodyURITemplate;

    private static final List<PreDefAvatar>        unmodList;
    private static final Map<String, PreDefAvatar> unmodMap;

    static{
        DocumentBuilder builder;
        try{
            Schema xsdSchema = SchemaUtilities.createSchema(SCHEMA_AVATARDEF);
            builder = SchemaUtilities.createBuilder(xsdSchema);
        }catch(RuntimeException e){
            throw e;
        }catch(Exception e){
            throw new ExceptionInInitializerError(e);
        }

        List<PreDefAvatar> avatarList;
        try{
            Element list = loadAvatarList(builder);

            faceURITemplate = list.getAttribute("faceIconURITemplate");
            bodyURITemplate = list.getAttribute("bodyIconURITemplate");
            if(faceURITemplate == null || bodyURITemplate == null){
                throw new SAXException("no *URITemplate attribute");
            }

            avatarList = registAvatarList(list);
        }catch(RuntimeException e){
            throw e;
        }catch(Exception e){
            throw new ExceptionInInitializerError(e);
        }

        Map<String, PreDefAvatar> avatarMap =
                new HashMap<String, PreDefAvatar>();
        for(PreDefAvatar avatar : avatarList){
            String avatarId = avatar.getAvatarId();
            avatarMap.put(avatarId, avatar);
        }

        unmodList = Collections.unmodifiableList(avatarList);
        unmodMap  = Collections.unmodifiableMap(avatarMap);
    }

    /**
     * プリセットAvatarに関する定義をロードする。
     * @param builder DOMビルダ
     * @return ルート要素
     * @throws javax.xml.parsers.ParserConfigurationException
     * @throws java.io.IOException
     * @throws org.xml.sax.SAXException
     */
    public static Element loadAvatarList(DocumentBuilder builder)
            throws ParserConfigurationException,
                   IOException,
                   SAXException {
        InputStream is = PreDefAvatar.class.getResourceAsStream(RES_AVATARDEF);

        Document document = builder.parse(is);

        Element root = document.getDocumentElement();
        String tagName = root.getTagName();
        if( ! tagName.equals("preDefinedAvatarList") ){
            throw new SAXException("illegal root " + tagName);
        }

        return root;
    }

    /**
     * 要素内部を探索し、プリセットAvatarを登録する。
     * @param list ルート要素
     * @return プリセットAvatarが登録されたList
     * @throws org.xml.sax.SAXException
     */
    private static List<PreDefAvatar> registAvatarList(Element list)
            throws SAXException {
        NodeList elems = list.getElementsByTagName("preDefinedAvatar");
        int avatarNum = elems.getLength();
        if(avatarNum <= 0){
            throw new SAXException("there is no <preDefinedAvatar>");
        }
        List<PreDefAvatar> avatarList = new ArrayList<PreDefAvatar>(avatarNum);

        for(int index=0; index < avatarNum; index++){
            Node node = elems.item(index);
            Element elem = (Element) node;
            PreDefAvatar avatar = buildAvatar(elem);
            avatarList.add(avatar);
        }

        return avatarList;
    }

    /**
     * 個々のプリセットAvatar定義をオブジェクトに変換する。
     * @param avatarDef プリセットAvatar定義要素
     * @return プリセットAvatar定義オブジェクト
     * @throws org.xml.sax.SAXException
     */
    private static PreDefAvatar buildAvatar(Element avatarDef)
            throws SAXException {
        String avatarId  = avatarDef.getAttribute("avatarId");
        String jobTitle  = avatarDef.getAttribute("jobTitle");
        String shortName = avatarDef.getAttribute("shortName");
        String serialNum = avatarDef.getAttribute("serialNum");

        String fullName = jobTitle + "\u0020" + shortName;

        int serialNo;
        try{
            serialNo = Integer.parseInt(serialNum);
        }catch(NumberFormatException e){
            throw new SAXException("illegal number form", e);
        }

        PreDefAvatar avatar;
        try{
            avatar = new PreDefAvatar(avatarId,
                                      fullName,
                                      jobTitle,
                                      shortName,
                                      serialNo );
        }catch(URISyntaxException e){
            throw new SAXException("illegal URL", e);
        }

        return avatar;
    }

    /**
     * プリセットAvatarのListを返す。
     * @return Map
     */
    public static List<PreDefAvatar> getPreDefAvatarList(){
        return unmodList;
    }

    /**
     * Avatar識別子とプリセットAvatarが対応づけられたMapを返す。
     * @return Map
     */
    public static Map<String, PreDefAvatar> getAvatarIdMap(){
        return unmodMap;
    }

    /**
     * Avatar識別子に対応するプリセットAvatarを返す。
     * @param avatarId Avatar識別子
     * @return プリセットAvatar
     */
    public static PreDefAvatar getPreDefAvatar(String avatarId){
        PreDefAvatar result = unmodMap.get(avatarId);
        return result;
    }

    private final String avatarId;
    private final String fullName;
    private final String jobTitle;
    private final String shortName;
    private final int serialNo;
    private final URI faceIconURI;
    private final URI bodyIconURI;

    /**
     * コンストラクタ
     * @param avatarId Avatar識別子
     * @param fullName フルネーム
     * @param jobTitle 職業名
     * @param shortName 省略名
     * @param serialNo 通し番号
     */
    private PreDefAvatar(String avatarId,
                           String fullName,
                           String jobTitle,
                           String shortName,
                           int serialNo )
        throws URISyntaxException {

        super();

        if(   avatarId  == null
           || fullName  == null
           || jobTitle  == null
           || shortName == null ){
            throw new NullPointerException();
        }

        this.avatarId  = avatarId.intern();
        this.fullName  = fullName.intern();
        this.jobTitle  = jobTitle.intern();
        this.shortName = shortName.intern();
        this.serialNo  = serialNo;

        String urlString;

        urlString = MessageFormat.format(faceURITemplate, this.serialNo);
        this.faceIconURI = new URI(urlString);

        urlString = MessageFormat.format(bodyURITemplate, this.serialNo);
        this.bodyIconURI = new URI(urlString);

        return;
    }

    /**
     * Avatar識別子を返す。
     * @return Avatar識別子
     */
    public String getAvatarId(){
        return this.avatarId;
    }

    /**
     * フルネームを返す。
     * @return フルネーム
     */
    public String getFullName(){
        return this.fullName;
    }

    /**
     * 職業名を返す。
     * @return 職業名
     */
    public String getJobTitle(){
        return this.jobTitle;
    }

    /**
     * 省略名を返す。
     * @return 省略名
     */
    public String getShortName(){
        return this.shortName;
    }

    /**
     * 通し番号を返す。
     * @return 通し番号
     */
    public int getSerialNo(){
        return this.serialNo;
    }

    /**
     * 顔アイコン画像のURIを返す。
     * ※ 相対URI
     * @return 画像URI
     */
    public URI getFaceIconURI(){
        return this.faceIconURI;
    }

    /**
     * 全身アイコン画像のURIを返す。
     * ※ 相対URI
     * @return 画像URI
     */
    public URI getBodyIconURI(){
        return this.bodyIconURI;
    }

    /**
     * このオブジェクトの文字列化
     * @return 文字列
     */
    @Override
    public String toString(){
        StringBuilder result = new StringBuilder();
        result.append(getAvatarId()).append(' ');
        result.append(getFullName()).append(' ');
        result.append(getJobTitle()).append(' ');
        result.append(getShortName()).append(' ');
        result.append(getSerialNo()).append(' ');
        result.append(getFaceIconURI()).append(' ');
        result.append(getBodyIconURI());
        return result.toString();
    }

}
